/***************************************************************************
                    messagelog.cpp - Kommander dialog output
                             -------------------
    copyright          : (C) 2004      Michal Rudolf <mrudolf@kdewebdwev.org>
    
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "messagelog.h"

#include <tdeapplication.h>
#include <tdefiledialog.h>
#include <kiconloader.h>
#include <tdelistbox.h>
#include <tdemessagebox.h>
#include <tdepopupmenu.h>
#include <tdeprocess.h>

#include <tqclipboard.h>
#include <tqfile.h>
#include <tqtextstream.h>

MessageLog::MessageLog(TQWidget* parent, const char* name) : TQTabWidget(parent, name)
{
  m_popupMenu = new TDEPopupMenu(this);
  m_popupMenu->insertItem(SmallIconSet("edit-copy"), i18n("Copy Current &Line"), this, TQ_SLOT(copyLine()));
  m_popupMenu->insertItem(SmallIconSet("edit-copy"), i18n("&Copy Content"), this, TQ_SLOT(copyContent()));
  m_popupMenu->insertItem(SmallIconSet("document-save-as"), i18n("&Save As..."), this, TQ_SLOT(saveToFile()));
  m_popupMenu->insertSeparator();
  m_popupMenu->insertItem(SmallIconSet("edit-clear"), i18n("Clear"), this, TQ_SLOT(clearContent()));
  
  for (int i = 0; i < m_listCount; i++)
  {
    m_lists[i] = new TDEListBox(this);
    addTab(m_lists[i], m_listNames[i]);
    m_seenEOL[i] = false;
    connect(m_lists[i], TQ_SIGNAL(contextMenuRequested(TQListBoxItem*, const TQPoint&)),
            this, TQ_SLOT(showMenu(TQListBoxItem*, const TQPoint&)));
  }
}
  
MessageLog::~MessageLog()
{
}
  
void MessageLog::insertItem(InfoType i, TQString text)
{
  bool seenEOL = text.endsWith("\n");
  if (seenEOL)
    text.truncate(text.length() - 1);
  TQStringList items(TQStringList::split('\n', text));
  for (TQStringList::ConstIterator it = items.begin(); it != items.end(); ++it ) 
  {
    if (!m_seenEOL[i] && m_lists[i]->count() && it == items.begin())
      m_lists[i]->changeItem(m_lists[i]->text(m_lists[i]->count() - 1) + *it, m_lists[i]->count() - 1);
    else
      m_lists[i]->insertItem(*it);
  }
  m_seenEOL[i] = seenEOL;
  m_lists[i]->setCurrentItem(m_lists[i]->count()-1);
  m_lists[i]->ensureCurrentVisible();
}

TQString MessageLog::content()
{
  TQString p_content;
  TDEListBox* list = m_lists[currentPageIndex()];
  for (uint i=0; i < list->count(); i++)
    p_content.append(list->text(i) + "\n");
  return p_content;
}

void MessageLog::clear(InfoType i)
{
  if (i != All)
  {
    m_lists[(int)i]->clear(); 
    m_seenEOL[i] = false;
  }
  else 
    for (int i = 0; i < m_listCount; i++)
      clear((InfoType)i);
}

void MessageLog::receivedStdout(TDEProcess*, char* buffer, int buflen)
{
  insertItem(Stdout, TQString::fromLocal8Bit(buffer, buflen));
}

void MessageLog::receivedStderr(TDEProcess*, char* buffer, int buflen)
{
  insertItem(Stderr, TQString::fromLocal8Bit(buffer, buflen));
}

void MessageLog::clearContent()
{
  clear((InfoType)currentPageIndex()); 
}

void MessageLog::copyLine()
{
  if (m_lists[currentPageIndex()]->count())
    tdeApp->clipboard()->setText(m_lists[currentPageIndex()]->currentText(), TQClipboard::Clipboard);
}

void MessageLog::copyContent()
{
  tdeApp->clipboard()->setText(content(), TQClipboard::Clipboard);
}

void MessageLog::saveToFile()
{
  KURL url=KFileDialog::getSaveURL(TQDir::currentDirPath(),
                                   i18n("*.log|Log Files (*.log)\n*|All Files"), this, i18n("Save Log File"));
  if (url.isEmpty())
    return;
  TQFileInfo fileinfo(url.path());
  if (fileinfo.exists() && KMessageBox::warningContinueCancel(0,
      i18n("<qt>File<br><b>%1</b><br>already exists. Overwrite it?</qt>")
          .arg(url.path()), TQString(), i18n("Overwrite")) == KMessageBox::Cancel)
    return;
  TQFile file(url.path());
  if (!file.open(IO_WriteOnly)) {
    KMessageBox::error(0, i18n("<qt>Cannot save log file<br><b>%1</b></qt>")
        .arg(url.url()));
    return;
  }
  TQTextStream textfile(&file);
  textfile << content();
  file.close();
}

void MessageLog::showMenu(TQListBoxItem*, const TQPoint& l_point)
{
  m_popupMenu->exec(l_point);
}

TQString MessageLog::m_listNames[m_listCount] = {i18n("Stdout"), i18n("Stderr")};

#include "messagelog.moc"

