/* ============================================================
 * File  : galleries.cpp
 * Author: Colin Guthrie <kde@colin.guthr.ie>
 * Date  : 2006-09-04
 * Copyright 2006 by Colin Guthrie <kde@colin.guthr.ie>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * ============================================================ */

#include <tqstring.h>

#include <tqwidget.h>
#include <tdeapplication.h>
#include <kdebug.h>
#include <tdeconfig.h>
#include <tdelocale.h>
#include <tdeversion.h>
#if KDE_IS_VERSION(3,2,0)
#include <tdewallet.h>
#endif

#include "galleries.h"

namespace KIPIGalleryExportPlugin
{


Gallery::Gallery(const TQString& name, const TQString& url,
                 const TQString& username, const TQString& password,
                 const unsigned int version,
                 const unsigned int galleryId)
  : mName(name),
    mUrl(url),
    mUsername(username),
    mPassword(password),
    mVersion(version),
    mGalleryId(galleryId)
{

}

Gallery::~Gallery()
{

}

TQString Gallery::name() const { return mName; }
TQString Gallery::url() const { return mUrl; }
TQString Gallery::username() const { return mUsername; }
TQString Gallery::password() const { return mPassword; }
unsigned int Gallery::version() const { return mVersion; }
unsigned int Gallery::galleryId() const { return mGalleryId; }

void Gallery::setName(TQString name) { mName = name; }
void Gallery::setUrl(TQString url) { mUrl = url; }
void Gallery::setUsername(TQString username) { mUsername = username; }
void Gallery::setPassword(TQString password) { mPassword = password; }
void Gallery::setVersion(unsigned int version) { mVersion = version; }
void Gallery::setGalleryId(unsigned int galleryId) { mGalleryId = galleryId; }


TQListViewItem* Gallery::asTQListViewItem(TQListView* pParent)
{
  TQListViewItem* p_lvi = (TQListViewItem*) new GalleryTQListViewItem(this, pParent);
  return p_lvi;
}



GalleryTQListViewItem::GalleryTQListViewItem(Gallery* pGallery, TQListView* pParent)
  : TQListViewItem(pParent, pGallery->name(), pGallery->url(), pGallery->username()),
    mpGallery(pGallery)
{
}

Gallery* GalleryTQListViewItem::GetGallery()
{
  return mpGallery;
}

void GalleryTQListViewItem::Refresh()
{
  setText(0, mpGallery->name());
  setText(1, mpGallery->url());
  setText(2, mpGallery->username());
}



Galleries::Galleries()
 : mpWallet(0),
   mMaxGalleryId(0)
{
}

Galleries::~Galleries()
{
  if (mpWallet)
    delete mpWallet;

  // Todo: clear up mGalleries
}

void Galleries::Load()
{
  static bool bln_loaded = false;
  if (bln_loaded) return;
  bln_loaded = true;

  bool bln_use_wallet = false;
#if KDE_IS_VERSION(3,2,0)
  mpWallet = TDEWallet::Wallet::openWallet(TDEWallet::Wallet::NetworkWallet(),
                                         kapp->activeWindow()->winId(),
                                         TDEWallet::Wallet::Synchronous);
  if (!mpWallet)
  {
    kdWarning() << "Failed to open tdewallet" << endl;
  }
  else
  {
    if (!mpWallet->hasFolder("KIPIGallerySyncPlugin"))
    {
      if (!mpWallet->createFolder("KIPIGallerySyncPlugin"))
        kdWarning() << "Failed to create tdewallet folder" << endl;
    }

    if (!mpWallet->setFolder("KIPIGallerySyncPlugin"))
      kdWarning() << "Failed to set tdewallet folder" << endl;
    else
      bln_use_wallet = true;
  }
#endif


  // read config
  TDEConfig config("kipirc");
  config.setGroup("GallerySync Settings");
  TQValueList<int> gallery_ids = config.readIntListEntry("Galleries");

  config.setGroup("GallerySync Galleries");
  TQString name, url, username, password = "";
  int version;
  for (TQValueList<int>::Iterator it = gallery_ids.begin(); it != gallery_ids.end(); ++it)
  {
    unsigned int gallery_id = (*it);

    if (gallery_id > mMaxGalleryId)
      mMaxGalleryId = gallery_id;

    // Load the gallery with this id.
    name = config.readEntry(TQString("Name%1").arg(gallery_id));
    url = config.readEntry(TQString("URL%1").arg(gallery_id));
    username = config.readEntry(TQString("Username%1").arg(gallery_id));
    version = config.readNumEntry(TQString("Version%1").arg(gallery_id));
    if (bln_use_wallet)
      mpWallet->readPassword(TQString("Password%1").arg(gallery_id), password);

    Gallery* p_gallery = new Gallery(name, url, username, password, version, gallery_id);
    mGalleries.append(p_gallery);
  }
}


void Galleries::Add(Gallery* pGallery)
{
  mGalleries.append(pGallery);
}

void Galleries::Remove(Gallery* pGallery)
{
  mGalleries.remove(pGallery);

  // Slight cosmetic thing for gallery numbering.
  if (mGalleries.isEmpty())
    mMaxGalleryId = 0;
}


void Galleries::Save()
{
  TQValueList<int> gallery_ids;
  TDEConfig config("kipirc");
  config.deleteGroup("GallerySync Galleries");
  config.setGroup("GallerySync Galleries");

  bool bln_use_wallet = false;
  if (mpWallet)
  {
    if (mpWallet->hasFolder("KIPIGallerySyncPlugin"))
    {
      if (!mpWallet->removeFolder("KIPIGallerySyncPlugin"))
        kdWarning() << "Failed to clear tdewallet folder" << endl;
    }
    if (!mpWallet->createFolder("KIPIGallerySyncPlugin"))
      kdWarning() << "Failed to create tdewallet folder" << endl;

    if (!mpWallet->setFolder("KIPIGallerySyncPlugin"))
      kdWarning() << "Failed to set tdewallet folder" << endl;
    else
      bln_use_wallet = true;
  }

  for (GalleryPtrList::iterator it = mGalleries.begin(); it != mGalleries.end(); ++it)
  {
    Gallery* p_gallery = (*it);
    if (!p_gallery->galleryId())
      p_gallery->setGalleryId(++mMaxGalleryId);
    unsigned int gallery_id = p_gallery->galleryId();
    gallery_ids.append(gallery_id);

    config.writeEntry(TQString("Name%1").arg(gallery_id), p_gallery->name());
    config.writeEntry(TQString("URL%1").arg(gallery_id), p_gallery->url());
    config.writeEntry(TQString("Username%1").arg(gallery_id), p_gallery->username());
    config.writeEntry(TQString("Version%1").arg(gallery_id), p_gallery->version());
    if (bln_use_wallet)
      mpWallet->writePassword(TQString("Password%1").arg(gallery_id), p_gallery->password());
  }

  config.setGroup("GallerySync Settings");
  config.writeEntry("Galleries", gallery_ids);
}

TQListView* Galleries::asTQListView(TQWidget* pParent)
{
  Load();

  TQListView* p_lv = new TQListView (pParent);
  p_lv->addColumn(i18n("Gallery Name"));
  p_lv->addColumn(i18n("URL"));
  p_lv->addColumn(i18n("User"));
  p_lv->setAllColumnsShowFocus(true);

  for (GalleryPtrList::iterator it = mGalleries.begin(); it != mGalleries.end(); ++it)
  {
    (*it)->asTQListViewItem(p_lv);
  }

  return p_lv;
}

}
