import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { Text } from 'ink';
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { render } from 'ink-testing-library';
import { ConsentPrompt } from './ConsentPrompt.js';
import { RadioButtonSelect } from './shared/RadioButtonSelect.js';
import { MarkdownDisplay } from '../utils/MarkdownDisplay.js';
vi.mock('./shared/RadioButtonSelect.js', () => ({
    RadioButtonSelect: vi.fn(() => null),
}));
vi.mock('../utils/MarkdownDisplay.js', () => ({
    MarkdownDisplay: vi.fn(() => null),
}));
const MockedRadioButtonSelect = vi.mocked(RadioButtonSelect);
const MockedMarkdownDisplay = vi.mocked(MarkdownDisplay);
describe('ConsentPrompt', () => {
    const onConfirm = vi.fn();
    const terminalWidth = 80;
    beforeEach(() => {
        vi.clearAllMocks();
    });
    it('renders a string prompt with MarkdownDisplay', () => {
        const prompt = 'Are you sure?';
        render(_jsx(ConsentPrompt, { prompt: prompt, onConfirm: onConfirm, terminalWidth: terminalWidth }));
        expect(MockedMarkdownDisplay).toHaveBeenCalledWith({
            isPending: true,
            text: prompt,
            terminalWidth,
        }, undefined);
    });
    it('renders a ReactNode prompt directly', () => {
        const prompt = _jsx(Text, { children: "Are you sure?" });
        const { lastFrame } = render(_jsx(ConsentPrompt, { prompt: prompt, onConfirm: onConfirm, terminalWidth: terminalWidth }));
        expect(MockedMarkdownDisplay).not.toHaveBeenCalled();
        expect(lastFrame()).toContain('Are you sure?');
    });
    it('calls onConfirm with true when "Yes" is selected', () => {
        const prompt = 'Are you sure?';
        render(_jsx(ConsentPrompt, { prompt: prompt, onConfirm: onConfirm, terminalWidth: terminalWidth }));
        const onSelect = MockedRadioButtonSelect.mock.calls[0][0].onSelect;
        onSelect(true);
        expect(onConfirm).toHaveBeenCalledWith(true);
    });
    it('calls onConfirm with false when "No" is selected', () => {
        const prompt = 'Are you sure?';
        render(_jsx(ConsentPrompt, { prompt: prompt, onConfirm: onConfirm, terminalWidth: terminalWidth }));
        const onSelect = MockedRadioButtonSelect.mock.calls[0][0].onSelect;
        onSelect(false);
        expect(onConfirm).toHaveBeenCalledWith(false);
    });
    it('passes correct items to RadioButtonSelect', () => {
        const prompt = 'Are you sure?';
        render(_jsx(ConsentPrompt, { prompt: prompt, onConfirm: onConfirm, terminalWidth: terminalWidth }));
        expect(MockedRadioButtonSelect).toHaveBeenCalledWith(expect.objectContaining({
            items: [
                { label: 'Yes', value: true, key: 'Yes' },
                { label: 'No', value: false, key: 'No' },
            ],
        }), undefined);
    });
});
//# sourceMappingURL=ConsentPrompt.test.js.map