"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultConfiguration = exports.getConfigFromEnvironmentVariables = exports.ConfigSchema = void 0;
const zod_1 = require("zod");
const logger_1 = require("./util/logger");
exports.ConfigSchema = zod_1.z.object({
    // Maximum number of files to analyze in the background. Set to 0 to disable background analysis.
    backgroundAnalysisMaxFiles: zod_1.z.number().int().min(0).default(500),
    // Enable diagnostics for source errors. Ignored if includeAllWorkspaceSymbols is true.
    enableSourceErrorDiagnostics: zod_1.z.boolean().default(false),
    // Glob pattern for finding and parsing shell script files in the workspace. Used by the background analysis features across files.
    globPattern: zod_1.z.string().trim().default('**/*@(.sh|.inc|.bash|.command)'),
    // Configure explainshell server endpoint in order to get hover documentation on flags and options.
    // And empty string will disable the feature.
    explainshellEndpoint: zod_1.z.string().trim().default(''),
    // Log level for the server. To set the right log level from the start please also use the environment variable 'BASH_IDE_LOG_LEVEL'.
    logLevel: zod_1.z.enum(logger_1.LOG_LEVELS).default(logger_1.DEFAULT_LOG_LEVEL),
    // Controls how symbols (e.g. variables and functions) are included and used for completion, documentation, and renaming.
    // If false, then we only include symbols from sourced files (i.e. using non dynamic statements like 'source file.sh' or '. file.sh' or following ShellCheck directives).
    // If true, then all symbols from the workspace are included.
    includeAllWorkspaceSymbols: zod_1.z.boolean().default(false),
    // Additional ShellCheck arguments. Note that we already add the following arguments: --shell, --format, --external-sources."
    shellcheckArguments: zod_1.z
        .preprocess((arg) => {
        let argsList = [];
        if (typeof arg === 'string') {
            argsList = arg.split(' ');
        }
        else if (Array.isArray(arg)) {
            argsList = arg;
        }
        return argsList.map((s) => s.trim()).filter((s) => s.length > 0);
    }, zod_1.z.array(zod_1.z.string()))
        .default([]),
    // Controls the executable used for ShellCheck linting information. An empty string will disable linting.
    shellcheckPath: zod_1.z.string().trim().default('shellcheck'),
});
function getConfigFromEnvironmentVariables() {
    const rawConfig = {
        backgroundAnalysisMaxFiles: toNumber(process.env.BACKGROUND_ANALYSIS_MAX_FILES),
        enableSourceErrorDiagnostics: toBoolean(process.env.ENABLE_SOURCE_ERROR_DIAGNOSTICS),
        explainshellEndpoint: process.env.EXPLAINSHELL_ENDPOINT,
        globPattern: process.env.GLOB_PATTERN,
        includeAllWorkspaceSymbols: toBoolean(process.env.INCLUDE_ALL_WORKSPACE_SYMBOLS),
        logLevel: process.env[logger_1.LOG_LEVEL_ENV_VAR],
        shellcheckArguments: process.env.SHELLCHECK_ARGUMENTS,
        shellcheckPath: process.env.SHELLCHECK_PATH,
    };
    const environmentVariablesUsed = Object.entries(rawConfig)
        .map(([key, value]) => (typeof value !== 'undefined' ? key : null))
        .filter((key) => key !== null)
        .filter((key) => key !== 'logLevel'); // logLevel is a special case that we ignore
    const config = exports.ConfigSchema.parse(rawConfig);
    return { config, environmentVariablesUsed };
}
exports.getConfigFromEnvironmentVariables = getConfigFromEnvironmentVariables;
function getDefaultConfiguration() {
    return exports.ConfigSchema.parse({});
}
exports.getDefaultConfiguration = getDefaultConfiguration;
const toBoolean = (s) => typeof s !== 'undefined' ? s === 'true' || s === '1' : undefined;
const toNumber = (s) => typeof s !== 'undefined' ? parseInt(s, 10) : undefined;
//# sourceMappingURL=config.js.map