/* Balsa E-Mail Client
 * Copyright (C) 1997 Jay Painter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <gtk/gtk.h>
#include "mailbox.h"
#include "error.h"
#include "main_window.h"
#include "index_interface.h"


void
mail_init ()
{
  static int mail_init_done = 0;

  if (!mail_init_done)
    {
      mail_link (&mboxdriver);              /* link in the mbox driver */
      mail_link (&imapdriver);              /* link in the imap driver */
      mail_link (&nntpdriver);              /* link in the nntp driver */
      mail_link (&pop3driver);              /* link in the pop3 driver */
      mail_link (&mhdriver);                /* link in the mh driver */
      mail_link (&mxdriver);                /* link in the mx driver */
      mail_link (&mbxdriver);               /* link in the mbx driver */
      mail_link (&tenexdriver);             /* link in the tenex driver */
      mail_link (&mtxdriver);               /* link in the mtx driver */
      mail_link (&mmdfdriver);              /* link in the mmdf driver */
      mail_link (&bezerkdriver);            /* link in the bezerk driver */
      mail_link (&newsdriver);              /* link in the news driver */
      mail_link (&philedriver);             /* link in the phile driver */
      mail_link (&dummydriver);             /* link in the dummy driver */
      auth_link (&auth_log);                /* link in the log authenticator */

      mail_init_done = 1;
    }
}



/*
 *  ==== MAILBOX METHODS ====
 */

Mailbox *
mailbox_new (PostOffice *postoffice,
	     char       *name,
	     char       *mbox)
{
  Mailbox *mailbox;

  mailbox = g_malloc (sizeof (Mailbox));
  mailbox->postoffice = postoffice;
  mailbox->name = g_strdup (name);
  mailbox->path = g_strdup (mbox);
  mailbox->stream = NIL;
  mailbox->message_list = NULL;

  return mailbox;
};


int
mailbox_open (Mailbox *mailbox)
{
  MAILSTREAM *stream;
  Message *message;
  GList *list;
  long i;


  /*
   * don't open a mailbox if it's already open -- runtime sanity
   */
  if (mailbox->postoffice->open == mailbox)
    return TRUE;


  /*
   * only one mailbox open at a time
   */
  if (mailbox->postoffice->open)
    mailbox_close (mailbox->postoffice->open);


  /*
   * try to open the mailbox -- return
   * FALSE on failure
   */
  stream = mail_open (NIL, mailbox->path, NIL);
  if (stream == NIL)
    {
      balsa_error ("Bummer Dude");
      return FALSE;
    }

  mailbox->stream = stream;

  list = NULL;
  for (i = 1; i <= stream->nmsgs; i++)
    {
      message = message_new (mailbox, i);
      list = g_list_append (list, message);
    }
  mailbox->message_list = list;


  mailbox->postoffice->open = mailbox;
  return TRUE;
}


void
mailbox_close (Mailbox *mailbox)
{
  Message *message;
  GList *list;

  /*
   * free the message list
   */
  list = mailbox->message_list;
  while (list)
    {
      message = list->data;
      list = list->next;

      message_delete (message);
    }
  g_list_free (list);
  mailbox->message_list = NULL;

  /*
   * now close the mail stream and expunge deleted
   * messages -- the expunge may not have to be done
   */
  mailbox->stream = mail_close_full (mailbox->stream, CL_EXPUNGE);
}




/*
 * ====MESSAGE METHODS====
 */

Message *
message_new (Mailbox *mailbox, long mesgno)
{
  Message *message;

  message = g_malloc (sizeof (Message));
  message->mailbox = mailbox;
  message->mesgno = mesgno;

  return message;
}


void
message_delete (Message *message)
{
  if (message)
    g_free (message);
}

/*
 *  ==== MAIL/MESSAGE Operations ====
 */
void
mail_message_delete (Message *message)
{
  char tmp[10];

  /*
   * remove the message from the linked list of messages
   * in the mailbox structure
   */
  message->mailbox->message_list = g_list_remove
    (message->mailbox->message_list, message);

  /*
   * tell the c-client that the message is deleted; it will be removed
   * from the mailbox when it is closed
   */
  if (message)
    {
      sprintf (tmp, "%ld", message->mesgno);
      mail_setflag (message->mailbox->stream, tmp, "\\DELETED");

      /*
       * free up the memory from the message structure
       */
      message_delete (message);
    }
}


void
mail_message_move (Message *message, Mailbox *mailbox)
{
  char tmp[10];


  if (!message || !mailbox)
    return;

  sprintf (tmp, "%ld", message->mesgno);
  mail_move (message->mailbox->stream, tmp, mailbox->path);
}


char *
mail_message_from (Message *message)
{
  static char from[27];

  if (message->mailbox->stream)
    {
      mail_fetchfrom (from, message->mailbox->stream,
       		      message->mesgno, (long)25);
      return from;
    }

  return NULL;
}


char *
mail_message_subject (Message *message)
{
  static char subject[40];

  if (message->mailbox->stream)
    {
      mail_fetchsubject (subject, message->mailbox->stream,
    		          message->mesgno, (long)38);
      return subject;
    }  

  return NULL;
}


char *
mail_message_date (Message *message)
{
  static char date[50];
  MESSAGECACHE *cache;

  if (message->mailbox->stream)
    {
      mail_fetchstructure (message->mailbox->stream, 
			   message->mesgno, NIL);
      cache = mail_elt (message->mailbox->stream, message->mesgno);
      mail_date (date, cache);
      
      /* trunckate date */
      date[12] = '\0';

      return date;
    }
       
  return NULL;
}





