/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * GtkFontSelection widget for Gtk+, by Damon Chaplin, May 1998.
 * Based on the GnomeFontSelector widget, by Elliot Lee, but major changes.
 * The GnomeFontSelector was derived from app/text_tool.c in the GIMP.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Note: I'm not sure what we should be using as the resolution in the XLFD
 *       fontnames? It was originally hard-coded as 75 dpi. I've changed it
 *       to use '*', but I don't know what it should really do.
 *	   see functions: gtk_font_selection_get_fonts ()
 *	                  gtk_font_selection_create_xlfd ()
 */

/*
 * Limits:
 *
 *  Fontnames	 - A maximum of MAX_FONTS (32767) fontnames will be retrieved
 *		   from X Windows with XListFonts(). Any more are ignored.
 *		   I think this limit may have been set because of a limit in
 *		   GtkList. It could possibly be increased since we are using
 *		   GtkClists now, but I'd be surprised if it was reached.
 *  Field length - XLFD_MAX_FIELD_LEN is the maximum length that any field of a
 *		   fontname can be for it to be considered valid. Others are
 *		   ignored.
 *  Properties   - Maximum of 65535 choices for each font property - guint16's
 *		   are used as indices, e.g. in the FontInfo struct.
 *  Combinations - Maximum of 65535 combinations of properties for each font
 *		   family - a guint16 is used in the FontInfo struct.
 *  Font size    - Minimum font size of 2 pixels/points, since trying to load
 *		   some fonts with a size of 1 can cause X to hang.
 *		   (e.g. the Misc Fixed fonts).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <X11/Xlib.h>

#include "gdk/gdkx.h"
#include "gdk/gdkkeysyms.h"

#include "gtkbutton.h"
#include "gtkclist.h"
#include "gtkentry.h"
#include "gtkfontsel.h"
#include "gtkframe.h"
#include "gtkhbbox.h"
#include "gtkhbox.h"
#include "gtklabel.h"
#include "gtknotebook.h"
#include "gtkradiobutton.h"
#include "gtksignal.h"
#include "gtktable.h"
#include "gtktooltips.h"
#include "gtkvbox.h"


/* The maximum number of fontnames requested with XListFonts(). */
#define MAX_FONTS 32767

/* This is the largest field length we will accept. If a fontname has a field
   larger than this we will skip it. */
#define XLFD_MAX_FIELD_LEN 64

/* These are what we use as the standard font sizes, for the size clist.
   Note that when using points we still show these integer point values but
   we work internally in decipoints (and decipoint values can be typed in). */
static gint font_sizes[] = {
  8, 9, 10, 11, 12, 13, 14, 16, 18, 20, 22, 24, 26, 28,
  32, 36, 40, 48, 56, 64, 72
};

/* Used to determine whether we are using point or pixel sizes. */
#define PIXELS_METRIC 0
#define POINTS_METRIC 1

/* Initial font metric & size (Remember point sizes are in decipoints).
   The font size should match one of those in the font_sizes array. */
#define INITIAL_METRIC		  POINTS_METRIC
#define INITIAL_FONT_SIZE	  140

/* This is the default text shown in the preview entry, though the user
   can set it. Remember that some fonts only have capital letters. */
#define PREVIEW_TEXT "abcdefghijk ABCDEFGHIJK"

/* This is the maximum height of the preview entry (it expands for big fonts)*/
#define MAX_PREVIEW_HEIGHT 300

/* These are the sizes of the font, style & size clists. */
#define FONT_LIST_HEIGHT	120
#define FONT_LIST_WIDTH		200
#define FONT_STYLE_LIST_WIDTH	160
#define FONT_SIZE_LIST_WIDTH	50

/* These are the field numbers in the X Logical Font Description, e.g.
   -adobe-courier-bold-o-normal--25-180-100-100-m-150-iso8859-1 */
#define XLFD_FOUNDRY		0
#define XLFD_FAMILY		1
#define XLFD_WEIGHT		2
#define XLFD_SLANT		3
#define XLFD_SET_WIDTH		4
#define XLFD_ADD_STYLE		5
#define XLFD_PIXELS		6
#define XLFD_POINTS		7
#define XLFD_RESOLUTION_X	8
#define XLFD_RESOLUTION_Y	9
#define XLFD_SPACING		10
#define XLFD_AVERAGE_WIDTH	11
#define XLFD_REGISTRY		12
#define XLFD_ENCODING		13

/* These are the names of the fields, used on the info page. */
static gchar* xlfd_field_names[GTK_XLFD_NUM_FIELDS] = {
  "Foundry:",
  "Family:",
  "Weight:",
  "Slant:",
  "Set Width:",
  "Add Style:",
  "Pixel Size:",
  "Point Size:",
  "Resolution X:",
  "Resolution Y:",
  "Spacing:",
  "Average Width:",
  "CharSet Registry:",
  "CharSet Encoding:",
};

/* These are the array indices of the font properties used in several arrays,
   and should match the xlfd_index array below. */
#define WEIGHT	  0
#define SLANT	  1
#define SET_WIDTH 2
#define SPACING	  3
#define FOUNDRY	  4

/* This is used to look up a field in a fontname given one of the above
   property indices. */
static gint xlfd_index[GTK_NUM_FONT_PROPERTIES + 1] = {
  XLFD_WEIGHT,
  XLFD_SLANT,
  XLFD_SET_WIDTH,
  XLFD_SPACING,
  XLFD_FOUNDRY
};


/* The initial size and increment of each of the arrays of property values. */
#define PROPERTY_ARRAY_INCREMENT	16

/* This struct represents one family of fonts (with one foundry), e.g. adobe
   courier or sony fixed. It stores the family name, the index of the foundry
   name, and an array of combinations of valid properties, e.g. 'medium',
   'bold', 'bold & italic'. Thus each combination represents one fontname as
   returned from XListFonts(). */
struct _FontInfo
{
  gchar   *family;
  guint16  foundry;
  gint	   combo_index;
  guint16  ncombos;
};


static void    gtk_font_selection_class_init	     (GtkFontSelectionClass *klass);
static void    gtk_font_selection_init		     (GtkFontSelection *fontsel);
static void    gtk_font_selection_destroy	     (GtkObject      *object);

/* These are all used for class initialization - loading in the fonts etc. */
static void    gtk_font_selection_get_fonts          (GtkFontSelectionClass *klass);
static void    gtk_font_selection_insert_font        (GtkFontSelectionClass *klass,
						      GSList         *fontnames[],
						      gint           *ntable,
						      gchar          *fontname);
static gint    gtk_font_selection_insert_field       (GtkFontSelectionClass *klass,
						      gchar          *fontname,
						      gint            prop);

/* These are the callbacks & related functions. */
static void    gtk_font_selection_select_font	     (GtkWidget      *w,
						      gint	      row,
						      gint	      column,
						      GdkEventButton *bevent,
						      gpointer        data);
static void    gtk_font_selection_set_available_styles
						     (GtkFontSelection *fs);
static void    gtk_font_selection_select_best_style  (GtkFontSelection *fs);

static void    gtk_font_selection_select_style	     (GtkWidget      *w,
						      gint	      row,
						      gint	      column,
						      GdkEventButton *bevent,
						      gpointer        data);
static gint    gtk_font_selection_size_key_press     (GtkWidget      *w,
						      GdkEventKey    *event,
						      gpointer        data);
static void    gtk_font_selection_select_size	     (GtkWidget      *w,
						      gint	      row,
						      gint	      column,
						      GdkEventButton *bevent,
						      gpointer        data);

static void    gtk_font_selection_metric_callback    (GtkWidget      *w,
						      gpointer        data);
static void    gtk_font_selection_expose_list	     (GtkWidget	     *w,
						      GdkEventExpose *event,
						      gpointer        data);

static void    gtk_font_selection_switch_page	     (GtkWidget      *w,
						      GtkNotebookPage *page,
						      gint             page_num,
						      gpointer         data);

/* Misc. utility functions. */
static gboolean gtk_font_selection_load_font          (GtkFontSelection *fs);
static void    gtk_font_selection_update_preview     (GtkFontSelection *fs);

static gint    gtk_font_selection_find_font	     (GtkFontSelection *fs,
						      gchar          *family,
						      guint16	      foundry);
static guint16 gtk_font_selection_field_to_index     (gchar         **table,
						      gint            ntable,
						      gchar          *field);
static void    gtk_font_selection_show_size	     (GtkFontSelection *fs);

static gchar*  gtk_font_selection_expand_slant_code  (gchar          *slant);
static gchar*  gtk_font_selection_expand_spacing_code(gchar          *spacing);

/* Functions for handling X Logical Font Description fontnames. */
static gboolean gtk_font_selection_is_xlfd_font_name (const gchar    *fontname);
static char*   gtk_font_selection_get_xlfd_field     (const gchar    *fontname,
						      gint            field_num,
						      gchar          *buffer);
static gchar * gtk_font_selection_create_xlfd        (gint            size,
						      gint            metric,
						      gchar          *foundry,
						      gchar          *family,
						      gchar          *weight,
						      gchar          *slant,
						      gchar          *set_width,
						      gchar          *spacing);



static GtkWindowClass *parent_class = NULL;


guint
gtk_font_selection_get_type()
{
  static guint font_selection_type = 0;
  if(!font_selection_type) {
    GtkTypeInfo fontsel_info = {
      "GtkFontSelection",
      sizeof(GtkFontSelection),
      sizeof(GtkFontSelectionClass),
      (GtkClassInitFunc) gtk_font_selection_class_init,
      (GtkObjectInitFunc) gtk_font_selection_init,
      (GtkArgSetFunc) NULL,
      (GtkArgGetFunc) NULL,
    };
    font_selection_type = gtk_type_unique(gtk_window_get_type(),
					  &fontsel_info);
  }
  return font_selection_type;
}

static void
gtk_font_selection_class_init(GtkFontSelectionClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (gtk_window_get_type ());

  object_class->destroy = gtk_font_selection_destroy;

  gtk_font_selection_get_fonts(klass);
}

static void
gtk_font_selection_init(GtkFontSelection *fontsel)
{
  GtkFontSelectionClass *klass;
  FontInfo *font_info, *font;
  GtkWidget *text_frame;
  GtkWidget *text_box;
  GtkWidget *table, *label, *vbox, *hbox;
  GtkTooltips *tooltips;
  gint i, prop;
  gint nfonts;
  gchar *titles[] = { "Font Property", "Requested Value", "Actual Value" };
  gchar buffer[128];
  gchar *size;
  gchar font_buffer[XLFD_MAX_FIELD_LEN * 2 + 4];
  gchar *font_item;
  gint size_to_match;
  gchar *row[3];

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font_info = klass->font_info;
  nfonts = klass->nfonts;

  /* Initialize the GtkFontSelection struct. We do this here in case any
     callbacks are triggered while creating the interface. */
  fontsel->font = NULL;
  fontsel->font_index = -1;
  fontsel->metric = INITIAL_METRIC;
  fontsel->size = INITIAL_FONT_SIZE;

  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++)
    fontsel->property_values[prop] = 0;

  fontsel->scroll_on_expose = TRUE;

  /* Create the dialog. */
  tooltips=gtk_tooltips_new();

  gtk_container_border_width (GTK_CONTAINER (fontsel), 4);
  gtk_window_set_policy(GTK_WINDOW(fontsel), FALSE, TRUE, FALSE);
  fontsel->top_vbox = gtk_vbox_new (FALSE, 4);
  gtk_widget_show (fontsel->top_vbox);
  gtk_container_add (GTK_CONTAINER (fontsel), fontsel->top_vbox);

  fontsel->notebook = gtk_notebook_new();
  gtk_widget_show (fontsel->notebook);
  gtk_container_add (GTK_CONTAINER (fontsel->top_vbox), fontsel->notebook);

  /* Create the shell and vertical & horizontal boxes */
  fontsel->main_vbox = gtk_vbox_new (FALSE, 4);
  gtk_widget_show (fontsel->main_vbox);
  gtk_container_border_width (GTK_CONTAINER (fontsel->main_vbox), 6);
  label = gtk_label_new("Font");
  gtk_widget_set_usize (label, 120, -1);
  gtk_notebook_append_page (GTK_NOTEBOOK (fontsel->notebook),
			    fontsel->main_vbox, label);

  /* Create the font info page */
  fontsel->info_vbox = gtk_vbox_new (FALSE, 4);
  gtk_widget_show (fontsel->info_vbox);
  gtk_container_border_width (GTK_CONTAINER (fontsel->info_vbox), 2);
  label = gtk_label_new("Font Information");
  gtk_widget_set_usize (label, 120, -1);
  gtk_notebook_append_page (GTK_NOTEBOOK (fontsel->notebook),
			    fontsel->info_vbox, label);

  /* Create the table of font, style & size. */
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_table_set_col_spacings(GTK_TABLE(table), 8);
  gtk_box_pack_start (GTK_BOX (fontsel->main_vbox), table, TRUE, TRUE, 0);

  label = gtk_label_new("Font:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK, 0, 0);
  label = gtk_label_new("Font Style:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK, 0, 0);
  label = gtk_label_new("Size:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK, 0, 0);
  
  fontsel->font_entry = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(fontsel->font_entry), FALSE);
  gtk_widget_set_usize (fontsel->font_entry, 20, -1);
  gtk_widget_show (fontsel->font_entry);
  gtk_table_attach (GTK_TABLE (table), fontsel->font_entry, 0, 1, 1, 2,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK, 0, 0);
  fontsel->font_style_entry = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(fontsel->font_style_entry), FALSE);
  gtk_widget_set_usize (fontsel->font_style_entry, 20, -1);
  gtk_widget_show (fontsel->font_style_entry);
  gtk_table_attach (GTK_TABLE (table), fontsel->font_style_entry, 1, 2, 1, 2,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK, 0, 0);
  fontsel->size_entry = gtk_entry_new();
  gtk_widget_set_usize (fontsel->size_entry, 20, -1);
  gtk_widget_show (fontsel->size_entry);
  gtk_table_attach (GTK_TABLE (table), fontsel->size_entry, 2, 3, 1, 2,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK, 0, 0);
  gtk_signal_connect (GTK_OBJECT (fontsel->size_entry), "key_press_event",
		      (GtkSignalFunc) gtk_font_selection_size_key_press,
		      fontsel);

  /* Create the clists  */
  fontsel->font_clist = gtk_clist_new(1);
  gtk_clist_column_titles_hide (GTK_CLIST(fontsel->font_clist));
  gtk_clist_set_policy(GTK_CLIST(fontsel->font_clist), GTK_POLICY_ALWAYS,
		       GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (fontsel->font_clist, FONT_LIST_WIDTH,
			FONT_LIST_HEIGHT);
  gtk_widget_show(fontsel->font_clist);
  gtk_table_attach (GTK_TABLE (table), fontsel->font_clist, 0, 1, 2, 3,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  fontsel->font_style_clist = gtk_clist_new(1);
  gtk_clist_column_titles_hide (GTK_CLIST(fontsel->font_style_clist));
  gtk_clist_set_policy(GTK_CLIST(fontsel->font_style_clist), GTK_POLICY_ALWAYS,
		       GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (fontsel->font_style_clist, FONT_STYLE_LIST_WIDTH, -1);
  gtk_widget_show(fontsel->font_style_clist);
  gtk_table_attach (GTK_TABLE (table), fontsel->font_style_clist, 1, 2, 2, 3,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

  /* The size clist also has the points & pixels buttons below it. */
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_table_attach (GTK_TABLE (table), vbox, 2, 3, 2, 3,
		    GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

  fontsel->size_clist = gtk_clist_new(1);
  gtk_clist_column_titles_hide (GTK_CLIST(fontsel->size_clist));
  gtk_clist_set_policy(GTK_CLIST(fontsel->size_clist), GTK_POLICY_ALWAYS,
		       GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (fontsel->size_clist, FONT_SIZE_LIST_WIDTH, -1);
  gtk_widget_show(fontsel->size_clist);
  gtk_box_pack_start (GTK_BOX (vbox), fontsel->size_clist, TRUE, TRUE, 0);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);

  fontsel->points_button = gtk_radio_button_new_with_label(NULL, "Pts");
  gtk_widget_show (fontsel->points_button);
  gtk_box_pack_start (GTK_BOX (hbox), fontsel->points_button, TRUE, TRUE, 0);
  gtk_tooltips_set_tip(tooltips, fontsel->points_button,
		       "Specify the size in points", NULL);
  if (INITIAL_METRIC == POINTS_METRIC)
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(fontsel->points_button),
				TRUE);

  fontsel->pixels_button = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(fontsel->points_button), "Pix");
  gtk_widget_show (fontsel->pixels_button);
  gtk_box_pack_start (GTK_BOX (hbox), fontsel->pixels_button, TRUE, TRUE, 0);
  gtk_tooltips_set_tip(tooltips, fontsel->pixels_button,
		       "Specify the size in pixels", NULL);
  if (INITIAL_METRIC == PIXELS_METRIC)
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(fontsel->pixels_button),
				TRUE);

  gtk_signal_connect(GTK_OBJECT(fontsel->points_button), "toggled",
		     (GtkSignalFunc) gtk_font_selection_metric_callback,
		     fontsel);
  gtk_signal_connect(GTK_OBJECT(fontsel->pixels_button), "toggled",
		     (GtkSignalFunc) gtk_font_selection_metric_callback,
		     fontsel);

  /* Insert the fonts. If there exist fonts with the same family but
     different foundries, then the foundry name is appended in brackets. */
  gtk_clist_freeze (GTK_CLIST(fontsel->font_clist));
  for (i = 0; i < nfonts; i++) {
    font = &font_info[i];
    if ((i > 0 && !strcmp(font->family, font_info[i-1].family))
	|| (i < nfonts - 1 && !strcmp(font->family, font_info[i+1].family))) {
      sprintf(font_buffer, "%s (%s)", font->family,
	      klass->properties[FOUNDRY][font->foundry]);
      font_item = font_buffer;
      gtk_clist_append(GTK_CLIST(fontsel->font_clist), &font_item);
    } else {
      gtk_clist_append(GTK_CLIST(fontsel->font_clist), &font->family);
    }
  }
  gtk_clist_thaw (GTK_CLIST(fontsel->font_clist));

  gtk_signal_connect (GTK_OBJECT (fontsel->font_clist), "select_row",
		      GTK_SIGNAL_FUNC(gtk_font_selection_select_font),
		      fontsel);
  gtk_signal_connect (GTK_OBJECT (fontsel->font_clist), "expose_event",
		      GTK_SIGNAL_FUNC(gtk_font_selection_expose_list),
		      fontsel);

  gtk_signal_connect (GTK_OBJECT (fontsel->font_style_clist), "select_row",
		      GTK_SIGNAL_FUNC(gtk_font_selection_select_style),
		      fontsel);

  /* Insert the standard font sizes */
  gtk_clist_freeze (GTK_CLIST(fontsel->size_clist));
  size_to_match = INITIAL_FONT_SIZE;
  if (INITIAL_METRIC == POINTS_METRIC) size_to_match = size_to_match / 10;
  for (i = 0; i < sizeof(font_sizes) / sizeof(font_sizes[0]); i++) {
    sprintf(buffer, "%i", font_sizes[i]);
    size = buffer;
    gtk_clist_append(GTK_CLIST(fontsel->size_clist), &size);
    if (font_sizes[i] == size_to_match) {
      gtk_clist_select_row(GTK_CLIST(fontsel->size_clist), i, 0);
      gtk_entry_set_text(GTK_ENTRY(fontsel->size_entry), buffer);
    }
  }
  gtk_clist_thaw (GTK_CLIST(fontsel->size_clist));

  gtk_signal_connect (GTK_OBJECT (fontsel->size_clist), "select_row",
		      GTK_SIGNAL_FUNC(gtk_font_selection_select_size),
		      fontsel);


  /* create the text entry widget */
  text_frame = gtk_frame_new ("Preview:");
  gtk_widget_show (text_frame);
  gtk_frame_set_shadow_type(GTK_FRAME(text_frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (fontsel->main_vbox), text_frame,
		      FALSE, TRUE, 0);

  text_box = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (text_box);
  gtk_container_add (GTK_CONTAINER (text_frame), text_box);
  gtk_container_border_width (GTK_CONTAINER (text_box), 4);

  fontsel->preview_entry = gtk_entry_new ();
  gtk_widget_show (fontsel->preview_entry);
  gtk_widget_set_usize (fontsel->preview_entry, -1, 50);
  gtk_box_pack_start (GTK_BOX (text_box), fontsel->preview_entry,
		      TRUE, TRUE, 0);

  /* Create the message area */
  fontsel->message_label = gtk_label_new("");
  gtk_widget_show (fontsel->message_label);
  gtk_box_pack_start (GTK_BOX (fontsel->main_vbox), fontsel->message_label, 
		      FALSE, FALSE, 0);


  /* Create the action area */
  fontsel->action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(fontsel->action_area),
			    GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(fontsel->action_area), 5);
  gtk_box_pack_start (GTK_BOX (fontsel->top_vbox), fontsel->action_area, 
		      FALSE, FALSE, 0);
  gtk_widget_show (fontsel->action_area);

  fontsel->ok_button = gtk_button_new_with_label("OK");
  GTK_WIDGET_SET_FLAGS (fontsel->ok_button, GTK_CAN_DEFAULT);
  gtk_widget_show(fontsel->ok_button);
  gtk_box_pack_start (GTK_BOX (fontsel->action_area), fontsel->ok_button,
		      TRUE, TRUE, 0);
  gtk_widget_grab_default (fontsel->ok_button);

  fontsel->apply_button = gtk_button_new_with_label("Apply");
  GTK_WIDGET_SET_FLAGS (fontsel->apply_button, GTK_CAN_DEFAULT);
  /*gtk_widget_show(fontsel->apply_button);*/
  gtk_box_pack_start (GTK_BOX(fontsel->action_area), fontsel->apply_button,
		      TRUE, TRUE, 0);

  fontsel->cancel_button = gtk_button_new_with_label("Cancel");
  GTK_WIDGET_SET_FLAGS (fontsel->cancel_button, GTK_CAN_DEFAULT);
  gtk_widget_show(fontsel->cancel_button);
  gtk_box_pack_start (GTK_BOX(fontsel->action_area), fontsel->cancel_button,
		      TRUE, TRUE, 0);
 

  /* Create the font info page */
  fontsel->info_clist = gtk_clist_new_with_titles(3, titles);
  gtk_widget_set_usize (fontsel->info_clist, 390, 150);
  gtk_clist_set_column_width(GTK_CLIST(fontsel->info_clist), 0, 140);
  gtk_clist_set_column_width(GTK_CLIST(fontsel->info_clist), 1, 140);
  gtk_clist_set_column_width(GTK_CLIST(fontsel->info_clist), 2, 140);
  gtk_clist_column_titles_passive(GTK_CLIST(fontsel->info_clist));
  gtk_clist_set_policy(GTK_CLIST(fontsel->info_clist), GTK_POLICY_AUTOMATIC,
		       GTK_POLICY_AUTOMATIC);
  gtk_widget_show(fontsel->info_clist);
  gtk_box_pack_start (GTK_BOX (fontsel->info_vbox), fontsel->info_clist,
		      TRUE, TRUE, 0);

  /* Insert the property names */
  gtk_clist_freeze (GTK_CLIST(fontsel->info_clist));
  row[1] = "";
  row[2] = "";
  for (i = 0; i < GTK_XLFD_NUM_FIELDS; i++) {
    row[0] = xlfd_field_names[i];
    gtk_clist_append(GTK_CLIST(fontsel->info_clist), row);
    gtk_clist_set_shift(GTK_CLIST(fontsel->info_clist), i, 0, 0, 4);
    gtk_clist_set_shift(GTK_CLIST(fontsel->info_clist), i, 1, 0, 4);
    gtk_clist_set_shift(GTK_CLIST(fontsel->info_clist), i, 2, 0, 4);
  }
  gtk_clist_thaw (GTK_CLIST(fontsel->info_clist));

  label = gtk_label_new("Requested Font Name:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (fontsel->info_vbox), label, FALSE, TRUE, 0);

  fontsel->requested_font_name = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(fontsel->requested_font_name), FALSE);
  gtk_widget_show (fontsel->requested_font_name);
  gtk_box_pack_start (GTK_BOX (fontsel->info_vbox),
		      fontsel->requested_font_name, FALSE, TRUE, 0);

  label = gtk_label_new("Actual Font Name:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (fontsel->info_vbox), label, FALSE, TRUE, 0);
  
  fontsel->actual_font_name = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(fontsel->actual_font_name), FALSE);
  gtk_widget_show (fontsel->actual_font_name);
  gtk_box_pack_start (GTK_BOX (fontsel->info_vbox),
		      fontsel->actual_font_name, FALSE, TRUE, 0);

  sprintf(buffer, "%i fonts available with a total of %i styles.",
	  klass->nfonts, klass->ncombos);
  label = gtk_label_new(buffer);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (fontsel->info_vbox), label, FALSE, FALSE, 0);

  gtk_signal_connect (GTK_OBJECT (fontsel->notebook), "switch_page",
		      GTK_SIGNAL_FUNC(gtk_font_selection_switch_page),
		      fontsel);
}

GtkWidget *
gtk_font_selection_new(const gchar *title)
{
  GtkFontSelection *fontsel;

  fontsel = gtk_type_new (gtk_font_selection_get_type ());
  gtk_window_set_title (GTK_WINDOW (fontsel), title ? title : "Font");

  return GTK_WIDGET (fontsel);
}

static void
gtk_font_selection_destroy (GtkObject *object)
{
  GtkFontSelection *fontsel;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_FONT_SELECTION (object));

  fontsel = GTK_FONT_SELECTION (object);

  /* All we have to do is unref the font, if we have one. */
  if (fontsel->font)
    gdk_font_unref (fontsel->font);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/* This is called when the clist is exposed. Here we scroll to the current
   font if necessary. */
static void
gtk_font_selection_expose_list (GtkWidget		*widget,
				GdkEventExpose		*event,
				gpointer		 data)
{
  GtkFontSelection *fontsel;
  GtkFontSelectionClass *klass;
  FontInfo *font_info;
  GList *selection;
  gint index;

  fontsel = GTK_FONT_SELECTION(data);

  if (fontsel->scroll_on_expose) {
    fontsel->scroll_on_expose = FALSE;

    klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
    font_info = klass->font_info;

    /* Try to scroll the font family clist to the selected item */
    selection = GTK_CLIST(fontsel->font_clist)->selection;
    if (selection) {
      index = (gint) selection->data;
      gtk_clist_moveto(GTK_CLIST(fontsel->font_clist), index, 0, 0.5, 0);
    }

    /* Try to scroll the font style clist to the selected item */
    selection = GTK_CLIST(fontsel->font_style_clist)->selection;
    if (selection) {
      index = (gint) selection->data;
      gtk_clist_moveto(GTK_CLIST(fontsel->font_style_clist), index, 0, 0.5, 0);
    }

    /* Try to scroll the font size clist to the selected item */
    selection = GTK_CLIST(fontsel->size_clist)->selection;
    if (selection) {
      index = (gint) selection->data;
      gtk_clist_moveto(GTK_CLIST(fontsel->size_clist), index, 0, 0.5, 0);
    }
  }
}


/* This is called when a family is selected in the list. */
static void
gtk_font_selection_select_font (GtkWidget      *w,
				gint	        row,
				gint	        column,
				GdkEventButton *bevent,
				gpointer        data)
{
  GtkFontSelection *fontsel;
  GtkFontSelectionClass *klass;
  FontInfo *font_info;
  FontInfo *font;

  fontsel = GTK_FONT_SELECTION(data);
  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font_info = klass->font_info;

  font = &font_info[row];
  gtk_entry_set_text(GTK_ENTRY(fontsel->font_entry), font->family);

  /* If it is already the current font, just return. */
  if (fontsel->font_index == row) return;

  fontsel->font_index = row;
  gtk_font_selection_set_available_styles (fontsel);
  gtk_font_selection_select_best_style (fontsel);
}


/* This fills the font style clist with all the possible style combinations
   for the current font family. */
static void
gtk_font_selection_set_available_styles (GtkFontSelection *fontsel)
{
  GtkFontSelectionClass *klass;
  FontInfo *font;
  gint combo;
  gint weight_index, slant_index, set_width_index, spacing_index;
  gchar *weight, *slant, *set_width, *spacing, *new_item;
  gchar buffer[XLFD_MAX_FIELD_LEN * 5 + 2];
  guint16 *combos;

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font = &klass->font_info[fontsel->font_index];
  combos = klass->combos + font->combo_index;

  gtk_clist_freeze (GTK_CLIST(fontsel->font_style_clist));
  gtk_clist_clear (GTK_CLIST(fontsel->font_style_clist));

  for (combo = 0; combo < font->ncombos; combo++)
    {
      weight_index    = combos[combo * GTK_NUM_FONT_PROPERTIES + WEIGHT];
      slant_index     = combos[combo * GTK_NUM_FONT_PROPERTIES + SLANT];
      set_width_index = combos[combo * GTK_NUM_FONT_PROPERTIES + SET_WIDTH];
      spacing_index   = combos[combo * GTK_NUM_FONT_PROPERTIES + SPACING];
      weight    = klass->properties[WEIGHT]   [weight_index];
      slant     = klass->properties[SLANT]    [slant_index];
      set_width = klass->properties[SET_WIDTH][set_width_index];
      spacing   = klass->properties[SPACING]  [spacing_index];

      /* Convert '(nil)' weights to 'regular', since it looks nicer. */
      if      (!g_strcasecmp(weight, "(nil)"))	   weight = "regular";

      /* We don't show default values or '(nil)'s in the other properties. */
      if      (!g_strcasecmp(slant, "r"))          slant = NULL;
      else if (!g_strcasecmp(slant, "(nil)"))      slant = NULL;
      else if (!g_strcasecmp(slant, "i"))          slant = "italic";
      else if (!g_strcasecmp(slant, "o"))          slant = "oblique";
      else if (!g_strcasecmp(slant, "ri"))         slant = "reverse italic";
      else if (!g_strcasecmp(slant, "ro"))         slant = "reverse oblique";
      else if (!g_strcasecmp(slant, "ot"))         slant = "other";

      if      (!g_strcasecmp(set_width, "normal")) set_width = NULL;
      else if (!g_strcasecmp(set_width, "(nil)"))  set_width = NULL;

      if      (!g_strcasecmp(spacing, "p"))        spacing = NULL;
      else if (!g_strcasecmp(spacing, "(nil)"))    spacing = NULL;
      else if (!g_strcasecmp(spacing, "m"))        spacing = "[M]";
      else if (!g_strcasecmp(spacing, "c"))        spacing = "[C]";

      /* Add the strings together, making sure there is 1 space between them */
      strcpy(buffer, weight);
      if (slant) {
	strcat(buffer, " ");
	strcat(buffer, slant);
      }
      if (set_width) {
	strcat(buffer, " ");
	strcat(buffer, set_width);
      }
      if (spacing) {
	strcat(buffer, " ");
	strcat(buffer, spacing);
      }

      new_item = buffer;
      gtk_clist_append(GTK_CLIST(fontsel->font_style_clist), &new_item);
    }
  gtk_clist_thaw (GTK_CLIST(fontsel->font_style_clist));
}


/* This selects a style when the user selects a font. It just uses the first
   available style at present. I was thinking of trying to maintain the
   selected style, e.g. bold italic, when the user selects different fonts.
   However, the interface is so easy to use now I'm not sure it's worth it. */
static void
gtk_font_selection_select_best_style(GtkFontSelection *fontsel)
{
  GtkFontSelectionClass *klass;
  FontInfo *font;
  guint16 *combos;
  gint prop, best_combo = 0;

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font = &klass->font_info[fontsel->font_index];
  combos = klass->combos + font->combo_index;

  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++)
    fontsel->property_values[prop]
      = combos[best_combo * GTK_NUM_FONT_PROPERTIES + prop];

  /* This will activate the select style callback and load the font. */
  gtk_clist_select_row(GTK_CLIST(fontsel->font_style_clist), best_combo, 0);
}


/* This is called when a style is selected in the list. */
static void
gtk_font_selection_select_style (GtkWidget      *w,
				 gint	        row,
				 gint	        column,
				 GdkEventButton *bevent,
				 gpointer        data)
{
  GtkFontSelection *fontsel;
  GtkFontSelectionClass *klass;
  FontInfo *font_info;
  FontInfo *font;
  guint16 *combos;
  gint combo, prop;
  gchar *text;

  fontsel = GTK_FONT_SELECTION(data);
  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font_info = klass->font_info;
  font = &font_info[fontsel->font_index];
  combos = klass->combos + font->combo_index;

  gtk_clist_get_text(GTK_CLIST(fontsel->font_style_clist), row, 0, &text);
  gtk_entry_set_text(GTK_ENTRY(fontsel->font_style_entry), text);

  /* The list row corresponds to the combo index, so we just need to copy
     the combo values into the fontsel and reload the font. */
  combo = row;
  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++)
    fontsel->property_values[prop] = combos[combo * GTK_NUM_FONT_PROPERTIES
					   + prop];
  gtk_font_selection_load_font (fontsel);
}


/* This tries to select a size in the list, based on the fonts current size. */
static void
gtk_font_selection_show_size (GtkFontSelection *fontsel)
{
  gboolean match = TRUE, found = FALSE;
  gint size_to_match;
  gint i;

  size_to_match = fontsel->size;
  if (fontsel->metric == POINTS_METRIC) {
    /* If were using points and the value has deci-points then we can't match
       it. But we can move the clist to show the nearest values. */
    if (size_to_match % 10 != 0) match = FALSE;
    size_to_match = (size_to_match + 5) / 10;
  }

  /* We assume the font sizes are in ascending order to find the closest
     match. */
  for (i = 0; i < sizeof(font_sizes) / sizeof(font_sizes[0]); i++) {
    if (size_to_match == font_sizes[i]) {
      if (match) {
	gtk_clist_select_row(GTK_CLIST(fontsel->size_clist), i, 0);
	found = TRUE;
      }
      gtk_clist_moveto(GTK_CLIST(fontsel->size_clist), i, 0, 0.5, 0);
      break;
    } else if (size_to_match < font_sizes[i]) {
      gtk_clist_moveto(GTK_CLIST(fontsel->size_clist), i, 0, 0.5, 0);
      break;
    }
  }

  /* If the size isn't in the list we clear the selection. */
  if (!found) {
    GList *selection = GTK_CLIST(fontsel->size_clist)->selection;
    if (selection) {
      gint row = (gint) selection->data;
      gtk_clist_unselect_row(GTK_CLIST(fontsel->size_clist), row, 0);
    }
  }
}


/* If the user hits return in the font size entry, we change to the new font
   size. */
static gint
gtk_font_selection_size_key_press (GtkWidget   *w,
				   GdkEventKey *event,
				   gpointer     data)
{
  GtkFontSelection *fontsel;
  gint new_size;
  gfloat new_size_float;
  gchar *text;
  gchar buffer[32];

  fontsel = GTK_FONT_SELECTION(data);

  if (event->keyval == GDK_Return)
    {
      text = gtk_entry_get_text (GTK_ENTRY (fontsel->size_entry));
      if (fontsel->metric == PIXELS_METRIC) {
	new_size = atoi (text);
	if (new_size < 2) {
	  new_size = 2;
	}
	sprintf(buffer, "%i", new_size);
	gtk_entry_set_text (GTK_ENTRY (fontsel->size_entry), buffer);
      } else {
	new_size_float = atof (text) * 10;
	new_size = (gint) new_size_float;
	if (new_size < 20) {
	  new_size = 20;
	}
	if (new_size % 10 == 0)
	  sprintf(buffer, "%i", new_size / 10);
	else
	  sprintf(buffer, "%i.%i", new_size / 10, new_size % 10);
	gtk_entry_set_text (GTK_ENTRY (fontsel->size_entry), buffer);
      }

      /* Check if the font size has changed, and return if it hasn't. */
      if (fontsel->size == new_size) return TRUE;

      fontsel->size = new_size;
      gtk_font_selection_show_size (fontsel);
      gtk_font_selection_load_font (fontsel);
      return TRUE;
    }

  return FALSE;
}


/* This is called when a size is selected in the list. */
static void
gtk_font_selection_select_size (GtkWidget      *w,
				gint	        row,
				gint	        column,
				GdkEventButton *bevent,
				gpointer        data)
{
  GtkFontSelection *fontsel;
  gint new_size;
  gchar *text;

  fontsel = GTK_FONT_SELECTION(data);

  gtk_clist_get_text(GTK_CLIST(fontsel->size_clist), row, 0, &text);
  gtk_entry_set_text(GTK_ENTRY(fontsel->size_entry), text);

  /* Check if the font size has changed, and return if it hasn't. */
  if (fontsel->metric == PIXELS_METRIC) {
    new_size = atoi(text);
  } else {
    new_size = atoi(text) * 10;
  }

  if (fontsel->size == new_size) return;

  fontsel->size = new_size;
  gtk_font_selection_load_font (fontsel);
}



static void
gtk_font_selection_metric_callback (GtkWidget *w,
				    gpointer   data)
{
  GtkFontSelection *fontsel = GTK_FONT_SELECTION(data);
  gchar buffer[32];

  if (GTK_TOGGLE_BUTTON(fontsel->pixels_button)->active)
    {
      if (fontsel->metric == PIXELS_METRIC) return;
      fontsel->metric = PIXELS_METRIC;
      fontsel->size = (fontsel->size + 5) / 10;
      sprintf(buffer, "%i", fontsel->size);
      gtk_entry_set_text (GTK_ENTRY (fontsel->size_entry), buffer);
      gtk_font_selection_show_size (fontsel);
    }
  else
    {
      if (fontsel->metric == POINTS_METRIC) return;
      fontsel->metric = POINTS_METRIC;
      fontsel->size *= 10;
    }
  gtk_font_selection_load_font (fontsel);
}


/* This searches the given property table and returns the index of the given
   string, or 0, which is the wildcard '*' index, if it's not found. */
static guint16
gtk_font_selection_field_to_index (gchar **table,
				   gint    ntable,
				   gchar  *field)
{
  gint i;

  for (i = 0; i < ntable; i++)
    if (strcmp (field, table[i]) == 0)
      return i;

  return 0;
}



/* This attempts to load the current font, and returns TRUE if it succeeds. */
static gboolean
gtk_font_selection_load_font (GtkFontSelection *fontsel)
{
  GdkFont *font;
  gchar *fontname;

  if (fontsel->font)
    gdk_font_unref (fontsel->font);
  fontsel->font = NULL;

  /* If no family has been selected yet, just return FALSE. */
  if (fontsel->font_index == -1) return FALSE;

  fontname = gtk_font_selection_get_font_name (fontsel);
  if (fontname)
    {
      font = gdk_font_load (fontname);
      g_free(fontname);

      if (font)
	{
	  fontsel->font = font;
	  gtk_label_set(GTK_LABEL(fontsel->message_label), "");
	  gtk_font_selection_update_preview (fontsel);
	  return TRUE;
	}
      else 
	{
	  gtk_label_set(GTK_LABEL(fontsel->message_label),
			"The selected font is not available.");
	}
    }
  else
    {
      gtk_label_set(GTK_LABEL(fontsel->message_label),
		    "The selected font is not a valid font.");
    }

  return FALSE;
}


/* This sets the font in the preview entry to the selected font, and tries to
   make sure that the preview entry is a reasonable size, i.e. so that the
   text can be seen with a bit of space to spare. But it tries to avoid
   resizing the entry every time the font changes.
   This also used to shrink the preview if the font size was decreased, but
   that made it awkward if the user wanted to resize the window themself. */
static void
gtk_font_selection_update_preview (GtkFontSelection *fontsel)
{
  GtkWidget *preview_entry;
  GtkStyle *style;
  gint text_height;
  gchar *text;

  style = gtk_style_new ();
  gdk_font_unref (style->font);
  style->font = fontsel->font;
  gdk_font_ref (style->font);

  preview_entry = fontsel->preview_entry;
  gtk_widget_set_style (preview_entry, style);
  gtk_style_unref(style);

  text_height = preview_entry->style->font->ascent
    + preview_entry->style->font->descent;
  /* We don't ever want to be over MAX_PREVIEW_HEIGHT pixels high. */
  if (text_height > MAX_PREVIEW_HEIGHT) text_height = MAX_PREVIEW_HEIGHT;
  if (preview_entry->allocation.height < text_height + 10) {
    gtk_widget_set_usize(preview_entry, -1, text_height + 20);
  }

  /* Try to ensure that the sample text is visible - sometimes after
     fonts using different languages are displayed the text disappears. */
  text = gtk_entry_get_text(GTK_ENTRY(fontsel->preview_entry));
  if (strlen(text) == 0)
    gtk_entry_set_text(GTK_ENTRY(fontsel->preview_entry), PREVIEW_TEXT);
  gtk_entry_set_position(GTK_ENTRY(fontsel->preview_entry), 0);
}


static void
gtk_font_selection_switch_page (GtkWidget       *w,
				GtkNotebookPage *page,
				gint             page_num,
				gpointer         data)
{
  GtkFontSelection *fontsel = GTK_FONT_SELECTION(data);
  Atom font_atom, atom;
  Bool status;
  char *name;
  gchar *fontname;
  gchar field_buffer[XLFD_MAX_FIELD_LEN];
  gchar *field;
  gint i;
  gboolean shown_actual_fields = FALSE;

  if (page_num != 1) return;

  /* This function strangely gets called when the window is destroyed,
     so we check here to see if the notebook is visible. */
  if (!GTK_WIDGET_VISIBLE(w)) return;

  fontname = gtk_font_selection_get_font_name(fontsel);
  gtk_entry_set_text(GTK_ENTRY(fontsel->requested_font_name),
		     fontname ? fontname : "");

  gtk_clist_freeze (GTK_CLIST(fontsel->info_clist));
  for (i = 0; i < GTK_XLFD_NUM_FIELDS; i++) {
    if (fontname)
      field = gtk_font_selection_get_xlfd_field (fontname, i, field_buffer);
    else
      field = NULL;
    if (field) {
      if (i == XLFD_SLANT)
	field = gtk_font_selection_expand_slant_code(field);
      else if (i == XLFD_SPACING)
	field = gtk_font_selection_expand_spacing_code(field);
    }
    gtk_clist_set_text(GTK_CLIST(fontsel->info_clist), i, 1,
		  field ? field : "");
  }

  if (fontsel->font) {
    font_atom = XInternAtom(GDK_DISPLAY(), "FONT", True);
    if (font_atom != None) {
      status = XGetFontProperty(GDK_FONT_XFONT(fontsel->font), font_atom,
				&atom);
      if (status == True) {
	name = XGetAtomName(GDK_DISPLAY(), atom);
	gtk_entry_set_text(GTK_ENTRY(fontsel->actual_font_name), name);
	
	for (i = 0; i < GTK_XLFD_NUM_FIELDS; i++) {
	  field = gtk_font_selection_get_xlfd_field (name, i, field_buffer);
	  if (i == XLFD_SLANT)
	    field = gtk_font_selection_expand_slant_code(field);
	  else if (i == XLFD_SPACING)
	    field = gtk_font_selection_expand_spacing_code(field);
	  gtk_clist_set_text(GTK_CLIST(fontsel->info_clist), i, 2,
			     field ? field : "");
	}
	shown_actual_fields = TRUE;
	XFree(name);
      }
    }
  }
  if (!shown_actual_fields) {
    for (i = 0; i < GTK_XLFD_NUM_FIELDS; i++) {
      gtk_clist_set_text(GTK_CLIST(fontsel->info_clist), i, 2,
			 fontname ? "(unknown)" : "");
    }
  }
  gtk_clist_thaw (GTK_CLIST(fontsel->info_clist));
  g_free(fontname);
}


static gchar*
gtk_font_selection_expand_slant_code(gchar *slant)
{
  if      (!g_strcasecmp(slant, "r"))   return("r (roman)");
  else if (!g_strcasecmp(slant, "i"))   return("i (italic)");
  else if (!g_strcasecmp(slant, "o"))   return("o (oblique)");
  else if (!g_strcasecmp(slant, "ri"))  return("ri (reverse italic)");
  else if (!g_strcasecmp(slant, "ro"))  return("ro (reverse oblique)");
  else if (!g_strcasecmp(slant, "ot"))  return("ot (other)");
  return slant;
}

static gchar*
gtk_font_selection_expand_spacing_code(gchar *spacing)
{
  if      (!g_strcasecmp(spacing, "p")) return("p (proportional)");
  else if (!g_strcasecmp(spacing, "m")) return("m (monospaced)");
  else if (!g_strcasecmp(spacing, "c")) return("c (char cell)");
  return spacing;
}


/*****************************************************************************
 * These functions all deal with creating the main class arrays containing
 * the data about all available fonts.
 *****************************************************************************/
static void
gtk_font_selection_get_fonts (GtkFontSelectionClass *klass)
{
  gchar **xfontnames;
  GSList **fontnames;
  gchar *fontname;
  GSList * temp_list;
  gint num_fonts;
  gint i, prop, combo;
  FontInfo *font;
  guint16 *current_combo, *prev_combo;
  gboolean matched_combo;

  /* Get a maximum of MAX_FONTS fontnames from the X server. */
  xfontnames = XListFonts (GDK_DISPLAY(), "-*-*-*-*-*-*-0-0-*-*-*-0-*-*",
			   MAX_FONTS, &num_fonts);
  /* Output a warning if we actually get MAX_FONTS fonts. */
  if (num_fonts == MAX_FONTS)
    g_warning("MAX_FONTS exceeded. Some fonts may be missing.");

  /* The maximum size of both these tables is the number of font names
     returned. We realloc them both later when we know exactly how many
     unique entries there are. */
  klass->font_info = g_new (FontInfo, num_fonts);
  klass->combos = g_new (guint16, num_fonts * GTK_NUM_FONT_PROPERTIES);
  fontnames = g_new (GSList*, num_fonts);

  /* Create the initial arrays for the property value strings, though they
     may be realloc'ed later. Put the wildcard '*' in the first elements. */
  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++) {
    klass->properties[prop] = g_new(gchar*, PROPERTY_ARRAY_INCREMENT);
    klass->space_allocated[prop] = PROPERTY_ARRAY_INCREMENT;
    klass->nproperties[prop] = 1;
    klass->properties[prop][0] = "*";
  }


  /* Insert the font families into the main table, sorted by family and
     foundry (fonts with different foundries are placed in seaparate FontInfos.
     All fontnames in each family + foundry are placed into the fontnames
     array of lists. */
  klass->nfonts = 0;
  for (i = 0; i < num_fonts; i++)
    {
      /*g_print("%6i: %s\n", i, xfontnames[i]);*/
      if (gtk_font_selection_is_xlfd_font_name (xfontnames[i]))
	{
	  gtk_font_selection_insert_font (klass, fontnames,
					  &klass->nfonts, xfontnames[i]);
	}
      else
	{
	  /* This is just for debugging. */
	  /*g_warning("Skipping invalid font: %s", xfontnames[i]);*/
	}
    }


  /* Since many font names will be in the same FontInfo not all of the
     allocated FontInfo table will be used, so we will now reallocate it
     with the real size. */
  klass->font_info = g_realloc(klass->font_info,
			       sizeof(FontInfo) * klass->nfonts);


  /* Now we work out which choices of weight/slant etc. are valid for each
     font. */
  klass->ncombos = 0;
  current_combo = klass->combos;
  for (i = 0; i < klass->nfonts; i++)
    {
      font = &klass->font_info[i];

      /* Use the next free position in the combos array. */
      font->combo_index = klass->ncombos * GTK_NUM_FONT_PROPERTIES;

      /* Now step through each of the fontnames with this family, and create
	 a combo (array of five guint16s) for each fontname. Each contains
	 the index into the weights/slants etc. arrays. */
      combo = 0;
      temp_list = fontnames[i];
      while (temp_list)
	{
	  fontname = temp_list->data;
	  temp_list = temp_list->next;

	  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++) {
	    current_combo[prop]
	      = gtk_font_selection_insert_field (klass, fontname, prop);
	  }

	  /* Now we check to make sure that the combo is unique. If it isn't
	     we forget it. */
	  prev_combo = klass->combos + font->combo_index;
	  matched_combo = FALSE;
	  while (prev_combo < current_combo) {
	    matched_combo = TRUE;
	    for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++) {
	      if (prev_combo[prop] != current_combo[prop]) {
		matched_combo = FALSE;
		break;
	      }
	    }
	    if (matched_combo) break;
	    prev_combo += GTK_NUM_FONT_PROPERTIES;
	  }

	  if (!matched_combo) {
	    combo++;
	    klass->ncombos++;
	    current_combo += GTK_NUM_FONT_PROPERTIES;
	  }
	}
      g_slist_free(fontnames[i]);

      /* Set ncombos to the real value, minus duplicated fontnames.
	 Note that we aren't using all the allocated memory if fontnames are
	 duplicated. */
      font->ncombos = combo;
    }

  /* Since some repeated combos may be skipped we won't have used all the
     allocated space, so we will now reallocate it with the real size. */
  klass->combos = g_realloc(klass->combos, sizeof(guint16)
			    * GTK_NUM_FONT_PROPERTIES * klass->ncombos);
  g_free(fontnames);
  XFreeFontNames (xfontnames);


  /* Debugging Output */
  /* This outputs all FontInfos. */
  /*
  g_print("\n\n Font Family           Foundry    Set Width  Spacing    Weight     Slant\n\n");
  for (i = 0; i < klass->nfonts; i++)
    {
      FontInfo *font = &klass->font_info[i];
      guint16 *combos = klass->combos + font->combo_index;
      for (combo = 0; combo < font->ncombos; combo++)
	{
	  g_print("%5i %-16.16s ", i, font->family);
	  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++)
	    g_print("%-10.10s ", klass->properties[prop][*combos++]);
	  g_print("\n");
	}
    }
    */

  /* This outputs all available properties. */
  /*
  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++)
    {
      g_print("Property: %s\n", xlfd_field_names[xlfd_index[prop]]);
      for (i = 0; i < klass->nproperties[prop]; i++)
	{
	  g_print("  %s\n", klass->properties[prop][i]);
	}
    }
    */
}

/* This inserts the given fontname into the FontInfo table.
   If a FontInfo already exists with the same family and foundry, then the
   fontname is added to the FontInfos list of fontnames, else a new FontInfo
   is created and inserted in alphabetical order in the table. */
static void
gtk_font_selection_insert_font (GtkFontSelectionClass *klass,
				GSList		      *fontnames[],
				gint		      *ntable,
				gchar		      *fontname)
{
  FontInfo *table;
  FontInfo temp_info;
  GSList *temp_fontname;
  gchar *family;
  gint foundry;
  gint lower, upper;
  gint middle, cmp;
  gchar family_buffer[XLFD_MAX_FIELD_LEN];

  table = klass->font_info;

  /* insert a fontname into a table */
  family = gtk_font_selection_get_xlfd_field (fontname, XLFD_FAMILY,
					      family_buffer);
  if (!family)
    return;

  foundry = gtk_font_selection_insert_field(klass, fontname, FOUNDRY);

  lower = 0;
  if (*ntable > 0)
    {
      /* Do a binary search to determine if we have already encountered
       *  a font with this family & foundry. */
      upper = *ntable;
      while (lower < upper)
	{
	  middle = (lower + upper) >> 1;

	  cmp = strcmp (family, table[middle].family);
	  /* If the family matches we sort by the foundry. */
	  if (cmp == 0)
	    cmp = strcmp(klass->properties[FOUNDRY][foundry],
			 klass->properties[FOUNDRY][table[middle].foundry]);

	  if (cmp == 0)
	    {
	      fontnames[middle] = g_slist_prepend (fontnames[middle],
						   fontname);
	      return;
	    }
	  else if (cmp < 0)
	    upper = middle;
	  else
	    lower = middle+1;
	}
    }

  /* Add another entry to the table for this new font family */
  temp_info.family = g_strdup(family);
  temp_info.foundry = foundry;
  temp_fontname = g_slist_prepend (NULL, fontname);

  (*ntable)++;

  /* Quickly insert the entry into the table in sorted order
   *  using a modification of insertion sort and the knowledge
   *  that the entries proper position in the table was determined
   *  above in the binary search and is contained in the "lower"
   *  variable. */
  if (*ntable > 1)
    {
      upper = *ntable - 1;
      while (lower != upper)
	{
	  table[upper] = table[upper-1];
	  fontnames[upper] = fontnames[upper-1];
	  upper--;
	}
    }
  table[lower] = temp_info;
  fontnames[lower] = temp_fontname;
}


/* This checks that the specified field of the given fontname is in the
   appropriate properties array. If not it is added. Thus eventually we get
   arrays of all possible weights/slants etc. It returns the array index. */
static gint
gtk_font_selection_insert_field (GtkFontSelectionClass *klass,
				 gchar		       *fontname,
				 gint			prop)
{
  gchar field_buffer[XLFD_MAX_FIELD_LEN];
  gchar *field;
  guint16 index;

  field = gtk_font_selection_get_xlfd_field (fontname, xlfd_index[prop],
					     field_buffer);
  if (!field) return 0;
  
  /* If the field is already in the array just return its index. */
  for (index = 0; index < klass->nproperties[prop]; index++) {
    if (!strcmp(field, klass->properties[prop][index])) return index;
  }

  /* Make sure we have enough space to add the field. */
  if (klass->nproperties[prop] == klass->space_allocated[prop]) {
    klass->space_allocated[prop] += PROPERTY_ARRAY_INCREMENT;
    klass->properties[prop] = g_realloc(klass->properties[prop],
					sizeof(gchar*)
					* klass->space_allocated[prop]);
  }

  /* Add the new field. */
  index = klass->nproperties[prop];
  klass->properties[prop][index] = g_strdup(field);
  klass->nproperties[prop]++;
  return index;
}


/*****************************************************************************
 * These functions are the main public interface for getting/setting the font.
 *****************************************************************************/

GdkFont*
gtk_font_selection_get_font (GtkFontSelection *fontsel)
{
  g_return_val_if_fail (fontsel != NULL, NULL);
  return fontsel->font;
}


gchar *
gtk_font_selection_get_font_name (GtkFontSelection *fontsel)
{
  GtkFontSelectionClass *klass;
  FontInfo *font;
  gchar *family_str, *foundry_str;
  gchar *property_str[GTK_NUM_FONT_PROPERTIES];
  gint prop;

  g_return_val_if_fail (fontsel != NULL, NULL);
  g_return_val_if_fail (GTK_IS_FONT_SELECTION (fontsel), NULL);

  /* If no family has been selected return NULL. */
  if (fontsel->font_index == -1) return NULL;

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font = &klass->font_info[fontsel->font_index];
  family_str = font->family;
  foundry_str = klass->properties[FOUNDRY][font->foundry];

  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++) {
    property_str[prop]
      = klass->properties[prop][fontsel->property_values[prop]];
    if (strcmp (property_str[prop], "(nil)") == 0) property_str[prop] = "";
  }

  return gtk_font_selection_create_xlfd (fontsel->size,
					 fontsel->metric,
					 foundry_str,
					 family_str,
					 property_str[WEIGHT],
					 property_str[SLANT],
					 property_str[SET_WIDTH],
					 property_str[SPACING]);
}


/* This returns the combo with the best match to the current fontsel setting */
static gint
gtk_font_selection_get_best_match(GtkFontSelection *fontsel)
{
  GtkFontSelectionClass *klass;
  FontInfo *font;
  guint16 *combos;
  gint prop, combo, best_combo = 0, matched, best_matched = 0;

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font = &klass->font_info[fontsel->font_index];
  combos = klass->combos + font->combo_index;

  /* Find the combo with the most matches. */
  for (combo = 0; combo < font->ncombos; combo++) {
    matched = 0;
    for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++) {
      if (fontsel->property_values[prop]
	  == combos[combo * GTK_NUM_FONT_PROPERTIES + prop])
	matched++;
    }
    if (matched > best_matched) {
      best_matched = matched;
      best_combo = combo;
    }
  }

  return best_combo;
}


/* This sets the current font, selecting the appropriate clist rows.
   First we check the fontname is valid and try to find the font family
   - i.e. the name in the main list. If we can't find that, then just return.
   Next we try to set each of the properties according to the fontname.
   Finally we select the font family & style in the clists.
   Note that we have to be careful to make sure any callbacks do not try
   to load the font unless we want them to. This is usually done by
   setting the font/size in the fontsel struct before selecting rows or
   buttons in the interface. The callbacks simply return if the value has
   not changed. */
gboolean
gtk_font_selection_set_font_name (GtkFontSelection *fontsel,
				  const gchar      *fontname)
{
  GtkFontSelectionClass *klass;
  gchar *family, *field;
  gint index, prop, size, combo;
  guint16 foundry, value;
  gchar family_buffer[XLFD_MAX_FIELD_LEN];
  gchar field_buffer[XLFD_MAX_FIELD_LEN];
  gchar buffer[16];

  g_return_val_if_fail (fontsel != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_FONT_SELECTION (fontsel), FALSE);
  g_return_val_if_fail (fontname != NULL, FALSE);

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);

  /* Check it is a valid fontname. */
  if (!gtk_font_selection_is_xlfd_font_name(fontname))
    return FALSE;

  family = gtk_font_selection_get_xlfd_field (fontname, XLFD_FAMILY,
					      family_buffer);
  if (!family)
    return FALSE;

  field = gtk_font_selection_get_xlfd_field (fontname, XLFD_FOUNDRY,
					       field_buffer);
  foundry = gtk_font_selection_field_to_index (klass->properties[FOUNDRY],
					       klass->nproperties[FOUNDRY],
					       field);

  index = gtk_font_selection_find_font(fontsel, family, foundry);
  if (index == -1)
    return FALSE;

  /* Convert the property fields into indices and set them. */
  for (prop = 0; prop < GTK_NUM_FONT_PROPERTIES; prop++) 
    {
      field = gtk_font_selection_get_xlfd_field (fontname, xlfd_index[prop],
						 field_buffer);
      value = gtk_font_selection_field_to_index (klass->properties[prop],
						 klass->nproperties[prop],
						 field);
      fontsel->property_values[prop] = value;
    }

  field = gtk_font_selection_get_xlfd_field (fontname, XLFD_POINTS,
					     field_buffer);
  size = atoi(field);
  if (size > 0) {
    if (size < 20) size = 20;
    fontsel->size = size;
    fontsel->metric = POINTS_METRIC;
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(fontsel->points_button),
				TRUE);
    if (size % 10 == 0)
      sprintf (buffer, "%i", size / 10);
    else
      sprintf (buffer, "%i.%i", size / 10, size % 10);
  } else {
    field = gtk_font_selection_get_xlfd_field (fontname, XLFD_PIXELS,
					       field_buffer);
    size = atoi(field);
    if (size < 2) size = 2;
    fontsel->size = size;
    fontsel->metric = PIXELS_METRIC;
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(fontsel->pixels_button),
				TRUE);
    sprintf (buffer, "%i", size);
  }
  gtk_entry_set_text (GTK_ENTRY (fontsel->size_entry), buffer);
  gtk_font_selection_show_size (fontsel);


  /* Now find the best style match. */
  fontsel->font_index = index;
  gtk_clist_select_row(GTK_CLIST(fontsel->font_clist), index, 0);
  if (GTK_WIDGET_MAPPED (fontsel->font_clist))
    gtk_clist_moveto(GTK_CLIST(fontsel->font_clist), index, 0, 0.5, 0);
  else
    fontsel->scroll_on_expose = TRUE;

  gtk_font_selection_set_available_styles (fontsel);
  combo = gtk_font_selection_get_best_match (fontsel);

  /* This time we actually want the callback to load the font. */
  gtk_clist_select_row(GTK_CLIST(fontsel->font_style_clist), combo, 0);

  return TRUE;
}


/* Returns the index of the given family, or -1 if not found */
static gint
gtk_font_selection_find_font (GtkFontSelection *fontsel,
			      gchar	       *family,
			      guint16		foundry)
{
  GtkFontSelectionClass *klass;
  FontInfo *font_info;
  gint lower, upper, middle = -1, cmp, nfonts;

  klass = GTK_FONT_SELECTION_CLASS(GTK_OBJECT(fontsel)->klass);
  font_info = klass->font_info;
  nfonts = klass->nfonts;
  if (nfonts == 0) return -1;

  /* Do a binary search to find the font family. */
  lower = 0;
  upper = nfonts;
  while (lower < upper)
    {
      middle = (lower + upper) >> 1;
      
      cmp = strcmp (family, font_info[middle].family);
      if (cmp == 0)
	cmp = strcmp(klass->properties[FOUNDRY][foundry],
		     klass->properties[FOUNDRY][font_info[middle].foundry]);

      if (cmp == 0)
	return middle;
      else if (cmp < 0)
	upper = middle;
      else if (cmp > 0)
	lower = middle+1;
    }

  /* If we can't match family & foundry see if just the family matches */
  if (!strcmp (family, font_info[middle].family)) return middle;

  return -1;
}


/*****************************************************************************
 * These functions all deal with X Logical Font Description (XLFD) fontnames.
 * See the freely available documentation about this.
 *****************************************************************************/

/*
 * Returns TRUE if the fontname is a valid XLFD.
 * (It just checks if the number of dashes is 14, and that each
 * field < XLFD_MAX_FIELD_LEN  characters long - that's not in the XLFD but it
 * makes it easier for me).
 */
static gboolean
gtk_font_selection_is_xlfd_font_name (const gchar *fontname)
{
  gint i = 0;
  gint field_len = 0;

  while (*fontname) {
    if (*fontname++ == '-')
      {
	if (field_len > XLFD_MAX_FIELD_LEN) return FALSE;
	field_len = 0;
	i++;
      }
    else
      {
	field_len++;
      }
  }

  return (i == 14) ? TRUE : FALSE;
}

/*
 * This fills the buffer with the specified field from the X Logical Font
 * Description name, and returns it. If fontname is NULL or the field is
 * longer than XFLD_MAX_FIELD_LEN it returns NULL.
 */
static gchar*
gtk_font_selection_get_xlfd_field (const gchar *fontname,
				   gint         field_num,
				   gchar       *buffer)
{
  const gchar *t1, *t2;
  gint len;

  if (!fontname) return NULL;

  /* we assume this is a valid fontname...that is, it has 14 fields */

  t1 = fontname;
  while (*t1 && (field_num >= 0))
    if (*t1++ == '-')
      field_num--;

  t2 = t1;
  while (*t2 && (*t2 != '-'))
    t2++;

  if (t1 != t2)
    {
      /* Check we don't overflow the buffer */
      len = (long) t2 - (long) t1;
      if (len > XLFD_MAX_FIELD_LEN - 1) return NULL;
      strncpy (buffer, t1, len);
      buffer[len] = 0;
    }
  else
    {
      strcpy(buffer, "(nil)");
    }

  return buffer;
}

/*
 * This returns a X Logical Font Description font name, given all the pieces.
 * Note: this retval must be freed by the caller.
 */
static gchar *
gtk_font_selection_create_xlfd (gint     size,
				gint     metric,
				gchar   *foundry,
				gchar   *family,
				gchar   *weight,
				gchar   *slant,
				gchar   *set_width,
				gchar   *spacing)
{
  gchar buffer[16];
  gchar *pixel_size = "*", *point_size = "*", *fontname;
  gint length;

  if (size <= 0) return NULL;

  sprintf (buffer, "%d", (int) size);
  if (metric == PIXELS_METRIC) pixel_size = buffer;
  else point_size = buffer;
    
  /* Note: be careful here - don't overrun the allocated memory. */
  length = strlen(foundry) + strlen(family) + strlen(weight) + strlen(slant)
    + strlen(set_width) + strlen(pixel_size) + strlen(point_size)
    + strlen(spacing) + 1 + 1 + 1 + 1 + 1 + 3 + 1 + 5 + 6  + 1;

  fontname = g_new(gchar, length);
  /* **NOTE**: If you change this string please change length above! */
  sprintf(fontname, "-%s-%s-%s-%s-%s-*-%s-%s-*-*-%s-*-*-*",
	  foundry, family, weight, slant, set_width, pixel_size,
	  point_size, spacing);
  return fontname;
}

