/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * gmutex.c:
 * Copyright 1998 Sebastian Wilhelmi; University of Karlsruhe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* 
 * MT safe
 */

#include <glib.h>

gboolean g_mutex_use_default_impl = TRUE;
gboolean g_mutex_supported = FALSE;
gboolean g_cond_supported = FALSE;
gboolean g_private_supported = FALSE;
static GMutex *g_mutex_protect_static_mutex_allocation = NULL;
GMutexFunctions g_mutex_functions_for_glib_use; /* is NULLified as default */

/* If this is set, noone might call the thread init, beacuse that
   would hose the whole system, as it is already setup not to use
   threads */
gboolean g_forbid_thread_initilization = FALSE;

GMutex* g_static_mutex_get_mutex_impl(GMutex** mutex)
{
  if( !g_mutex_supported )
    {
      return NULL;
    }
  g_assert( g_mutex_protect_static_mutex_allocation );

  g_mutex_lock( g_mutex_protect_static_mutex_allocation );

  if( !(*mutex) ) *mutex = g_mutex_new(); 

  g_mutex_unlock( g_mutex_protect_static_mutex_allocation );
  
  return *mutex;
}

static G_LOCK(gprivate_global);
gpointer g_private_key_for_static_private = NULL;
static guint g_static_private_next_id = 1;
static GPtrArray* g_static_private_fallback_array = NULL;
static GPtrArray* g_static_private_destructor_array = NULL;
static GSList* g_static_private_all_threads_data = NULL;

static void
g_static_private_free_data_for_dead_thread(gpointer data)
{
  GPtrArray* array = data;
  guint i;

  if( !array )
    return;
  
  g_lock(gprivate_global);
  for( i = 1; i < array->len; i++ )
    {  
      GDestroyNotify destructor = 
	g_static_private_destructor_array->pdata[ i ];

      if( array->pdata[ i ] && destructor )
	{
	  (*destructor)( array->pdata[ i ]);
	}
    }
  g_static_private_all_threads_data = 
    g_slist_remove(g_static_private_all_threads_data, array);
  g_unlock(gprivate_global);
}

static guint
g_static_private_get_new_id(GDestroyNotify destructor)
{
  guint new_id = g_static_private_next_id++;
  if( g_static_private_destructor_array->len < new_id + 1 )
    {
      g_ptr_array_set_size(g_static_private_destructor_array, new_id + 1);
    }
  g_static_private_destructor_array->pdata[ new_id ] = destructor;
  return new_id;
}

static GPtrArray*
g_static_private_get_array()
{
  GPtrArray* result;
  if( g_private_supported )
    {
      result = g_static_private_get(g_private_key_for_static_private);
    }
  else
    {
      result = g_static_private_fallback_array;
    }
  if( !result )
    {
      result = g_ptr_array_new();
      g_static_private_all_threads_data = 
	g_slist_prepend( g_static_private_all_threads_data, result );
    }
  if( result->len < g_static_private_next_id )    
    {
      g_ptr_array_set_size(result, g_static_private_next_id);
      if( g_private_supported )
	{
	  g_private_set(g_private_key_for_static_private, result);
	}
      else
	{
	  g_static_private_fallback_array = result;
	}
    }
  return result;
}

GStaticPrivate* 
g_static_private_new_for_size(guint size)
{
  GStaticPrivate* result = g_new(GStaticPrivate,1);
  result->constructor = NULL;
  result->destructor = NULL;
  result->size = size;
  g_lock(gprivate_global);
  result->id = g_static_private_get_new_id(g_free);
  g_unlock(gprivate_global);
  return result;
}

GStaticPrivate* 
g_static_private_new_for_type(GNewFunc constructor, 
			      GDestroyNotify destructor)
{ 
  GStaticPrivate* result = g_new(GStaticPrivate,1);
  result->constructor = constructor;
  result->destructor = destructor;
  result->size = 0;
  g_lock(gprivate_global);
  result->id = g_static_private_get_new_id(destructor);
  g_unlock(gprivate_global);
  return result;
}

gpointer 
g_static_private_get( GStaticPrivate* private )
{
  GPtrArray* array;
  gpointer result;
  g_assert( private );
  g_lock(gprivate_global);
  if( private->id == 0)
    { 
      if( private->size != 0 )
	{
	  private->id = g_static_private_get_new_id(g_free);
	}
      else
	{
	  private->id = g_static_private_get_new_id(private->destructor);
	}
    }
  array = g_static_private_get_array();
  if( !array->pdata[ private->id ] )
    {
      if( private->size != 0)
	{
	  array->pdata[ private->id ] = g_malloc0( private->size );
	}
      else if( private->constructor != NULL )
	{
	  array->pdata[ private->id ] = (*private->constructor)();
	}
    }
  result = array->pdata[ private->id ];
  g_unlock(gprivate_global);
  return result;
}

void 
g_static_private_set( GStaticPrivate* private, gpointer value )
{
  GPtrArray* array;
  g_assert( private );
  g_lock(gprivate_global);
  if( private->id == 0)
    {
      if( private->size != 0 )
	{
	  private->id = g_static_private_get_new_id(g_free);
	}
      else
	{
	  private->id = g_static_private_get_new_id(private->destructor);
	}
    }
  array = g_static_private_get_array();
  array->pdata[ private->id ] = value;
  g_unlock(gprivate_global);
}

void g_static_private_free(GStaticPrivate* private)
{
  GSList* foreach = g_static_private_all_threads_data;
  GDestroyNotify destructor;

  g_assert( private );
  g_assert( private->id );

  destructor = g_static_private_destructor_array->pdata[ private->id ] ;
  g_lock(gprivate_global);
  if( destructor )
    {
      while( foreach )
	{
	  gpointer data = ((GPtrArray*)foreach->data)->pdata[ private->id ];
	  if( data )
	    {
	      (*destructor)( data );
	    }
	}
    }
  private->id = 0;
  g_unlock(gprivate_global);
  g_free(private);
}

void
gmutex_c_thread_init()
{
  g_mutex_protect_static_mutex_allocation = g_mutex_new();
  g_private_key_for_static_private = 
    g_private_new(g_static_private_free_data_for_dead_thread);
}

