/*********************************************************************
 * Copyright (c) 2023 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/

package org.eclipse.osee.testscript.internal;

import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.osee.accessor.ArtifactAccessor;
import org.eclipse.osee.accessor.internal.ArtifactAccessorImpl;
import org.eclipse.osee.framework.core.data.ArtifactId;
import org.eclipse.osee.framework.core.data.AttributeTypeId;
import org.eclipse.osee.framework.core.data.BranchId;
import org.eclipse.osee.framework.core.enums.CoreArtifactTypes;
import org.eclipse.osee.framework.core.enums.CoreAttributeTypes;
import org.eclipse.osee.framework.core.enums.CoreRelationTypes;
import org.eclipse.osee.framework.jdk.core.util.SortOrder;
import org.eclipse.osee.framework.jdk.core.util.Strings;
import org.eclipse.osee.orcs.OrcsApi;
import org.eclipse.osee.testscript.ScriptBatchApi;

/**
 * @author Ryan T. Baldwin
 */
public class ScriptBatchApiImpl implements ScriptBatchApi {
   private ArtifactAccessor<ScriptBatchToken> accessor;
   private final List<AttributeTypeId> attributes;
   public ScriptBatchApiImpl(OrcsApi orcsApi) {
      this.setAccessor(new ScriptBatchAccessor(orcsApi));
      attributes = new LinkedList<AttributeTypeId>();
      attributes.add(CoreAttributeTypes.Name);
   }

   private void setAccessor(ArtifactAccessor<ScriptBatchToken> scriptProgramTypeAccessor) {
      this.accessor = scriptProgramTypeAccessor;
   }

   @Override
   public ScriptBatchToken get(BranchId branch, ArtifactId setId) {
      try {
         return this.accessor.get(branch, setId);
      } catch (Exception ex) {
         return new ScriptBatchToken();
      }
   }

   @Override
   public Collection<ScriptBatchToken> getAllFromScriptSet(BranchId branch, ArtifactId setId, ArtifactId viewId,
      String filter, long pageNum, long pageSize, AttributeTypeId orderByAttributeType,
      SortOrder orderByAttributeDirection) {
      try {
         return this.accessor.getAllByRelationAndFilter(branch, CoreRelationTypes.ScriptSetToScriptBatch_ScriptSet,
            setId, filter, Arrays.asList(CoreAttributeTypes.Name), pageNum, pageSize, orderByAttributeType,
            orderByAttributeDirection, viewId);
      } catch (Exception ex) {
         System.out.println(ex);
      }
      return new LinkedList<>();
   }

   @Override
   public int getAllFromScriptSetCount(BranchId branch, ArtifactId setId, ArtifactId viewId, String filter) {
      if (Strings.isValid(filter)) {
         return this.accessor.getAllByRelationAndFilterAndCount(branch,
            CoreRelationTypes.ScriptSetToScriptBatch_ScriptSet, setId, filter, Arrays.asList(CoreAttributeTypes.Name),
            viewId);
      }
      return this.accessor.getAllByRelationAndCount(branch, CoreRelationTypes.ScriptSetToScriptBatch_ScriptSet, setId,
         viewId);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch) {
      return this.getAll(branch, ArtifactId.SENTINEL);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, ArtifactId viewId) {
      return this.getAll(branch, viewId, AttributeTypeId.SENTINEL);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, AttributeTypeId orderByAttribute) {
      return this.getAll(branch, ArtifactId.SENTINEL, orderByAttribute);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, long pageNum, long pageSize,
      AttributeTypeId orderByAttribute) {
      return this.getAll(branch, ArtifactId.SENTINEL, pageNum, pageSize, orderByAttribute);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, ArtifactId viewId, AttributeTypeId orderByAttribute) {
      return this.getAll(branch, viewId, 0L, 0L, orderByAttribute);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, long pageNum, long pageSize) {
      return this.getAll(branch, ArtifactId.SENTINEL, pageNum, pageSize);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, ArtifactId viewId, long pageNum, long pageSize) {
      return this.getAll(branch, viewId, pageNum, pageSize, AttributeTypeId.SENTINEL);
   }

   @Override
   public Collection<ScriptBatchToken> getAll(BranchId branch, ArtifactId viewId, long pageNum, long pageSize,
      AttributeTypeId orderByAttribute) {
      try {
         return this.accessor.getAll(branch, pageNum, pageSize, orderByAttribute);
      } catch (Exception ex) {
         return new LinkedList<ScriptBatchToken>();
      }
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, String filter) {
      return this.getAllByFilter(branch, ArtifactId.SENTINEL, filter);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, ArtifactId viewId, String filter) {
      return this.getAllByFilter(branch, viewId, filter, AttributeTypeId.SENTINEL);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, String filter,
      AttributeTypeId orderByAttribute) {
      return this.getAllByFilter(branch, ArtifactId.SENTINEL, filter, orderByAttribute);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, ArtifactId viewId, String filter,
      AttributeTypeId orderByAttribute) {
      return this.getAllByFilter(branch, viewId, filter, 0L, 0L, orderByAttribute);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, String filter, long pageNum, long pageSize) {
      return this.getAllByFilter(branch, ArtifactId.SENTINEL, filter, pageNum, pageSize);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, ArtifactId viewId, String filter, long pageNum,
      long pageSize) {
      return this.getAllByFilter(branch, viewId, filter, pageNum, pageSize, AttributeTypeId.SENTINEL);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, String filter, long pageNum, long pageSize,
      AttributeTypeId orderByAttribute) {
      return this.getAllByFilter(branch, ArtifactId.SENTINEL, filter, pageNum, pageSize, orderByAttribute);
   }

   @Override
   public Collection<ScriptBatchToken> getAllByFilter(BranchId branch, ArtifactId viewId, String filter, long pageNum,
      long pageSize, AttributeTypeId orderByAttribute) {
      try {
         return this.accessor.getAllByFilter(branch, filter, attributes, pageNum, pageSize, orderByAttribute);
      } catch (Exception ex) {
         return new LinkedList<ScriptBatchToken>();
      }
   }

   @Override
   public int getCountWithFilter(BranchId branch, ArtifactId viewId, String filter) {
      return this.accessor.getAllByFilterAndCount(branch, filter, Arrays.asList(CoreAttributeTypes.Name), viewId);
   }

   private class ScriptBatchAccessor extends ArtifactAccessorImpl<ScriptBatchToken> {

      public ScriptBatchAccessor(OrcsApi orcsApi) {
         super(CoreArtifactTypes.ScriptBatch, orcsApi);
      }

   }

}
