/*********************************************************************
 * Copyright (c) 2021 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/
package org.eclipse.osee.mim.types;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.eclipse.osee.accessor.types.ArtifactAccessorResultWithGammas;
import org.eclipse.osee.accessor.types.AttributePojo;
import org.eclipse.osee.framework.core.data.ApplicabilityId;
import org.eclipse.osee.framework.core.data.ApplicabilityToken;
import org.eclipse.osee.framework.core.data.ArtifactReadable;
import org.eclipse.osee.framework.core.data.ArtifactToken;
import org.eclipse.osee.framework.core.data.AttributeTypeToken;
import org.eclipse.osee.framework.core.data.GammaId;
import org.eclipse.osee.framework.core.enums.CoreArtifactTypes;
import org.eclipse.osee.framework.core.enums.CoreAttributeTypes;
import org.eclipse.osee.framework.core.enums.CoreRelationTypes;
import org.eclipse.osee.framework.jdk.core.type.Id;
import org.eclipse.osee.framework.jdk.core.util.Strings;
import org.eclipse.osee.orcs.rest.model.transaction.Attribute;
import org.eclipse.osee.orcs.rest.model.transaction.CreateArtifact;

/**
 * @author Luciano T. Vaglienti
 */
public class InterfaceStructureToken extends ArtifactAccessorResultWithGammas {
   public static final InterfaceStructureToken SENTINEL = new InterfaceStructureToken();

   private AttributePojo<String> nameAbbrev =
      AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.NameAbbrev, GammaId.SENTINEL, "", "");

   private AttributePojo<String> InterfaceStructureCategory =
      AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceStructureCategory, GammaId.SENTINEL, "", "");

   private AttributePojo<String> InterfaceMinSimultaneity =
      AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceMinSimultaneity, GammaId.SENTINEL, "", "");

   private AttributePojo<String> InterfaceMaxSimultaneity =
      AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceMaxSimultaneity, GammaId.SENTINEL, "", "");

   private AttributePojo<Integer> InterfaceTaskFileType =
      AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceTaskFileType, GammaId.SENTINEL, 0, "");

   private Integer numElements = 0;

   private Double sizeInBytes = 0.0;

   private AttributePojo<String> Description =
      AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.Description, GammaId.SENTINEL, "", "");

   private boolean autogenerated = false;

   private Collection<InterfaceStructureElementToken> elements = new LinkedList<>();

   private ApplicabilityToken applicability;

   public InterfaceStructureToken(ArtifactToken art) {
      this((ArtifactReadable) art);
   }

   public InterfaceStructureToken(ArtifactReadable art) {
      super(art);
      this.setId(art.getId());
      this.setName(AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.Name, "")));
      this.setNameAbbrev(AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.NameAbbrev, "")));
      this.setDescription(AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.Description, "")));
      this.setInterfaceMaxSimultaneity(
         AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.InterfaceMaxSimultaneity, "")));
      this.setInterfaceMinSimultaneity(
         AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.InterfaceMinSimultaneity, "")));
      this.setInterfaceStructureCategory(
         AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.InterfaceStructureCategory, "")));
      this.setInterfaceTaskFileType(
         AttributePojo.valueOf(art.getSoleAttribute(CoreAttributeTypes.InterfaceTaskFileType, 0)));
      this.setElements(
         art.getRelated(CoreRelationTypes.InterfaceStructureContent_DataElement).getList().stream().filter(
            a -> !a.getExistingAttributeTypes().isEmpty()).map(a -> new InterfaceStructureElementToken(a)).collect(
               Collectors.toList()));
      this.setApplicability(
         !art.getApplicabilityToken().getId().equals(-1L) ? art.getApplicabilityToken() : ApplicabilityToken.SENTINEL);
   }

   /**
    * @param id
    * @param name
    */
   public InterfaceStructureToken(Long id, String name) {
      super(id, AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.Name, GammaId.SENTINEL, name, ""));
   }

   /**
    *
    */
   public InterfaceStructureToken() {
      super();
   }

   public AttributePojo<String> getNameAbbrev() {
      return nameAbbrev;
   }

   public void setNameAbbrev(String nameAbbrev) {
      this.nameAbbrev =
         AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.NameAbbrev, GammaId.SENTINEL, nameAbbrev, "");
   }

   @JsonProperty
   public void setNameAbbrev(AttributePojo<String> nameAbbrev) {
      this.nameAbbrev = nameAbbrev;
   }

   /**
    * @return the interfaceStructureCategory
    */
   public AttributePojo<String> getInterfaceStructureCategory() {
      return InterfaceStructureCategory;
   }

   /**
    * @param interfaceStructureCategory the interfaceStructureCategory to set
    */
   public void setInterfaceStructureCategory(String interfaceStructureCategory) {
      InterfaceStructureCategory = AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceStructureCategory,
         GammaId.SENTINEL, interfaceStructureCategory, "");
   }

   @JsonProperty
   public void setInterfaceStructureCategory(AttributePojo<String> interfaceStructureCategory) {
      this.InterfaceStructureCategory = interfaceStructureCategory;
   }

   /**
    * @return the interfaceMinSimultaneity
    */
   public AttributePojo<String> getInterfaceMinSimultaneity() {
      return InterfaceMinSimultaneity;
   }

   /**
    * @param interfaceMinSimultaneity the interfaceMinSimultaneity to set
    */
   public void setInterfaceMinSimultaneity(String interfaceMinSimultaneity) {
      InterfaceMinSimultaneity = AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceMinSimultaneity,
         GammaId.SENTINEL, interfaceMinSimultaneity, "");
   }

   @JsonProperty
   public void setInterfaceMinSimultaneity(AttributePojo<String> interfaceMinSimultaneity) {
      this.InterfaceMinSimultaneity = interfaceMinSimultaneity;
   }

   /**
    * @return the interfaceMaxSimultaneity
    */
   public AttributePojo<String> getInterfaceMaxSimultaneity() {
      return InterfaceMaxSimultaneity;
   }

   /**
    * @param interfaceMaxSimultaneity the interfaceMaxSimultaneity to set
    */
   public void setInterfaceMaxSimultaneity(String interfaceMaxSimultaneity) {
      InterfaceMaxSimultaneity = AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceMaxSimultaneity,
         GammaId.SENTINEL, interfaceMaxSimultaneity, "");
   }

   @JsonProperty
   public void setInterfaceMaxSimultaneity(AttributePojo<String> interfaceMaxSimultaneity) {
      this.InterfaceMaxSimultaneity = interfaceMaxSimultaneity;
   }

   /**
    * @return the interfaceTaskFileType
    */
   public AttributePojo<Integer> getInterfaceTaskFileType() {
      return InterfaceTaskFileType;
   }

   /**
    * @param interfaceTaskFileType the interfaceTaskFileType to set
    */
   public void setInterfaceTaskFileType(Integer interfaceTaskFileType) {
      InterfaceTaskFileType = AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.InterfaceTaskFileType,
         GammaId.SENTINEL, interfaceTaskFileType, "");
   }

   @JsonProperty
   public void setInterfaceTaskFileType(AttributePojo<Integer> interfaceTaskFileType) {
      this.InterfaceTaskFileType = interfaceTaskFileType;
   }

   /**
    * @return the description
    */
   public AttributePojo<String> getDescription() {
      return Description;
   }

   /**
    * @param description the description to set
    */
   public void setDescription(String description) {
      Description =
         AttributePojo.valueOf(Id.SENTINEL, CoreAttributeTypes.Description, GammaId.SENTINEL, description, "");
   }

   @JsonProperty
   public void setDescription(AttributePojo<String> description) {
      this.Description = description;
   }

   /**
    * @return the elements
    */
   public List<InterfaceStructureElementToken> getElements() {
      return (List<InterfaceStructureElementToken>) elements;
   }

   /**
    * @param elements the elements to set
    */
   public void setElements(Collection<InterfaceStructureElementToken> elements) {
      this.elements = elements;
      setNumElements(elements.stream().filter(e -> e.isIncludedInCounts()).collect(Collectors.toList()).size());
      setSizeInBytes(elements.stream().filter(e -> e.isIncludedInCounts()).collect(
         Collectors.summingDouble(InterfaceStructureElementToken::getElementSizeInBytes)));
   }

   /**
    * @return the numElements
    */
   public Integer getNumElements() {
      return this.numElements;
   }

   public void setNumElements(Integer numElements) {
      this.numElements = numElements;
   }

   /**
    * @return the sizeInBytes
    */
   public Double getSizeInBytes() {
      return this.sizeInBytes;
   }

   public void setSizeInBytes(Double sizeInBytes) {
      this.sizeInBytes = sizeInBytes;
   }

   public boolean getIncorrectlySized() {
      return this.getSizeInBytes() % 8 != 0;
   }

   /**
    * @return the bytesPerSecondMaximum
    */
   public Double getBytesPerSecondMaximum() {
      return this.getSizeInBytes() * (Strings.isNumeric(
         this.getInterfaceMaxSimultaneity().getValue()) ? Double.parseDouble(
            this.getInterfaceMaxSimultaneity().getValue()) : 0.0);
   }

   /**
    * @return the bytesPerSecondMinimum
    */
   public Double getBytesPerSecondMinimum() {
      return this.getSizeInBytes() * (Strings.isNumeric(
         this.getInterfaceMinSimultaneity().getValue()) ? Double.parseDouble(
            this.getInterfaceMinSimultaneity().getValue()) : 0.0);
   }

   /**
    * @return the applicability
    */
   public ApplicabilityToken getApplicability() {
      return applicability;
   }

   /**
    * @param applicability the applicability to set
    */
   public void setApplicability(ApplicabilityToken applicability) {
      this.applicability = applicability;
   }

   /**
    * @return the autogenerated
    */
   public boolean isAutogenerated() {
      return autogenerated;
   }

   /**
    * @param autogenerated the autogenerated to set
    */
   public void setAutogenerated(boolean autogenerated) {
      this.autogenerated = autogenerated;
   }

   public CreateArtifact createArtifact(String key, ApplicabilityId applicId) {
      // @formatter:off
      Map<AttributeTypeToken, String> values = new HashMap<>();
      values.put(CoreAttributeTypes.Description, this.getDescription().getValue());
      values.put(CoreAttributeTypes.NameAbbrev, this.getNameAbbrev().getValue());
      values.put(CoreAttributeTypes.InterfaceStructureCategory, this.getInterfaceStructureCategory().getValue());
      values.put(CoreAttributeTypes.InterfaceMinSimultaneity, this.getInterfaceMinSimultaneity().getValue());
      values.put(CoreAttributeTypes.InterfaceMaxSimultaneity, this.getInterfaceMaxSimultaneity().getValue());
      values.put(CoreAttributeTypes.InterfaceTaskFileType, Integer.toString(this.getInterfaceTaskFileType().getValue()));
      // @formatter:on

      CreateArtifact art = new CreateArtifact();
      art.setName(this.getName().getValue());
      art.setTypeId(CoreArtifactTypes.InterfaceStructure.getIdString());

      List<Attribute> attrs = new LinkedList<>();

      for (AttributeTypeToken type : CoreArtifactTypes.InterfaceStructure.getValidAttributeTypes()) {
         String value = values.get(type);
         if (Strings.isInValid(value)) {
            continue;
         }
         Attribute attr = new Attribute(type.getIdString());
         attr.setValue(Arrays.asList(value));
         attrs.add(attr);
      }

      art.setAttributes(attrs);
      art.setApplicabilityId(applicId.getIdString());
      art.setkey(key);
      return art;
   }

}
