/*********************************************************************
 * Copyright (c) 2004, 2007 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/

package org.eclipse.osee.orcs.core.internal.search;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.osee.framework.core.OrcsTokenService;
import org.eclipse.osee.framework.core.data.ApplicabilityId;
import org.eclipse.osee.framework.core.data.ArtifactId;
import org.eclipse.osee.framework.core.data.BranchId;
import org.eclipse.osee.framework.core.sql.OseeSql;
import org.eclipse.osee.orcs.OrcsApi;
import org.eclipse.osee.orcs.core.ds.ApplicabilityDsQuery;
import org.eclipse.osee.orcs.core.ds.QueryData;
import org.eclipse.osee.orcs.core.ds.QueryEngine;
import org.eclipse.osee.orcs.search.ApplicabilityQuery;
import org.eclipse.osee.orcs.search.BranchQuery;
import org.eclipse.osee.orcs.search.QueryBuilder;
import org.eclipse.osee.orcs.search.QueryFactory;
import org.eclipse.osee.orcs.search.TransactionQuery;
import org.eclipse.osee.orcs.search.TupleQuery;

/**
 * @author Roberto E. Escobar
 */
public class QueryFactoryImpl implements QueryFactory {

   private final CallableQueryFactory artQueryFactory;
   private final BranchCriteriaFactory branchCriteriaFactory;
   private final TransactionCriteriaFactory txCriteriaFactory;
   private final TupleQuery tupleQuery;
   private final ApplicabilityDsQuery applicabilityDsQuery;
   private final QueryEngine queryEngine;
   private final OrcsTokenService tokenService;
   private final OrcsApi orcsApi;

   public QueryFactoryImpl(OrcsApi orcsApi, CallableQueryFactory artQueryFactory, BranchCriteriaFactory branchCriteriaFactory, TransactionCriteriaFactory txCriteriaFactory, TupleQuery tupleQuery, ApplicabilityDsQuery applicabilityDsQuery, QueryEngine queryEngine, OrcsTokenService tokenService) {
      this.orcsApi = orcsApi;
      this.artQueryFactory = artQueryFactory;
      this.branchCriteriaFactory = branchCriteriaFactory;
      this.txCriteriaFactory = txCriteriaFactory;
      this.tupleQuery = tupleQuery;
      this.applicabilityDsQuery = applicabilityDsQuery;
      this.queryEngine = queryEngine;
      this.tokenService = tokenService;
   }

   @Override
   public BranchQuery branchQuery() {
      return new BranchQueryImpl(queryEngine, branchCriteriaFactory,
         new QueryData(this, queryEngine, artQueryFactory, tokenService));
   }

   @Override
   public QueryBuilder fromBranch(BranchId branch) {
      return new QueryData(this, queryEngine, artQueryFactory, tokenService, tokenService.getBranch(branch));
   }

   @Override
   public QueryBuilder fromBranch(BranchId branch, ArtifactId view) {
      return new QueryData(this, queryEngine, artQueryFactory, tokenService, tokenService.getBranch(branch), view);
   }

   @Override
   public QueryBuilder fromBranch(BranchId branch, ApplicabilityId appId) {
      return new QueryData(this, queryEngine, artQueryFactory, tokenService, tokenService.getBranch(branch), appId);
   }

   @Override
   public TransactionQuery transactionQuery() {
      return new TransactionQueryImpl(queryEngine, txCriteriaFactory,
         new QueryData(this, queryEngine, artQueryFactory, tokenService), orcsApi.userService());
   }

   @Override
   public TupleQuery tupleQuery() {
      return tupleQuery;
   }

   @Override
   public ApplicabilityQuery applicabilityQuery() {
      return new ApplicabilityQueryImpl(applicabilityDsQuery, this, orcsApi);
   }

   @Override
   public boolean artIdExists(ArtifactId artifactId) {
      int artifactCount =
         orcsApi.getJdbcService().getClient().fetch(0, OseeSql.ARTIFACT_ID_COUNT.getSql(), artifactId.getId());
      return artifactCount > 0;
   }

   @Override
   public Collection<BranchId> getBranches(ArtifactId sourceArtifact) {
      List<BranchId> ids = new LinkedList<>();
      orcsApi.getJdbcService().getClient().runQuery(stmt -> ids.add(BranchId.valueOf(stmt.getLong("branch_id"))),
         OseeSql.ARTIFACT_ID_BRANCHES.getSql(), sourceArtifact.getIdString());
      return ids;
   }
}