/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Injector;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternModel;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.testing.core.MatchSetRecordDiff;
import org.eclipse.viatra.query.testing.core.ModelLoadHelper;
import org.eclipse.viatra.query.testing.core.SnapshotHelper;
import org.eclipse.viatra.query.testing.core.TestingSeverityAggregatorLogAppender;
import org.eclipse.viatra.query.testing.core.api.MatchRecordEquivalence;
import org.eclipse.viatra.query.testing.core.internal.DefaultMatchRecordEquivalence;
import org.eclipse.viatra.query.testing.snapshot.MatchRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSetRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSubstitutionRecord;
import org.eclipse.viatra.query.testing.snapshot.QuerySnapshot;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.junit.Assert;

/**
 * Primitive methods for executing a functional test for VIATRA Queries.
 */
@SuppressWarnings("all")
public class TestExecutor {
  public final static String CORRECTRESULTS = "Correct result set";
  
  public final static String CORRECT_SINGLE = "Correct single match for parameterless pattern";
  
  public final static String CORRECT_EMPTY = "Correct empty match set";
  
  public final static String UNEXPECTED_MATCH = "Unexpected match";
  
  public final static String EXPECTED_NOT_FOUND = "Expected match not found";
  
  public final static String MULTIPLE_FOR_EXPECTED = "Multiple matches found for expected match";
  
  public final static String MATCHSETRECORD_NOT_IN_SNAPSHOT = "Expected match set record is not part of snapshot";
  
  public final static String PATTERNNAME_DIFFERENT = "Expected pattern qualified name different from actual";
  
  @Inject
  @Extension
  private ModelLoadHelper _modelLoadHelper;
  
  @Inject
  @Extension
  private SnapshotHelper _snapshotHelper;
  
  @Inject
  private Injector injector;
  
  @Inject
  private Logger logger;
  
  /**
   * Checks the pattern name of the matcher against the one stored in
   *  the record and checks parameterless patterns as well.
   * 
   * Returns true if further comparison is allowed, false otherwise.
   */
  public boolean validateMatcherBeforeCompare(final ViatraQueryMatcher<?> matcher, final MatchSetRecord expected, final Set<String> diff) {
    boolean _equals = matcher.getPatternName().equals(expected.getPatternQualifiedName());
    boolean _not = (!_equals);
    if (_not) {
      String _patternQualifiedName = expected.getPatternQualifiedName();
      String _plus = ((TestExecutor.PATTERNNAME_DIFFERENT + " (") + _patternQualifiedName);
      String _plus_1 = (_plus + "!=");
      String _patternName = matcher.getPatternName();
      String _plus_2 = (_plus_1 + _patternName);
      String _plus_3 = (_plus_2 + ")");
      diff.add(_plus_3);
      return false;
    }
    int _size = matcher.getParameterNames().size();
    boolean _equals_1 = (_size == 0);
    if (_equals_1) {
      int _size_1 = expected.getMatches().size();
      boolean _equals_2 = (_size_1 == 1);
      if (_equals_2) {
        int _countMatches = matcher.countMatches();
        boolean _equals_3 = (_countMatches == 1);
        if (_equals_3) {
          diff.add(TestExecutor.CORRECT_SINGLE);
          return true;
        } else {
          int _countMatches_1 = matcher.countMatches();
          boolean _equals_4 = (_countMatches_1 == 0);
          if (_equals_4) {
            diff.add(TestExecutor.CORRECT_EMPTY);
            return true;
          }
        }
      }
    }
    return true;
  }
  
  /**
   * Compares the match set of a given matcher with the given match record
   *  using VIATRA Query as a compare tool.
   * Therefore the comparison depends on correct VIATRA Query query evaluation
   *  (for a given limited pattern language feature set).
   */
  public <MATCH extends IPatternMatch> HashSet<String> compareResultSetsAsRecords(final ViatraQueryMatcher<MATCH> matcher, final MatchSetRecord expected) {
    DefaultMatchRecordEquivalence _defaultMatchRecordEquivalence = new DefaultMatchRecordEquivalence();
    return this.<MATCH>compareResultSetsAsRecords(matcher, expected, _defaultMatchRecordEquivalence);
  }
  
  /**
   * Compares the match set of a given matcher with the given match record
   * using VIATRA Query as a compare tool.
   * 
   * The comparison logic can be specified via the equivalence parameter.
   * 
   * @since 1.6
   */
  public <MATCH extends IPatternMatch> HashSet<String> compareResultSetsAsRecords(final ViatraQueryMatcher<MATCH> matcher, final MatchSetRecord expected, final MatchRecordEquivalence equivalence) {
    final HashSet<String> diff = CollectionLiterals.<String>newHashSet();
    boolean correctResults = this.validateMatcherBeforeCompare(matcher, expected, diff);
    if ((!correctResults)) {
      return diff;
    }
    EObject _eContainer = expected.eContainer();
    boolean _not = (!(_eContainer instanceof QuerySnapshot));
    if (_not) {
      diff.add(TestExecutor.MATCHSETRECORD_NOT_IN_SNAPSHOT);
      return diff;
    }
    EObject _eContainer_1 = expected.eContainer();
    final QuerySnapshot snapshot = ((QuerySnapshot) _eContainer_1);
    final MATCH partialMatch = this._snapshotHelper.<MATCH>createMatchForMatchRecord(matcher.getSpecification(), expected.getFilter());
    final MatchSetRecord actual = this._snapshotHelper.<MATCH>saveMatchesToSnapshot(matcher, partialMatch, snapshot);
    final MatchSetRecordDiff matchdiff = MatchSetRecordDiff.compute(expected, actual, equivalence);
    final Function1<MatchRecord, String> _function = (MatchRecord it) -> {
      String _prettyPrint = this._snapshotHelper.prettyPrint(it);
      String _plus = ((TestExecutor.UNEXPECTED_MATCH + " (") + _prettyPrint);
      return (_plus + ")");
    };
    Iterables.<String>addAll(diff, IterableExtensions.<MatchRecord, String>map(matchdiff.getAdditions(), _function));
    final Function1<MatchRecord, String> _function_1 = (MatchRecord it) -> {
      String _prettyPrint = this._snapshotHelper.prettyPrint(it);
      String _plus = ((TestExecutor.EXPECTED_NOT_FOUND + " (") + _prettyPrint);
      return (_plus + ")");
    };
    Iterables.<String>addAll(diff, IterableExtensions.<MatchRecord, String>map(matchdiff.getRemovals(), _function_1));
    return diff;
  }
  
  /**
   * Compares the match set of a given matcher with the given match record using the
   *  records as partial matches on the matcher.
   * Therefore the comparison does not depend on correct VIATRA Query query evaluation.
   */
  public <MATCH extends IPatternMatch> HashSet<String> compareResultSets(final ViatraQueryMatcher<MATCH> matcher, final MatchSetRecord expected) {
    final HashSet<String> diff = CollectionLiterals.<String>newHashSet();
    boolean correctResults = this.validateMatcherBeforeCompare(matcher, expected, diff);
    if ((!correctResults)) {
      return diff;
    }
    final ArrayList<MATCH> foundMatches = CollectionLiterals.<MATCH>newArrayList();
    EList<MatchRecord> _matches = expected.getMatches();
    for (final MatchRecord matchRecord : _matches) {
      {
        final MATCH partialMatch = this._snapshotHelper.<MATCH>createMatchForMatchRecord(matcher.getSpecification(), matchRecord);
        final int numMatches = matcher.countMatches(partialMatch);
        if ((numMatches == 0)) {
          StringBuilder _printMatchRecord = this.printMatchRecord(matchRecord);
          String _plus = ((TestExecutor.EXPECTED_NOT_FOUND + " (") + _printMatchRecord);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
          correctResults = false;
        } else {
          if ((numMatches == 1)) {
            foundMatches.add(partialMatch);
          } else {
            StringBuilder _printMatchRecord_1 = this.printMatchRecord(matchRecord);
            String _plus_2 = ((TestExecutor.MULTIPLE_FOR_EXPECTED + " (") + _printMatchRecord_1);
            String _plus_3 = (_plus_2 + ")");
            diff.add(_plus_3);
            correctResults = false;
          }
        }
      }
    }
    final Consumer<MATCH> _function = (MATCH it) -> {
      boolean _contains = foundMatches.contains(it);
      boolean _not = (!_contains);
      if (_not) {
        String _prettyPrint = it.prettyPrint();
        String _plus = ((TestExecutor.UNEXPECTED_MATCH + " (") + _prettyPrint);
        String _plus_1 = (_plus + ")");
        diff.add(_plus_1);
      }
    };
    matcher.forEachMatch(this._snapshotHelper.<MATCH>createMatchForMatchRecord(matcher.getSpecification(), expected.getFilter()), _function);
    return diff;
  }
  
  public StringBuilder printMatchRecord(final MatchRecord record) {
    StringBuilder _xblockexpression = null;
    {
      final StringBuilder sb = new StringBuilder();
      EObject _eContainer = record.eContainer();
      final MatchSetRecord matchSet = ((MatchSetRecord) _eContainer);
      final Consumer<MatchSubstitutionRecord> _function = (MatchSubstitutionRecord it) -> {
        int _length = sb.length();
        boolean _greaterThan = (_length > 0);
        if (_greaterThan) {
          sb.append(",");
        }
        sb.append(it.getParameterName()).append("=").append(this._snapshotHelper.derivedValue(it));
      };
      record.getSubstitutions().forEach(_function);
      String _patternQualifiedName = matchSet.getPatternQualifiedName();
      String _plus = (_patternQualifiedName + "(");
      sb.insert(0, _plus);
      _xblockexpression = sb.append(")");
    }
    return _xblockexpression;
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model
   *  based on the input specification of the snapshot.
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final PatternModel patternModel, final QuerySnapshot snapshot) {
    final HashSet<Object> diff = CollectionLiterals.<Object>newHashSet();
    final Notifier input = this._snapshotHelper.getEMFRootForSnapshot(snapshot);
    EMFScope _eMFScope = new EMFScope(input);
    final ViatraQueryEngine engine = ViatraQueryEngine.on(_eMFScope);
    this.registerLogger(engine);
    final Consumer<MatchSetRecord> _function = (MatchSetRecord matchSet) -> {
      final ViatraQueryMatcher<? extends IPatternMatch> matcher = this._modelLoadHelper.initializeMatcherFromModel(patternModel, engine, matchSet.getPatternQualifiedName());
      if ((matcher != null)) {
        final HashSet<String> result = this.compareResultSets(matcher, matchSet);
        boolean _not = (!(((result == null) || CollectionLiterals.<String>newHashSet(TestExecutor.CORRECT_EMPTY).equals(result)) || CollectionLiterals.<String>newHashSet(TestExecutor.CORRECT_SINGLE).equals(result)));
        if (_not) {
          diff.addAll(result);
        }
      }
    };
    snapshot.getMatchSetRecords().forEach(_function);
    Assert.assertTrue(this.logDifference(diff, engine), diff.isEmpty());
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model
   *  based on the input specification of the snapshot (specified as a platform URI).
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final PatternModel patternModel, final String snapshotUri) {
    final QuerySnapshot snapshot = this._modelLoadHelper.loadExpectedResultsFromUri(snapshotUri);
    this.assertMatchResults(patternModel, snapshot);
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model (specified as a platform URI)
   *  based on the input specification of the snapshot (specified as a platform URI).
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final String patternUri, final String snapshotUri) {
    final PatternModel patternModel = this._modelLoadHelper.loadPatternModelFromUri(patternUri, this.injector);
    this.assertMatchResults(patternModel, snapshotUri);
  }
  
  public void registerLogger(final ViatraQueryEngine engine) {
    ConsoleAppender _consoleAppender = new ConsoleAppender();
    this.logger.addAppender(_consoleAppender);
  }
  
  public String retrieveLoggerOutput(final ViatraQueryEngine engine) {
    final Logger logger = this.logger;
    final Enumeration appers = logger.getAllAppenders();
    while (appers.hasMoreElements()) {
      {
        final Object apper = appers.nextElement();
        if ((apper instanceof TestingSeverityAggregatorLogAppender)) {
          return ((TestingSeverityAggregatorLogAppender)apper).getOutput();
        }
      }
    }
    return "Logger output not recorded";
  }
  
  public String logDifference(final Set<?> diff) {
    String _xblockexpression = null;
    {
      final StringBuilder stringBuilder = new StringBuilder();
      this.logDifference(diff, stringBuilder);
      _xblockexpression = stringBuilder.toString();
    }
    return _xblockexpression;
  }
  
  public String logDifference(final Set<Object> diff, final ViatraQueryEngine engine) {
    String _xblockexpression = null;
    {
      final StringBuilder stringBuilder = new StringBuilder();
      this.logDifference(diff, stringBuilder);
      stringBuilder.append(this.retrieveLoggerOutput(engine));
      _xblockexpression = stringBuilder.toString();
    }
    return _xblockexpression;
  }
  
  private void logDifference(final Set<?> diff, final StringBuilder stringBuilder) {
    final Consumer<Object> _function = (Object it) -> {
      stringBuilder.append(("\n" + it));
    };
    diff.forEach(_function);
  }
}
