// Copyright 2025 Ben Manes. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.github.benmanes.caffeine.cache;

import com.github.benmanes.caffeine.cache.References.LookupKeyReference;
import com.github.benmanes.caffeine.cache.References.SoftValueReference;
import com.github.benmanes.caffeine.cache.References.WeakKeyReference;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import org.jspecify.annotations.Nullable;

/**
 * <em>WARNING: GENERATED CODE</em>
 *
 * <p>A cache entry that provides the following features:
 *
 * <ul>
 *   <li>WeakKeys
 *   <li>SoftValues
 * </ul>
 *
 * @author ben.manes@gmail.com (Ben Manes)
 */
@SuppressWarnings({"PMD.UnusedFormalParameter", "unchecked"})
class FD<K, V> extends Node<K, V> implements NodeFactory<K, V> {
  protected static final VarHandle VALUE;

  static {
    MethodHandles.Lookup lookup = MethodHandles.lookup();
    try {
      VALUE = lookup.findVarHandle(FD.class, NodeFactory.VALUE, SoftValueReference.class);
    } catch (ReflectiveOperationException e) {
      throw new ExceptionInInitializerError(e);
    }
  }

  volatile SoftValueReference<V> value;

  FD() {}

  FD(
      K key,
      ReferenceQueue<K> keyReferenceQueue,
      V value,
      ReferenceQueue<V> valueReferenceQueue,
      int weight,
      long now) {
    this(new WeakKeyReference<K>(key, keyReferenceQueue), value, valueReferenceQueue, weight, now);
  }

  FD(Object keyReference, V value, ReferenceQueue<V> valueReferenceQueue, int weight, long now) {
    VALUE.set(this, new SoftValueReference<V>(keyReference, value, valueReferenceQueue));
  }

  public final Object getKeyReference() {
    SoftValueReference<V> valueRef = (SoftValueReference<V>) VALUE.getAcquire(this);
    return valueRef.getKeyReference();
  }

  public final @Nullable Object getKeyReferenceOrNull() {
    SoftValueReference<V> valueRef = (SoftValueReference<V>) VALUE.getAcquire(this);
    Reference<K> keyRef = (Reference<K>) valueRef.getKeyReference();
    return (keyRef.get() == null) ? null : keyRef;
  }

  public final K getKey() {
    SoftValueReference<V> valueRef = (SoftValueReference<V>) VALUE.getAcquire(this);
    Reference<K> keyRef = (Reference<K>) valueRef.getKeyReference();
    return keyRef.get();
  }

  public final V getValue() {
    // The referent may be observed as null if it is cleared after a write but before it is read.
    // This can occur due to garbage collection or proactive clearing to assist the GC when the
    // reference and its referent reside in different memory regions (e.g., across generations). To
    // preserve linearizability, the loop validates that any observed null was not the result of a
    // stale reference by confirming it matches the current reference.
    Reference<V> ref = (Reference<V>) VALUE.getAcquire(this);
    for (; ; ) {
      V referent = ref.get();
      if (referent != null) {
        return referent;
      }
      VarHandle.loadLoadFence();
      Reference<V> current = (Reference<V>) VALUE.getAcquire(this);
      if (ref == current) {
        return null;
      }
      ref = current;
    }
  }

  public final void setValue(V value, ReferenceQueue<V> referenceQueue) {
    Reference<V> ref = (Reference<V>) VALUE.getAcquire(this);
    VALUE.setRelease(this, new SoftValueReference<V>(getKeyReference(), value, referenceQueue));
    VarHandle.storeStoreFence();
    ref.clear();
  }

  public final Object getValueReference() {
    return VALUE.getAcquire(this);
  }

  public final boolean containsValue(Object value) {
    return getValue() == value;
  }

  public Node<K, V> newNode(
      K key,
      ReferenceQueue<K> keyReferenceQueue,
      V value,
      ReferenceQueue<V> valueReferenceQueue,
      int weight,
      long now) {
    return new FD<>(key, keyReferenceQueue, value, valueReferenceQueue, weight, now);
  }

  public Node<K, V> newNode(
      Object keyReference, V value, ReferenceQueue<V> valueReferenceQueue, int weight, long now) {
    return new FD<>(keyReference, value, valueReferenceQueue, weight, now);
  }

  public Object newLookupKey(Object key) {
    return new LookupKeyReference<>(key);
  }

  public Object newReferenceKey(K key, ReferenceQueue<K> referenceQueue) {
    return new WeakKeyReference<K>(key, referenceQueue);
  }

  public boolean softValues() {
    return true;
  }

  public final boolean isAlive() {
    Object key = getKeyReference();
    return (key != RETIRED_WEAK_KEY) && (key != DEAD_WEAK_KEY);
  }

  public final boolean isRetired() {
    return (getKeyReference() == RETIRED_WEAK_KEY);
  }

  public final void retire() {
    SoftValueReference<V> valueRef = (SoftValueReference<V>) VALUE.getOpaque(this);
    Reference<K> keyRef = (Reference<K>) valueRef.getKeyReference();
    keyRef.clear();
    valueRef.setKeyReference(RETIRED_WEAK_KEY);
    valueRef.clear();
  }

  public final boolean isDead() {
    return (getKeyReference() == DEAD_WEAK_KEY);
  }

  public final void die() {
    SoftValueReference<V> valueRef = (SoftValueReference<V>) VALUE.getOpaque(this);
    Reference<K> keyRef = (Reference<K>) valueRef.getKeyReference();
    keyRef.clear();
    valueRef.setKeyReference(DEAD_WEAK_KEY);
    valueRef.clear();
  }
}
