/*****************************************************************************
 * Copyright (c) 2019 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *
 *****************************************************************************/

package org.eclipse.papyrus.cdo.security.internal.ui;

import java.util.Collection;
import java.util.Collections;

import org.eclipse.emf.cdo.security.User;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.papyrus.infra.properties.ui.runtime.DisplayEngine;
import org.eclipse.papyrus.infra.properties.ui.runtime.PropertiesRuntime;
import org.eclipse.papyrus.infra.properties.ui.util.PropertiesDisplayHelper;
import org.eclipse.papyrus.infra.properties.ui.widgets.layout.PropertiesLayout;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.SelectionDialog;

/**
 * This allows to define the dialog to edit CDO user.
 */
public class CDOUserDialog extends SelectionDialog {

	/** The properties composite for the package. */
	private Composite propertiesComposite;

	/** The display engine. */
	private DisplayEngine displayEngine;

	/** The selected requirements. */
	private Collection<User> users;

	/**
	 * Constructor.
	 *
	 * @param parentShell
	 *            The parent shell.
	 * @param user
	 *            The selected user to manage.
	 */
	public CDOUserDialog(final Shell parentShell, final User user) {
		this(parentShell, Collections.singletonList(user));
	}

	/**
	 * Constructor.
	 *
	 * @param parentShell
	 *            The parent shell.
	 * @param users
	 *            The selected users to manage.
	 */
	public CDOUserDialog(final Shell parentShell, final Collection<User> users) {
		super(parentShell);
		this.users = users;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see org.eclipse.jface.dialogs.Dialog#create()
	 */
	@Override
	public void create() {
		super.create();
		if (getShell().getText() == null || getShell().getText().isEmpty()) {
			getShell().setText("Edit User"); //$NON-NLS-1$
		}
		getShell().setImage(org.eclipse.papyrus.infra.widgets.Activator.getDefault().getImage("/icons/papyrus.png")); //$NON-NLS-1$
		getShell().addDisposeListener(new DisposeListener() {

			@Override
			public void widgetDisposed(DisposeEvent e) {
				dispose();
			}
		});

		// Create the parent composite
		final Composite parent = new Composite(getDialogArea(), SWT.NONE);
		parent.setLayout(new PropertiesLayout());
		parent.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		// Create properties view
		createPropertiesGroup(parent);

		// display the package into the properties composite
		if (users.size() == 1) {
			displayElement(users.iterator().next());
		} else {
			displayElement(users);
		}

		getShell().pack();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see org.eclipse.jface.dialogs.Dialog#getDialogArea()
	 */
	@Override
	public Composite getDialogArea() {
		return (Composite) super.getDialogArea();
	}

	/**
	 * Create the properties group.
	 *
	 * @param parent
	 *            The parent composite.
	 */
	protected void createPropertiesGroup(final Composite parent) {
		propertiesComposite = new Composite(parent, SWT.NONE);
		final PropertiesLayout layout = new PropertiesLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		propertiesComposite.setLayout(layout);
		final GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.minimumHeight = 500;
		data.widthHint = 1000;
		propertiesComposite.setLayoutData(data);
	}

	/**
	 * This allows to display element in the properties composite.
	 *
	 * @param object
	 *            The object to display.
	 */
	protected void displayElement(final Object object) {
		if (null != displayEngine) {
			displayEngine.removeSection(propertiesComposite);
		}
		if (object instanceof Collection) {
			final TreePath[] treePaths = new TreePath[((Collection<?>) object).size()];
			int currentIt = 0;
			for (Object obj : ((Collection<?>) object)) {
				final Object[] objArray = new Object[1];
				objArray[0] = obj;
				treePaths[currentIt++] = new TreePath(objArray);
			}
			TreeSelection treeSelection = new TreeSelection(treePaths);
			displayEngine = PropertiesDisplayHelper.display(PropertiesRuntime.getConstraintEngine(), treeSelection, propertiesComposite);
		} else {
			displayEngine = PropertiesDisplayHelper.display(object, propertiesComposite);
		}

		propertiesComposite.layout();
		propertiesComposite.getParent().layout();
		propertiesComposite.getParent().pack();
		getShell().layout();
	}

	/**
	 * Disposes this dialog.
	 */
	public void dispose() {
		if (null != displayEngine) {
			displayEngine.dispose();
		}
		displayEngine = null;
		propertiesComposite = null;
	}

}
