/*****************************************************************************
 * Copyright (c) 2017 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Nicolas FAUVERGUE (CEA LIST) nicolas.fauvergue@cea.fr - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrus.interoperability.sysml14.sysml.blackboxes;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.eclipse.m2m.qvt.oml.blackbox.java.Operation;
import org.eclipse.m2m.qvt.oml.blackbox.java.Operation.Kind;
import org.eclipse.papyrus.interoperability.sysml14.sysml.Activator;
import org.eclipse.papyrus.interoperability.sysml14.sysml.utils.SysMLMigrationConstantsUtils;
import org.eclipse.uml2.uml.Model;

/**
 * This allows to manage the versioning of transformations.
 */
public class VersioningHelper {

	/**
	 * Get the EAnnotation name.
	 * 
	 * @return The EAnnotation name.
	 */
	@Operation(kind = Kind.HELPER)
	public String getEAnnotationSourceNameForVersioning() {
		return SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_SOURCE;
	}

	/**
	 * Get the papyrus model name from a model.
	 * 
	 * @param model
	 *            The model.
	 * @return The papyrus model name.
	 */
	@Operation(kind = Kind.HELPER)
	public String getPapyrusResourceName(final Model model) {
		String result = model.getName();
		if (null != model.eResource()) {
			result = model.eResource().getURI().trimFileExtension().lastSegment();
		}
		return result;
	}

	/**
	 * Get the key of the EAnnotation for the source project.
	 * 
	 * @return The key of the EAnnotation for the source project.
	 */
	@Operation(kind = Kind.HELPER)
	public String getKeyForSourceModelName() {
		return SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_SOURCE_PROJECT_NAME;
	}

	/**
	 * Get the key of the EAnnotation for the migration date.
	 * 
	 * @return The key of the EAnnotation for the migration date.
	 */
	@Operation(kind = Kind.HELPER)
	public String getKeyForPapyrusMigrationDate() {
		return SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_MIGRATION_DATE;
	}

	/**
	 * Get the current date.
	 * 
	 * @return The current date with format 'yyyyMMddHHmmss'.
	 */
	@Operation(kind = Kind.HELPER)
	public String getPapyrusMigrationDate() {
		final DateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
		final Date date = new Date();
		return dateFormat.format(date);
	}


	/**
	 * Get the key of the EAnnotation for the bundle version.
	 * 
	 * @return The key of the EAnnotation for the bundle version.
	 */
	@Operation(kind = Kind.HELPER)
	public String getKeyForPapyrusMigrationBundleVersion() {
		return SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_MIGRATION_BUNDLE_VERSION;
	}

	/**
	 * Get the version of the SysML migration bundle.
	 * 
	 * @return The version of the SysML migration bundle.
	 */
	@Operation(kind = Kind.HELPER)
	public String getImportBundleSysMLMigrationVersion() {
		return Activator.getDefault().getBundle().getVersion().toString();
	}


}
