/*
 * Copyright (c) 2023 Eike Stepper (Loehne, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 */
package org.eclipse.emf.cdo.lm.ui.dialogs;

import org.eclipse.emf.cdo.eresource.CDOResource;
import org.eclipse.emf.cdo.eresource.CDOResourceFolder;
import org.eclipse.emf.cdo.eresource.CDOResourceNode;
import org.eclipse.emf.cdo.lm.assembly.Assembly;
import org.eclipse.emf.cdo.lm.assembly.AssemblyModule;
import org.eclipse.emf.cdo.lm.internal.client.LMResourceSetConfiguration;
import org.eclipse.emf.cdo.lm.ui.bundle.OM;
import org.eclipse.emf.cdo.ui.AbstractResourceSelectionDialog;
import org.eclipse.emf.cdo.util.CDOUtil;
import org.eclipse.emf.cdo.view.CDOView;

import org.eclipse.swt.widgets.Shell;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * @author Eike Stepper
 */
public class SelectModuleResourcesDialog extends AbstractResourceSelectionDialog<LMResourceSetConfiguration>
{
  private final LMResourceSetConfiguration configuration;

  private final Map<AssemblyModule, CDOView> views = new LinkedHashMap<>();

  public SelectModuleResourcesDialog(Shell shell, boolean multi, LMResourceSetConfiguration configuration)
  {
    super(shell, multi, "Select Module Resources", "Select resources from the current module and its dependencies.",
        OM.Activator.INSTANCE.loadImageDescriptor("icons/NewModule.png"));
    this.configuration = configuration;

    if (configuration != null)
    {
      Assembly assembly = configuration.getAssembly();
      AssemblyModule rootModule = assembly.getRootModule();
      views.put(rootModule, CDOUtil.getViewSet(configuration.getResourceSet()).getViews()[0]);
      assembly.forEachDependency(module -> views.put(module, configuration.getView(module)));
    }
  }

  @Override
  protected LMResourceSetConfiguration getInput()
  {
    return configuration;
  }

  @Override
  protected boolean elementHasChildren(Object object, Predicate<Object> defaultHasChildren)
  {
    if (object == views)
    {
      return !views.isEmpty();
    }

    return super.elementHasChildren(object, defaultHasChildren);
  }

  @Override
  protected Object[] elementGetChildren(Object object, Function<Object, Object[]> defaultGetChildren)
  {
    if (object == configuration)
    {
      // Return the assembly modules.
      return views.keySet().toArray();
    }

    if (object instanceof AssemblyModule)
    {
      AssemblyModule module = (AssemblyModule)object;
      CDOView view = views.get(module);
      CDOResource rootResource = view.getRootResource();
      return defaultGetChildren.apply(rootResource);
    }

    return super.elementGetChildren(object, defaultGetChildren);
  }

  @Override
  protected Object elementGetParent(Object object, Function<Object, Object> defaultGetParent)
  {
    if (object instanceof CDOResourceNode)
    {
      CDOResourceNode node = (CDOResourceNode)object;
      CDOResourceFolder folder = node.getFolder();

      if (folder == null)
      {
        CDOView view = node.cdoView();

        for (Map.Entry<AssemblyModule, CDOView> entry : views.entrySet())
        {
          if (entry.getValue() == view)
          {
            return entry.getKey();
          }
        }

        return null;
      }

      return folder;
    }

    if (object instanceof AssemblyModule)
    {
      return configuration;
    }

    return super.elementGetParent(object, defaultGetParent);
  }

  @Override
  protected String elementGetText(Object object, Function<Object, String> defaultGetText)
  {
    if (object instanceof AssemblyModule)
    {
      AssemblyModule module = (AssemblyModule)object;
      return module.getName() + " " + module.getVersion();
    }

    return super.elementGetText(object, defaultGetText);
  }
}
