package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link RootFs}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableRootFs.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableRootFs implements RootFs {
  private final String type;
  private final List<String> layers;

  private ImmutableRootFs(String type, List<String> layers) {
    this.type = type;
    this.layers = layers;
  }

  /**
   * @return The value of the {@code type} attribute
   */
  @JsonProperty("Type")
  @Override
  public String type() {
    return type;
  }

  /**
   * @return The value of the {@code layers} attribute
   */
  @JsonProperty("Layers")
  @Override
  public List<String> layers() {
    return layers;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link RootFs#type() type} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for type
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableRootFs withType(String value) {
    String newValue = Objects.requireNonNull(value, "type");
    if (this.type.equals(newValue)) return this;
    return new ImmutableRootFs(newValue, this.layers);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link RootFs#layers() layers}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableRootFs withLayers(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableRootFs(this.type, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link RootFs#layers() layers}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of layers elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableRootFs withLayers(Iterable<String> elements) {
    if (this.layers == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableRootFs(this.type, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableRootFs} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableRootFs
        && equalTo(0, (ImmutableRootFs) another);
  }

  private boolean equalTo(int synthetic, ImmutableRootFs another) {
    return type.equals(another.type)
        && layers.equals(another.layers);
  }

  /**
   * Computes a hash code from attributes: {@code type}, {@code layers}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + type.hashCode();
    h += (h << 5) + layers.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code RootFs} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "RootFs{"
        + "type=" + type
        + ", layers=" + layers
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link RootFs} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable RootFs instance
   */
  public static ImmutableRootFs copyOf(RootFs instance) {
    if (instance instanceof ImmutableRootFs) {
      return (ImmutableRootFs) instance;
    }
    return ImmutableRootFs.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableRootFs ImmutableRootFs}.
   * <pre>
   * ImmutableRootFs.builder()
   *    .type(String) // required {@link RootFs#type() type}
   *    .layer|addAllLayers(String) // {@link RootFs#layers() layers} elements
   *    .build();
   * </pre>
   * @return A new ImmutableRootFs builder
   */
  public static ImmutableRootFs.Builder builder() {
    return new ImmutableRootFs.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableRootFs ImmutableRootFs}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_TYPE = 0x1L;
    private long initBits = 0x1L;

    private String type;
    private List<String> layers = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code RootFs} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(RootFs instance) {
      Objects.requireNonNull(instance, "instance");
      this.type(instance.type());
      addAllLayers(instance.layers());
      return this;
    }

    /**
     * Initializes the value for the {@link RootFs#type() type} attribute.
     * @param type The value for type 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Type")
    public final Builder type(String type) {
      this.type = Objects.requireNonNull(type, "type");
      initBits &= ~INIT_BIT_TYPE;
      return this;
    }

    /**
     * Adds one element to {@link RootFs#layers() layers} list.
     * @param element A layers element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder layer(String element) {
      this.layers.add(Objects.requireNonNull(element, "layers element"));
      return this;
    }

    /**
     * Adds elements to {@link RootFs#layers() layers} list.
     * @param elements An array of layers elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder layers(String... elements) {
      for (String element : elements) {
        this.layers.add(Objects.requireNonNull(element, "layers element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link RootFs#layers() layers} list.
     * @param elements An iterable of layers elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Layers")
    public final Builder layers(Iterable<String> elements) {
      this.layers.clear();
      return addAllLayers(elements);
    }

    /**
     * Adds elements to {@link RootFs#layers() layers} list.
     * @param elements An iterable of layers elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllLayers(Iterable<String> elements) {
      for (String element : elements) {
        this.layers.add(Objects.requireNonNull(element, "layers element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableRootFs ImmutableRootFs}.
     * @return An immutable instance of RootFs
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableRootFs build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableRootFs(type, createUnmodifiableList(true, layers));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_TYPE) != 0) attributes.add("type");
      return "Cannot build RootFs, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
