

/*
Run-time CPU feature detection on PowerPC64 AIX by using getsystemcfg.

Refs:
- https:

As of nightly-2024-09-07, is_powerpc_feature_detected doesn't support run-time detection on AIX.
https:

Run-time detection on PowerPC64 AIX is currently disabled by default as experimental
because we cannot run tests on the VM or real machine.
*/

include!("common.rs");


mod ffi {
    pub(crate) use crate::utils::ffi::{c_int, c_ulong};

    
    // https://github.com/rust-lang/libc/blob/0.2.158/src/unix/aix/mod.rs#L2058
    
    pub(crate) const SC_IMPL: c_int = 2;
    pub(crate) const POWER_8: c_ulong = 0x10000;
    pub(crate) const POWER_9: c_ulong = 0x20000;
    pub(crate) const POWER_10: c_ulong = 0x40000;

    
    extern "C" {
        
        // https://github.com/rust-lang/libc/blob/0.2.158/src/unix/aix/powerpc64.rs#L643
        pub(crate) fn getsystemcfg(name: c_int) -> c_ulong;
    }
}

#[cold]
fn _detect(info: &mut CpuInfo) {
    
    let impl_ = unsafe { ffi::getsystemcfg(ffi::SC_IMPL) };
    if impl_ == ffi::c_ulong::MAX {
        return;
    }
    
    
    
    if impl_ & (ffi::POWER_8 | ffi::POWER_9 | ffi::POWER_10) != 0 {
        info.set(CpuInfo::HAS_QUADWORD_ATOMICS);
    }
}

#[allow(
    clippy::alloc_instead_of_core,
    clippy::std_instead_of_alloc,
    clippy::std_instead_of_core,
    clippy::undocumented_unsafe_blocks,
    clippy::wildcard_imports
)]
#[cfg(test)]
mod tests {
    use super::*;

    
    
    
    
    
    
    
    
    #[allow(
        clippy::cast_possible_wrap,
        clippy::cast_sign_loss,
        clippy::cast_possible_truncation,
        clippy::no_effect_underscore_binding
    )]
    const _: fn() = || {
        let mut _getsystemcfg: unsafe extern "C" fn(ffi::c_int) -> ffi::c_ulong = ffi::getsystemcfg;
        _getsystemcfg = libc::getsystemcfg;
        static_assert!(ffi::SC_IMPL == libc::SC_IMPL);
        static_assert!(ffi::POWER_8 == libc::POWER_8 as ffi::c_ulong);
        static_assert!(ffi::POWER_9 == libc::POWER_9 as ffi::c_ulong);
        
    };
}
