powerdiverger <-
function (x, y = NULL, p = NULL, lambda = c( -2, -1, -1/2, 0, 2/3, 1 ), alternative = c("two.sided", "less",
    "greater"), df = NULL, conf.level = 0.94999999999999996, correct = FALSE ) {

    call <- match.call()

    alternative <- match.arg(alternative)
    if ((length(conf.level) != 1L) || is.na(conf.level) || (conf.level <= 0) || (conf.level >= 1)) stop("'conf.level' must be a single number between 0 and 1")


    DNAME <- deparse( substitute( x ) )

    if( is.data.frame( x ) ) x <- as.matrix( x )
    if( is.matrix( x ) && ( min(dim(x)) == 1L ) ) x <- as.vector(x)
    if( !is.matrix( x ) && !is.null( y ) ) {

        if( length(x) != length(y) ) stop( "powerdiverger: 'x' and 'y' must have the same length." )
	DNAME2 <- deparse(substitute(y))
	xname <- if( length( DNAME) > 1L || nchar( DNAME, "w") > 30 ) ""
		else DNAME

	yname <- if( length(DNAME2) > 1L || nchar( DNAME2, "w" ) > 30 ) ""
		else DNAME2
	
	OK <- complete.cases(x,y)

	x <- factor( x[OK] )
	y <- factor( y[OK] )
	if( !is.null(p) ) p <- p[OK]

	if( (nlevels(x) < 2L ) || (nlevels(y) < 2L ) ) stop( "powerdiverger: 'x' and 'y' must have at least 2 levels." )

	x <- table(x,y)

	names( dimnames(x) ) <- c( xname, yname )

	DNAME <- paste( paste( DNAME, collapse = "\n" ), "and", paste( DNAME2, collapse = "\n" ) )

    } # end of if 'y' is not null stmts.

    N <- sum(x)
    if( N == 0 ) stop( "powerdiverger: at least one entry of 'x' must be positive." )

    if( is.vector(x) ) {

        k <- length(x)
	TYPE <- "Goodness-of-Fit Test"

    } else k <- dim(x)[ 1 ]

    if (k > 2 || (k == 2) && !is.null(p)) alternative <- "two.sided"

    if( k == 1 && is.null(p) ) p <- 0.5
    if( is.vector(x) && is.null(p) ) p <- rep( 1/k, k )
    else if( is.vector(x) && length(p) != k ) stop( "powerdiverger: 'x' and 'p' must have the same length." )

    if( any( x < 0 ) || anyNA(x) ) stop( "powerdiverger: at least one entry of 'x' must be positive." )

    if( is.vector(x) ) {
    
        E <- N * p
	df <- k - 1

    } # end of if 'x' is a vector stmt.

    if( is.matrix(x) ) {

	if( is.null(p) ) TYPE <- "Test for independence"
	else TYPE <- "Test for given probabilities"

	if( is.null(p) ) {


            nr <- as.integer(nrow(x))
	    nc <- as.integer(ncol(x))

	    if( is.na(nr) || is.na(nc) || is.na(nr * nc) ) stop( "powerdiverger: invalid nrow(x) or ncol(x)", domain = NA )

	    sr <- rowSums(x)
	    sc <- colSums(x)

	    E <- p <- outer( sr, sc ) / N
	    dimnames(E) <- dimnames(p) <- dimnames(x)
	    df <- (nr - 1) * (nc - 1)

	} else E <- N * p
	# end of if else 'p' is null stmts.

    } # end of if 'x' is a matrix (with nrow > 1 and ncol > 1) stmts.

    if (k  < 1L) stop("not enough data")
    if (!is.null(p) && (any((p <= 0) )) ) stop("elements of 'p' must be > 0")

    if( length(lambda) == 1 ) return( powerdivergerWork( x = x, p = p, E = E, k=k, n=N, lambda = lambda,
    		alternative = alternative, df = df, conf.level = conf.level, correct = correct,
		type = TYPE, DNAME = DNAME ) )

    else {

	res <- list()
	for( i in 1:length(lambda) ) {

	    res[[ i ]] <- powerdivergerWork( x = x, p = p, E = E, k=k, n=N, lambda = lambda[i],
	                    alternative = alternative, df = df, conf.level = conf.level, correct = correct,
			    type = TYPE, DNAME = DNAME ) 

	} # end of for 'i' loop.

	out <- list( results = res, lambda = lambda, call = call )

	class( out ) <- "powerdiverged"
	return( out )

    } # end of if else lambda is or is not a vector stmts.

} # end of 'powerdiverger' function.

powerdivergerWork <-
function (x, p = NULL, E, k, n, lambda, alternative = c("two.sided", "less", 
    "greater"), df = NULL, conf.level = 0.94999999999999996, correct = FALSE, type, DNAME ) 
{

    # Calculate the proportions.  
    NVAL <- p

    # x <- cbind( x, n - x )
    # E <- cbind( n * p, n * ( 1 - p ) )
    if( lambda == -1 ) STATISTIC <- 2 * sum( E * log( E / x ) )
    else if( lambda == 0 ) STATISTIC <- 2 * sum( x * log( x / E ) )
    else STATISTIC <- ( 2 / ( lambda * (1 + lambda) ) ) * sum( x * ( ( x / E )^lambda - 1 ) )
    

    if( lambda == -2 ) METHOD <- "Neyman modified chi-square"
    else if( lambda == -1 ) METHOD <- "Kullbak-Leibler"
    else if( lambda == -0.5 ) METHOD <- "Freeman-Tukey"
    else if( lambda == 0 ) METHOD <- "likelihood-ratio"
    else if( lambda == 2/3 ) METHOD <- "Cressie-Read"
    else if( lambda == 1 ) METHOD <- "Pearson chi-square"
    else METHOD <- paste( "Power-divergence with lambda =", lambda )
    if( correct ) METHOD <- paste( METHOD, "with mean correction" )
    names( STATISTIC ) <- METHOD

    if( is.null( p ) ) METHOD <- paste( METHOD, "test for equality" )
    else METHOD <- paste( METHOD, "test given" )
    
    if( !correct ) METHOD <- paste( METHOD, "without moment correction." )
    else METHOD <- paste( METHOD, "proportions with " )

    if (is.null(p)) {
        p <- sum(x)/sum(n)
        if( is.null( df ) ) PARAMETER <- k - 1
	else PARAMETER <- df
    }
    else {
        if( is.null( df ) ) PARAMETER <- k
	else PARAMETER <- df
        names(NVAL) <- names(E)
    }
    names(PARAMETER) <- "df"
    if( is.null( df ) ) df <- PARAMETER
    x <- cbind(x, n - x)
    E <- cbind(n * p, n * (1 - p))

    if( correct ) {

	if( df != k - 1 ) warning( "powerdivergerWork: correct is true but degrees of freedom not k - 1.  Correction may be invalid." )
	tau <- sum( 1 / p )
        fm <- (lambda - 1) * (2 - 3 * k + tau) / 3 + (lambda - 1) * (lambda - 2) * (1 - 2*k + tau) / 4
	fv <- 2 - 2*k - k^2 + tau + (lambda - 1)*(8 - 12*k - 2*k^2 + 6*tau ) +
	    (lambda - 1)^2*(4 - 6*k - 3*k^2 + 5*tau)/3 + (lambda - 1)*(lambda - 2) *(2 - 4*k + 2*tau)
	sigmalam <- 1 + fv/(2 * (k-1)*n)
	mulam <- (k-1)*(1-sigmalam) + fm/n
	STATISTIC <- (STATISTIC - mulam)/sigmalam

    } # end of if 'correct' stmt.

    if (any(E < 5)) 
        warning("Chi-squared approximation may be incorrect")
    if (alternative == "two.sided") 
        PVAL <- pchisq( STATISTIC, PARAMETER, lower.tail = FALSE )
    else PVAL <- pnorm( STATISTIC, lower.tail = (alternative == "less") )

    if( !correct ) {
    	RVAL <- list(statistic = STATISTIC, parameter = PARAMETER, 
            p.value = as.numeric(PVAL), null.value = NVAL, 
            alternative = alternative, method = paste( METHOD, type, sep="\n" ), 
            data.name = DNAME)
	} else {

	RVAL <- list(statistic = STATISTIC, parameter = PARAMETER,
	            p.value = as.numeric(PVAL), null.value = NVAL,
		    alternative = alternative, method = paste( METHOD, type, sep="\n" ),
		    data.name = DNAME, mu.lambda = mulam, sigma.lambda = sigmalam )

	} # end of if else 'correct' stmts.
    class(RVAL) <- "htest"
    return(RVAL)

} # end of 'powerdivergerWork' function.

print.powerdiverged <- function( x, ... ) {

    res <- x$results
    lam <- x$lambda
    print( res[[ 1 ]][["data.name"]] )
    print( res[[ 1 ]][["alternative"]] )
    STATISTIC <- unlist( lapply( res, function(x) return( x$statistic ) ) )
    df <- res[[1]][["parameter"]]
    p_value <- unlist( lapply( res, function(x) return( x$p.value ) ) )

    y <- rbind( "lambda" = lam, "statistic" = STATISTIC, "p-value" = p_value )

    print( y )

    invisible( y )

} # end of 'print.powerdiverged' function.
