#' Compute the \eqn{k}-th order moment for an univariate truncated normal 
#' distribution.
#' 
#' The \code{utrunmnt} function uses the moment generating function to compute 
#'   any order of moment for the truncated normal distribution.
#'     
#' @param k Order of moment. It must be a non-negative integer.
#' @param mu Mean of parent normal distribution. Defaults to 0.
#' @param sd Standard deviation of parent normal distribution. Defaults to 1.
#' @param lower Lower limit. Defaults to -Inf.
#' @param upper Upper limit. Defaults to Inf.
#' 
#' @return a numeric value.    
#' 
#' @references Burkardt, J. (2014). The truncated normal distribution, \emph{Online document}, Available from: 
#' \url{https://people.sc.fsu.edu/~jburkardt/presentations/truncated_normal.pdf}.
#' 
#' @export
#'  
#' @examples 
#' utrunmnt(4, mu = 5, upper = 10)
#' utrunmnt(1, mu = 5, lower = -3, upper = 4, sd = 2)
#' 
utrunmnt <- function(k,                          # order of moment
                     mu = 0.,                    # mean of parent normal distribution
                     lower = -Inf,               # lower bound
                     upper = Inf,                # upper bound 
                     sd = 1.){                    # standard deviation of parent normal distribution

  dimcheck <- c(length(k), length(lower), length(upper), length(mu), length(sd))
  if(any(dimcheck != 1)) stop("One of arguments has a nonconformable dimension")
  
  moment <- univariatetrunmnt(k, mu, lower, upper, sd) 
  return(moment)
}
