#' Tipping Point Analysis (Model-Based)
#'
#' Performs a model-based tipping point analysis on time-to-event data by
#' repeatedly imputing censored observations under varying assumptions. The model-based
#' framework assumes that censored patients have a multiple of hazard fitted via a parametric survival model
#' compared to the rest of patients in the same arm (Akinson et al, 2019).
#'
#' @details
#' The **model-based tipping point analysis** provides a reproducible and intuitive
#' framework for exploring the robustness of treatment effects in time-to-event
#' (survival) endpoints when censoring may differ between study arms.
#'
#' A parametric survival model is fitted using maximum likelihood.
#' This function applies a hazard deflation on control arm or hazard inflation on treatment
#' arm, and impute survival times based on the parametric model with additional sampling of the parameters from a multivariate normal distribution.
#' This imputation procedure is iterated across a range of
#' tipping point parameters `tipping_range`. For each parameter value:
#' \enumerate{
#'   \item Multiple imputed datasets are generated (`J` replicates), where censored
#'         observations in the selected arm are reassigned
#'         event times according to the imputation method.
#'   \item A Cox proportional hazards model is fitted to each imputed dataset.
#'   \item Model estimates are pooled using **Rubin’s rules** to obtain a combined
#'         hazard ratio and confidence interval for that tipping point parameter.
#' }
#'
#' The process yields a series of results showing how the treatment effect changes
#' as increasingly conservative or optimistic assumptions are made about censored
#' observations. The *tipping point* is defined as the smallest value (hazard inflation)
#' or biggest value (hazard deflation) of the sensitivity parameter for which the upper
#' bound of the hazard ratio confidence interval crosses 1 - i.e., where the
#' apparent treatment benefit is lost.
#'
#' @param dat data.frame containing at least 5 columns: TRT01P (treatment arm as factor), AVAL (survival time), EVENT (event indicator), CNSRRS (censoring reason) and MAXAVAL (maximum potential survival time, duration between randomization to data cut-off)
#' @param reason Vector specifying censoring reasons to be imputed.
#' @param impute a string specifying the treatment arm(s) which require imputation. It must be one of the arms from variable TRT01P, the first level of TRT01P is considered as the control arm.
#' @param imputation_model used to fit model to observed data (should be "Weibull" or "exponential")
#' @param J numeric indicating number of imputations.
#' @param tipping_range Numeric vector. Hazard inflation (>1) for treatment arm imputation or deflation (<1) range for control arm imputation.
#' @param verbose Logical. If `TRUE`, prints progress and analysis details.
#' @param seed Integer, default as NULL. Random seed for reproducibility.
#' @param cox_fit A Cox model that will be used to calculate HRs on imputed datasets.
#'   In case of inclusion of stratification factors or covariates, conditional HR will be used.
#'
#' @return A `tipse` object containing:
#' \describe{
#'   \item{original data}{Input argument from 'data'.}
#'   \item{imputation_results}{A data frame of combined pooled model results across tipping points}
#'   \item{original_HR}{The original hazard ratio.}
#'   \item{reason_to_impute}{Input argument from 'reason'.}
#'   \item{arm_to_impute}{Input argument from 'impute'.}
#'   \item{method_to_impute}{Input argument from 'method'.}
#'   \item{imputation_data}{A list of imputed datasets for each tipping point value.}
#'   \item{seed}{Random seed.}
#' }
#' @import dplyr
#' @importFrom utils head tail
#' @export
#'
#' @references Atkinson, A., Kenward, M. G., Clayton, T., & Carpenter, J. R. (2019).
#' Reference‐based sensitivity analysis for time‐to‐event data. Pharmaceutical statistics, 18(6), 645-658.
#'
#' @examples
#' cox1 <- survival::coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200)
#' result <- tipping_point_model_based(
#'   dat = codebreak200,
#'   reason = "Early dropout",
#'   impute = "docetaxel",
#'   imputation_model = "weibull",
#'   J = 10,
#'   tipping_range = seq(0.1, 1, by = 0.05),
#'   cox_fit = cox1,
#'   verbose = TRUE
#' )
tipping_point_model_based <- function(dat,
                                      reason,
                                      impute,
                                      imputation_model = "weibull",
                                      J = 10,
                                      tipping_range = seq(0.05, 1, by = 0.05),
                                      cox_fit = NULL,
                                      verbose = FALSE,
                                      seed = NULL) {
  #----------------------------#
  # Setup and validation
  #----------------------------#
  # handle seed
  if (!exists(".Random.seed", envir = .GlobalEnv, inherits = FALSE))
    runif(1)
  if (is.null(seed))
    RNGstate <- get(".Random.seed", envir = .GlobalEnv)
  else {
    R.seed <- get(".Random.seed", envir = .GlobalEnv)
    set.seed(seed)
    RNGstate <- structure(seed, kind = as.list(RNGkind()))
    on.exit(assign(".Random.seed", R.seed, envir = .GlobalEnv))
  }

  dat <- sanitize_dataframe(dat)

  if (!inherits(cox_fit, "coxph")) {
    stop("Argument 'cox_fit' must be a valid cox model object, e.g. coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200).")
  }

  tipping_range <- sanitize_tipping_range(tipping_range)

  if (all(tipping_range <= 1)) {
    method <- "hazard deflation"
  } else {
    method <- "hazard inflation"
  }

  control <- levels(dat[["TRT01P"]])[1]
  trt <- levels(dat[["TRT01P"]])[2]

  HR <- exp(cox_fit$coefficients[paste0("TRT01P", trt)])

  if (!(impute %in% c(control, trt))) {
    stop("Argument 'impute' must be one of the arms in column TRT01P.")
  }

  if (length(reason) == 0) {
    stop("Argument 'reason' must specify at least one censoring reason to impute.")
  }

  #----------------------------#
  # Print setup info
  #----------------------------#
  if (verbose) {
    cat("\u2192 Detected arms:\n")
    cat("   Control arm   :", control, "\n")
    cat("   Treatment arm :", trt, "\n")
    cat("   Imputing arm  :", impute, "\n\n")
    cat("Starting tipping point analysis using method:", method, "\n")
    cat("Replicates per tipping point parameter:", J, "\n\n")
  }

  #----------------------------#
  # Main computation function
  #----------------------------#
  run_imputation <- function(param, method, cox_fit, verbose, dat, reason, impute, imputation_model, J, seed) {
    if (verbose) {
      label <- paste0(ifelse(method == "hazard inflation",
        round((param - 1) * 100, 0),
        round((1 - param) * 100, 0)
      ), "%")
      cat(" \u2192 Imputing for:", method, label, "\n")
    }

    multiply_imputed_dfs <- impute_model(dat, reason, impute, imputation_model, alpha = param, J = J, seed = seed)

    pooled <- pool_results(multiply_imputed_dfs, cox_fit)
    pooled$parameter <- param

    list(pooled = pooled, km_data = multiply_imputed_dfs)
  }

  #----------------------------#
  # Run all imputations
  #----------------------------#
  results <- lapply(tipping_range, run_imputation,
    method = method, cox_fit = cox_fit,
    verbose = verbose, dat = dat, reason = reason, impute = impute, imputation_model = imputation_model,
    J = J, seed = seed
  )

  summary_results <- dplyr::bind_rows(lapply(results, `[[`, "pooled"))
  km_data_list <- lapply(results, `[[`, "km_data")
  names(km_data_list) <- as.character(tipping_range)

  #----------------------------#
  # Check tipping point
  #----------------------------#

  summary_results <- summary_results %>% mutate(tipping_point = FALSE)

  if (any(summary_results$HR_upperCI < 1) & any(summary_results$HR_upperCI > 1)) {
    if (method == "hazard deflation") {
      tip <- tail(which(summary_results$HR_upperCI >= 1), 1)
    } else {
      tip <- head(which(summary_results$HR_upperCI >= 1), 1)
    }

    if (abs(summary_results$HR_upperCI[tip] - 1) > 0.1) {
      warning(sprintf(
        "The upper CL at tipping point was too far away from 1.0.",
        ifelse(method == "hazard inflation", "consider increasing tipping_range", "Full hazard deflation does not tip results.")
      ))
    }

    summary_results$tipping_point[tip] <- TRUE
  }


  if (verbose) cat("\nTipping point analysis completed successfully \u2705 \n")

  #----------------------------#
  # Return structured output
  #----------------------------#

  tipse <- list(
    original_data = dat,
    original_HR = HR,
    reason_to_impute = reason,
    arm_to_impute = impute,
    method_to_impute = method,
    imputation_results = summary_results,
    imputation_data = km_data_list,
    seed = RNGstate
  )
  class(tipse) <- "tipse"
  return(tipse)
}
