% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/collapse_to_rvec.R
\name{collapse_to_rvec}
\alias{collapse_to_rvec}
\alias{collapse_to_rvec.data.frame}
\alias{collapse_to_rvec.grouped_df}
\alias{expand_from_rvec}
\alias{expand_from_rvec.data.frame}
\alias{expand_from_rvec.grouped_df}
\title{Convert a Data Frame Between 'Draws-and-Values' Format
and 'Rvec' Format}
\usage{
collapse_to_rvec(data, draw = draw, values = value, by = NULL, type = NULL)

\method{collapse_to_rvec}{data.frame}(data, draw = draw, values = value, by = NULL, type = NULL)

\method{collapse_to_rvec}{grouped_df}(data, draw = draw, values = value, by = NULL, type = NULL)

expand_from_rvec(data, draw = "draw")

\method{expand_from_rvec}{data.frame}(data, draw = "draw")

\method{expand_from_rvec}{grouped_df}(data, draw = "draw")
}
\arguments{
\item{data}{A data frame, possibly
\href{https://dplyr.tidyverse.org/reference/group_data.html}{grouped}.}

\item{draw}{<\code{\link[tidyselect:language]{tidyselect}}>
The variable that uniquely identifies
random draws within each combination of
values for the 'by' variables. Must be quoted
for \code{expand_from_rvec()}.}

\item{values}{<\code{\link[tidyselect:language]{tidyselect}}>
One or more variables in \code{data} that hold measurements.}

\item{by}{<\code{\link[tidyselect:language]{tidyselect}}>
Variables used to stratify or cross-classify the data.
See Details.}

\item{type}{String specifying the class of rvec
to use for each variable.
Optional. See Details.}
}
\value{
A data frame.
\itemize{
\item \code{collapse_to_rvec()} \strong{reduces} the number of rows
by a factor of \code{\link[=n_draw]{n_draw()}}.
\item \code{expand_from_rvec()} \strong{increases} the number of rows
by a factor of \code{\link[=n_draw]{n_draw()}}.
\item \code{collapse_to_rvec()} silently drops all variables
that are not draw, value, or grouping variables
if \code{data} is a
\href{https://dplyr.tidyverse.org/reference/group_data.html}{grouped}
data frame.
}
}
\description{
\code{collapse_to_rvec()} converts a data frame from
a 'draws-and-values' format to an 'rvec' format.
\code{expand_from_rvec()}, does the opposite,
converting a data frame from an rvecs format
to a draws-and-values format.
}
\details{
In a draws-and-values format, each row represents
one random draw. The data frame contains
a 'draw' variable that distinguishes different
draws within the same combination
of 'by' variables. In rvec format,
each row represents one
combination of 'by' variables, and
multiple draws are stored in an \link[=rvec]{rvec}.
See below for examples.
}
\section{\code{by} argument}{


The \code{by} argument is used to specify stratifying
variables. For instance if \code{by} includes \code{sex} and \code{age},
then data frame produced by \code{collapse_to_rvec()}
has separate rows for each
combination of \code{sex} and \code{age}.

If \code{data} is a
\href{https://dplyr.tidyverse.org/reference/group_data.html}{grouped}
data frame, then the grouping variables
take precedence over \code{by}.

If no value for \code{by} is provided,
and \code{data} is not a grouped data frame,
then \code{collapse_to_rvec()}
assumes that all variables in \code{data} that are
not included in \code{value}
and \code{draw} should be included in \code{by}.
}

\section{\code{type} argument}{


By default, \code{collapse_to_rvec()} calls function
\code{\link[=rvec]{rvec()}} on each values variable in \code{data}.
\code{\link[=rvec]{rvec()}} chooses the class of the output (ie
\code{rvec_chr}, \code{rvec_dbl}, \code{rvec_int}, or \code{rvec_lgl})
depending on the input. Types can instead
be specified in advance, using the \code{type} argument.
\code{type} is a string, each character of which
specifies the class of the corresponding values variable.
The characters have the following meanings:
\itemize{
\item \code{"c"}: \code{rvec_chr}
\item \code{"d"}: \code{rvec_dbl}
\item \code{"i"}: \code{rvec_int}
\item \code{"l"}: \code{rvec_lgl}
\item \code{"?"}: Depends on inputs.
}

The codes for \code{type} are modified from ones used by the
\href{https://readr.tidyverse.org}{readr} package.
}

\examples{
library(dplyr)
data_db <- tribble(
  ~occupation,    ~sim, ~pay,
  "Statistician", 1,    100,
  "Statistician", 2,    80,
  "Statistician", 3,    105,
  "Banker",       1,    400,
  "Banker",       2,    350,
  "Banker",       3,    420
)

## draws-and-values format to rvec format
data_rv <- data_db |>
  collapse_to_rvec(draw = sim,
                   values = pay)
data_rv

## rvec format to draws-and-values format
data_rv |>
  expand_from_rvec()

## provide a name for the draw variable
data_rv |>
  expand_from_rvec(draw = "sim")

## specify that rvec variable
## must be rvec_int
data_rv <- data_db |>
  collapse_to_rvec(draw = sim,
                   values = pay,
                   type = "i")

## specify stratifying variable explicitly,
## using 'by' argument
data_db |>
  collapse_to_rvec(draw = sim,
                   values = pay,
                   by = occupation)

## specify stratifying variable explicitly,
## using 'group_by'
library(dplyr)
data_db |>
  group_by(occupation) |>
  collapse_to_rvec(draw = sim,
                   values = pay)
}
\seealso{
\itemize{
\item \code{\link[=rvec]{rvec()}} Construct a single rvec
\item \code{\link[=as_list_col]{as_list_col()}} Convert an rvec
to a list variable
\item \href{https://dplyr.tidyverse.org/reference/group_data.html}{dplyr::group_vars()}
Names of grouping variables
}

\code{collapse_to_rvec()} and \code{expand_from_rvec()}
are analogous to
\href{https://tidyr.tidyverse.org/reference/nest.html}{tidyr::nest()}
and
\href{https://tidyr.tidyverse.org/reference/unnest.html}{tidyr::unnest()}
though \code{collapse_to_rvec()} and
\code{expand_from_rvec()} move values into and
out of rvecs, while \code{tidyr::nest()} and
\code{tidyr::unnest()} move them in and out
of data frames. (\code{tidyr::nest()} and
\code{tidyr::unnest()} are also a lot
more flexible.)
}
