#' Simulate Data for Penalized Partial Least Squares (PPLS)
#'
#' Generates a training and test dataset with non-linear relationships between predictors and response, as used in PPLS simulation studies.
#'
#' @name sim.data.ppls
#' @param ntrain Integer. Number of training observations.
#' @param ntest Integer. Number of test observations.
#' @param stnr Numeric. Signal-to-noise ratio (higher means less noise).
#' @param p Integer. Number of predictors (must be \code{>= 5}).
#' @param a Optional numeric vector of length 5. Linear coefficients for the first 5 variables. If \code{NULL}, drawn uniformly from \code{[-1, 1]}.
#' @param b Optional numeric vector of length 5. Nonlinear sine coefficients. If \code{NULL}, drawn uniformly from \code{[-1, 1]}.
#'
#' @details
#' The function simulates a response variable \code{y} as a combination of additive linear and sinusoidal effects of the first 5 predictors:
#' \deqn{f(x) = \sum_{j=1}^{5} a_j x_j + \sin(6 b_j x_j)}
#' The response \code{y} is then generated by adding Gaussian noise scaled to match the specified signal-to-noise ratio (stnr).
#'
#' Remaining variables (\code{p - 5}) are included as noise variables, making the dataset suitable to evaluate selection or regularization methods.
#'
#' @returns A list with the following components:
#' \describe{
#'   \item{Xtrain}{\code{ntrain x p} matrix of training predictors (uniform in \code{[-1, 1]}).}
#'   \item{ytrain}{Numeric vector of training responses.}
#'   \item{Xtest}{\code{ntest x p} matrix of test predictors.}
#'   \item{ytest}{Numeric vector of test responses.}
#'   \item{sigma}{Standard deviation of the added noise.}
#'   \item{a}{Linear coefficients used in the simulation.}
#'   \item{b}{Nonlinear sine coefficients used in the simulation.}
#' }
#'
#' @seealso \code{\link{ppls.splines.cv}}, \code{\link{graphic.ppls.splines}}
#'
#' @examples
#' set.seed(123)
#' sim <- sim.data.ppls(ntrain = 100, ntest = 100, stnr = 3, p = 10)
#' str(sim)
#' plot(sim$Xtrain[, 1], sim$ytrain, main = "Effect of x1 on y")
#'
#' @import stats
#' @export
sim.data.ppls <- function(ntrain,
                          ntest,
                          stnr,
                          p,
                          a = NULL,
                          b = NULL) {
  if (is.null(a)) a <- runif(5, -1, 1)
  if (is.null(b)) b <- runif(5, -1, 1)
  p <- max(p, 5)
  Xtrain <- matrix(runif(p * ntrain, -1, 1), ncol = p)
  Xtest <- matrix(runif(p * ntest, -1, 1), ncol = p)
  fxtrain <- 0
  fxtest <- 0
  for (j in 1:5) {
    fxtrain <- fxtrain + a[j] * Xtrain[, j] + sin(6 * b[j] * Xtrain[, j])
    fxtest <- fxtest + a[j] * Xtest[, j] + b[j] * sin(6 * Xtest[, j])
  }
  sigma <- sqrt(var(fxtrain) / stnr)
  ytrain <- fxtrain + rnorm(ntrain, 0, sigma)
  ytest <- fxtest + rnorm(ntest, 0, sigma)
  return(
    list(
      Xtrain = Xtrain,
      ytrain = ytrain,
      Xtest = Xtest,
      ytest = ytest,
      sigma = sigma,
      a = a,
      b = b
    )
  )
}
