#' Borda Count - BC Function
#' 
#' @rdname ElecFuns
#' @param m Number of seats to be awarded. 
#' @param mod Included in the BC function. If TRUE, makes the sequence run 
#' from 1 to the number of votes and inverses it;  
#' if FALSE, the sequence run from the number of votes to 1). Should be TRUE
#' if the intention is to use the BC system described in the details. 
#' @param n_cand_bc Number of candidates in the Borda Count system. This is only
#' used for the Borda Count system.
#' 
#' @section Details: The bc function is used in single-member or multi-member 
#' districts, though it is typically discussed in settings that return a single 
#' choice. Voters assign candidates a rank, but seats are then awarded by 
#' plurality rather than majority. Each rank is assigned a weight, and a 
#' candidate’s vote total is the sum of the full and fractional votes he or she 
#' receives. For example, a first place rank might be weighted by one, a second 
#' place rank by 1/2, a third place rank by 1/3. A candidate ranked first by one 
#' hundred voters, second by twenty voters, and third by ten voters would be 
#' awarded a total of 100/1 + 20/2 + 12/3 =114 votes. This, in fact, is the 
#' modified bc system used to elect the Parliament in the country of Nauru 
#' (not included in our dataset), an island in Micronesia. To our knowledge, 
#' the bc is not used in national elections elsewhere.
#' @return For Borda count and all PR formulas, a vector of seats awarded to each candidate.
#' @export
#' 
#' @examples  
#'                      
#' bc(v=ranked, 2, mod=TRUE, n_cand_bc=3)

bc <- function(v, #rank x voter
               m,
               mod = TRUE,
               n_cand_bc,
               ...){
  
  ##Convert to rank x candidate
  n_cand <- n_cand_bc
  n_votes <- nrow(v)
  v_tmp <- array(0, c(n_votes, n_cand))
  colnames(v_tmp) <- sort(unique(c(v)))
  for(i in 1:n_votes){
    tmp <- table(v[i, ]) # Table results
    if(names(tmp)[1] == "0"){ 
      tmp <- tmp[-1] # Remove 0 from table (0 means that voters abstain)
    }
    v_tmp[i, sort(unique(v[i, ]))] <- tmp
  }
  if(mod){
    w <- 1/seq(1:n_votes)
  } else {
    w <- seq(n_votes:1)
  }
  w_votes <- v_tmp * w
  totals <- colSums(w_votes)
  seated <- order(totals, decreasing = TRUE)[1:m]
  seats <- rep(0, ncol(v_tmp))
  names(seats) <- 1:n_cand
  seats[seated] <- 1 
  return(seats)
}
