#' Electoral Functions
#'
#' This file contains details and examples of the electoral functions (electoral formulas) implemented in the i3pack package.
#' 
#' @name ElecFuns
#' @param v Matrix with candidates/parties (the interpretation depends
#' on the function) in the columns. Each cell has the number of votes cast for
#' each candidate/party. For AV and STV, the matrix should have ranked votes, 
#' with each voter in a separate row and each candidate in a separate column. 
#' For Borda Count, the matrix should have ranks in the rows and voters in the columns, 
#' with each cell containing the candidate ID (numeric or character). 
#' @param ... Additional optional arguments (currently ignored).
#'
#' @section Details: The a_v function is used in single member districts and lets 
#' voters rank the candidates competing in the district from most to least preferred. 
#' If a candidate is ranked first by a majority of voters, he or she 
#' wins the single seat available. If no candidate obtains a majority of votes, the
#' candidate with the fewest first-place votes is eliminated and her votes are 
#' distributed to the candidates that her supporters ranked second. If this redistribution
#' gives another candidate a majority of the votes, that candidate is elected; if not, the
#' second weakest candidate is eliminated and his votes are redistributed among the
#' surviving candidates. This process of redistribution and recounting continues
#' until a candidate obtains majority support.
#'
#' @return For Alternative Vote, the name of the candidate that obtains majority support.
#' @examples
#' 
#' a_v(v=ranked)
#'  
#' @export 
#' @importFrom STV stv
#'

a_v <- function(v, # ranks x voters
                ...
){
  requireNamespace("STV", quietly = TRUE)
  ## Convert to format required by STV package
  cands <- unique(c(v))
  if(any(cands==0)){
    cands <- cands[-which(cands==0)]
  }
  n_cand <- length(cands)
  ballots <- t(apply(v, 2, 
                     function(x, n, id){
                       cands_tmp <- rep(NA,n)
                       if(all(x!=0)){
                         cands_tmp[match(x,id)] <- 1:length(x)
                       }
                       return(cands_tmp)
                     }, n = n_cand, id = cands))
  colnames(ballots) <- cands
  viable_party <- which(apply(ballots, 2, function(x)!all(is.na(x))))
  viable_votes <- which(apply(ballots, 1, function(x)!all(is.na(x))))
  stv_ballots <- as.data.frame(ballots[viable_votes,viable_party])
  return(STV::stv(stv_ballots, 1)$elected)
}