% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve_full_pregnancy.R
\name{solve_full_pregnancy}
\alias{solve_full_pregnancy}
\title{Solve_full_pregnancy}
\usage{
solve_full_pregnancy(
  chem.name = NULL,
  chem.cas = NULL,
  dtxsid = NULL,
  time.course = seq(0, 40 * 7, 1),
  dose = NULL,
  daily.dose = NULL,
  doses.per.day = NULL,
  class.exclude = TRUE,
  physchem.exclude = TRUE,
  track.vars = NULL,
  plt = FALSE
)
}
\arguments{
\item{chem.name}{Either the chemical name, CAS number, or DTXSID
must be specified.}

\item{chem.cas}{Either the chemical name, CAS number, or DTXSID must be specified.}

\item{dtxsid}{EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})}

\item{time.course}{Time sequence in days. Default is from 0th week of pregnancy to 
40th, incremented by day.}

\item{dose}{Amount of a single, initial dose (on day 0) in mg/kg BW.}

\item{daily.dose}{Total daily dose, mg/kg BW for 40 weeks.}

\item{doses.per.day}{Number of doses per day for 40 weeks.}

\item{class.exclude}{Exclude chemical classes identified as outside of 
domain of applicability for fetal_pbtk and 1tri_pbtk models (i.e. PFAS chemicals).}

\item{physchem.exclude}{Exclude chemicals on the basis of physico-chemical
properties (currently only Henry's law constant) as specified by the modelinfo
files for fetal_pbtk and 1tri_pbtk.}

\item{track.vars}{which variables to return in solution output dataframe}

\item{plt}{plots all outputs, if TRUE}
}
\value{
A matrix with columns for time (in days), each compartment, the 
area under the curve (for plasma vs time), and plasma, and a row for each time 
point.
}
\description{
This function solves for the amounts (in umol) or concentrations (in uM) of a 
chemical in different tissues of a maternal-fetal system over the full course 
of human pregnancy given a dose and dosing frequency.
}
\details{
The simulation starts at the 0th week and ends at 40 weeks of pregnancy (term), 
covering all trimesters of human pregnancy. This is accomplished by 
stitching together the 1tri and fetal PBTK models with appropriate initial 
conditions, as described in Truong et al. (TBD).
}
\examples{

\donttest{
library(httk)

# dosing schedule of 1 mg/kg BW/day for 40 weeks
# return solution by hour
out <- solve_full_pregnancy(chem.name = "fipronil",  
                           daily.dose = 1, 
                           doses.per.day = 1,
                           time.course = seq(0, 40*7, 1/24))
                   
                               
# return solution in chemical amounts for fetal compartments + placenta
maternal_compts <- c('gutlumen', 'gut', 'liver', 'kidney', 'lung', 'ven', 'art', 
'adipose','thyroid', 'rest')

fetal_compts <- c(maternal_compts[! maternal_compts \%in\% c('adipose', 'gutlumen') ], 
"brain")

amt.out <- solve_full_pregnancy(chem.name = "fipronil",  
                               daily.dose = 1, 
                               doses.per.day = 1,
                               time.course = seq(0, 40*7, 1), 
                               track.vars = c(paste0("Af", fetal_compts), "Aplacenta"))

# return solution in concentrations for fetal compartments + placenta 
conc.out <- solve_full_pregnancy(chem.name = "fipronil", 
                                daily.dose = 1, 
                                doses.per.day = 1,
                                time.course = seq(0, 40*7, 1), 
                                track.vars = c(paste0("Cf", fetal_compts), "Cplacenta"))

}
}
\references{
\insertRef{kapraun2022fetalmodel}{httk}

\insertRef{truong2025fullpregnancy}{httk}
}
\seealso{
\code{\link{solve_1tri_pbtk}}

\code{\link{solve_fetal_pbtk}}

\code{\link{parameterize_1tri_pbtk}}

\code{\link{parameterize_fetal_pbtk}}
}
\author{
Kimberly Truong
}
\keyword{Solve}
