/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5G_FRIEND     
#define H5L_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5Lpkg.h"      
#include "H5Opkg.h"      

static void  *H5O__linfo_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                size_t p_size, const uint8_t *p);
static herr_t H5O__linfo_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                const void *_mesg);
static void  *H5O__linfo_copy(const void *_mesg, void *_dest);
static size_t H5O__linfo_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__linfo_free(void *_mesg);
static herr_t H5O__linfo_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);
static void  *H5O__linfo_copy_file(H5F_t *file_src, void *native_src, H5F_t *file_dst, bool *recompute_size,
                                   unsigned *mesg_flags, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__linfo_post_copy_file(const H5O_loc_t *parent_src_oloc, const void *mesg_src,
                                        H5O_loc_t *dst_oloc, void *mesg_dst, unsigned *mesg_flags,
                                        H5O_copy_t *cpy_info);
static herr_t H5O__linfo_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_LINFO[1] = {{
    H5O_LINFO_ID,              
    "linfo",                   
    sizeof(H5O_linfo_t),       
    0,                         
    H5O__linfo_decode,         
    H5O__linfo_encode,         
    H5O__linfo_copy,           
    H5O__linfo_size,           
    NULL,                      
    H5O__linfo_free,           
    H5O__linfo_delete,         
    NULL,                      
    NULL,                      
    NULL,                      
    NULL,                      
    H5O__linfo_copy_file,      
    H5O__linfo_post_copy_file, 
    NULL,                      
    NULL,                      
    H5O__linfo_debug           
}};

#define H5O_LINFO_VERSION 0

#define H5O_LINFO_TRACK_CORDER 0x01
#define H5O_LINFO_INDEX_CORDER 0x02
#define H5O_LINFO_ALL_FLAGS    (H5O_LINFO_TRACK_CORDER | H5O_LINFO_INDEX_CORDER)

typedef struct {
    const H5O_loc_t *src_oloc;  
    H5O_loc_t       *dst_oloc;  
    H5O_linfo_t     *dst_linfo; 
    H5O_copy_t      *cpy_info;  
} H5O_linfo_postcopy_ud_t;

H5FL_DEFINE_STATIC(H5O_linfo_t);

static void *
H5O__linfo_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                  unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    const uint8_t *p_end = p + p_size - 1;         
    H5O_linfo_t   *linfo = NULL;                   
    unsigned char  index_flags;                    
    uint8_t        addr_size = H5F_SIZEOF_ADDR(f); 
    void          *ret_value = NULL;               

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

    
    if (*p++ != H5O_LINFO_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (linfo = H5FL_MALLOC(H5O_linfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    index_flags = *p++;
    if (index_flags & ~H5O_LINFO_ALL_FLAGS)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad flag value for message");
    linfo->track_corder = (index_flags & H5O_LINFO_TRACK_CORDER) ? true : false;
    linfo->index_corder = (index_flags & H5O_LINFO_INDEX_CORDER) ? true : false;

    
    linfo->nlinks = HSIZET_MAX;

    
    if (linfo->track_corder) {
        if (H5_IS_BUFFER_OVERFLOW(p, 8, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        INT64DECODE(p, linfo->max_corder);
        if (linfo->max_corder < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                        "invalid max creation order value for message: %" PRId64, linfo->max_corder);
    }
    else
        linfo->max_corder = 0;

    
    if (H5_IS_BUFFER_OVERFLOW(p, addr_size + addr_size, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

    
    H5F_addr_decode(f, &p, &(linfo->fheap_addr));

    
    H5F_addr_decode(f, &p, &(linfo->name_bt2_addr));

    
    if (linfo->index_corder) {
        H5_WARN_USELESS_COMPARISON_OFF
        if (H5_IS_BUFFER_OVERFLOW(p, addr_size, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5_WARN_USELESS_COMPARISON_ON
        H5F_addr_decode(f, &p, &(linfo->corder_bt2_addr));
    }
    else
        linfo->corder_bt2_addr = HADDR_UNDEF;

    
    ret_value = linfo;

done:
    if (ret_value == NULL)
        if (linfo != NULL)
            linfo = H5FL_FREE(H5O_linfo_t, linfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__linfo_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                  const void *_mesg)
{
    const H5O_linfo_t *linfo = (const H5O_linfo_t *)_mesg;
    unsigned char      index_flags; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(linfo);

    
    *p++ = H5O_LINFO_VERSION;

    
    index_flags = (uint8_t)(linfo->track_corder ? H5O_LINFO_TRACK_CORDER : 0);
    index_flags = (uint8_t)(index_flags | (linfo->index_corder ? H5O_LINFO_INDEX_CORDER : 0));
    *p++        = index_flags;

    
    if (linfo->track_corder)
        INT64ENCODE(p, linfo->max_corder);

    
    H5F_addr_encode(f, &p, linfo->fheap_addr);

    
    H5F_addr_encode(f, &p, linfo->name_bt2_addr);

    
    if (linfo->index_corder)
        H5F_addr_encode(f, &p, linfo->corder_bt2_addr);
    else
        assert(!H5_addr_defined(linfo->corder_bt2_addr));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__linfo_copy(const void *_mesg, void *_dest)
{
    const H5O_linfo_t *linfo     = (const H5O_linfo_t *)_mesg;
    H5O_linfo_t       *dest      = (H5O_linfo_t *)_dest;
    void              *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(linfo);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_linfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *linfo;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__linfo_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_linfo_t *linfo     = (const H5O_linfo_t *)_mesg;
    size_t             ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value =
        1                                       
        + 1                                     
        + (linfo->track_corder ? (size_t)8 : 0) 
        + (size_t)H5F_SIZEOF_ADDR(f)            
        + (size_t)H5F_SIZEOF_ADDR(f)            
        + (linfo->index_corder ? (size_t)H5F_SIZEOF_ADDR(f)
                               : 0); 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__linfo_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_linfo_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__linfo_delete(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, void *_mesg)
{
    H5O_linfo_t *linfo     = (H5O_linfo_t *)_mesg;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);

    
    if (H5_addr_defined(linfo->fheap_addr))
        if (H5G__dense_delete(f, linfo, true) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free dense link storage");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__linfo_copy_file(H5F_t H5_ATTR_UNUSED *file_src, void *native_src, H5F_t *file_dst,
                     bool H5_ATTR_UNUSED *recompute_size, unsigned H5_ATTR_UNUSED *mesg_flags,
                     H5O_copy_t *cpy_info, void *_udata)
{
    H5O_linfo_t        *linfo_src = (H5O_linfo_t *)native_src;
    H5O_linfo_t        *linfo_dst = NULL;
    H5G_copy_file_ud_t *udata     = (H5G_copy_file_ud_t *)_udata;
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_TAG(H5AC__COPIED_TAG)

    
    assert(linfo_src);
    assert(cpy_info);

    
    if (NULL == (linfo_dst = (H5O_linfo_t *)H5O__linfo_copy(linfo_src, NULL)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "memory allocation failed");

    
    if (cpy_info->max_depth >= 0 && cpy_info->curr_depth >= cpy_info->max_depth) {
        linfo_dst->nlinks          = 0;
        linfo_dst->max_corder      = 0;
        linfo_dst->fheap_addr      = HADDR_UNDEF;
        linfo_dst->name_bt2_addr   = HADDR_UNDEF;
        linfo_dst->corder_bt2_addr = HADDR_UNDEF;
    } 
    else {
        
        
        if (H5_addr_defined(linfo_src->fheap_addr)) {
            
            if (H5G__dense_create(file_dst, linfo_dst, udata->common.src_pline) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, NULL, "unable to create 'dense' form of new format group");
        } 
    }     

    
    ret_value = linfo_dst;

done:
    if (!ret_value)
        if (linfo_dst)
            linfo_dst = H5FL_FREE(H5O_linfo_t, linfo_dst);

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static herr_t
H5O__linfo_post_copy_file_cb(const H5O_link_t *src_lnk, void *_udata)
{
    H5O_linfo_postcopy_ud_t *udata = (H5O_linfo_postcopy_ud_t *)_udata; 
    H5O_link_t               dst_lnk;                                   
    bool                     dst_lnk_init = false;        
    herr_t                   ret_value    = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(src_lnk);
    assert(udata);

    
    if (H5L__link_copy_file(udata->dst_oloc->file, src_lnk, udata->src_oloc, &dst_lnk, udata->cpy_info) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, H5_ITER_ERROR, "unable to copy link");
    dst_lnk_init = true;

    
    H5_BEGIN_TAG(H5AC__COPIED_TAG)

    
    
    if (H5G__dense_insert(udata->dst_oloc->file, udata->dst_linfo, &dst_lnk) < 0)
        HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTINSERT, H5_ITER_ERROR, "unable to insert destination link");

    
    H5_END_TAG

done:
    
    if (dst_lnk_init)
        H5O_msg_reset(H5O_LINK_ID, &dst_lnk);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__linfo_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc,
                          void *mesg_dst, unsigned H5_ATTR_UNUSED *mesg_flags, H5O_copy_t *cpy_info)
{
    const H5O_linfo_t *linfo_src = (const H5O_linfo_t *)mesg_src;
    H5O_linfo_t       *linfo_dst = (H5O_linfo_t *)mesg_dst;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(src_oloc && src_oloc->file);
    assert(linfo_src);
    assert(dst_oloc && dst_oloc->file);
    assert(H5_addr_defined(dst_oloc->addr));
    assert(linfo_dst);
    assert(cpy_info);

    
    if (cpy_info->max_depth >= 0 && cpy_info->curr_depth >= cpy_info->max_depth)
        HGOTO_DONE(SUCCEED);

    
    if (H5_addr_defined(linfo_src->fheap_addr)) {
        H5O_linfo_postcopy_ud_t udata; 

        
        udata.src_oloc  = src_oloc;
        udata.dst_oloc  = dst_oloc;
        udata.dst_linfo = linfo_dst;
        udata.cpy_info  = cpy_info;

        
        if (H5G__dense_iterate(src_oloc->file, linfo_src, H5_INDEX_NAME, H5_ITER_NATIVE, (hsize_t)0, NULL,
                               H5O__linfo_post_copy_file_cb, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTNEXT, FAIL, "error iterating over links");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__linfo_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_linfo_t *linfo = (const H5O_linfo_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(linfo);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Track creation order of links:", linfo->track_corder ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Index creation order of links:", linfo->index_corder ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth, "Number of links:", linfo->nlinks);
    Rfprintf(stream, "%*s%-*s %" PRId64 "\n", indent, "", fwidth,
            "Max. creation order value:", linfo->max_corder);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "'Dense' link storage fractal heap address:", linfo->fheap_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "'Dense' link storage name index v2 B-tree address:", linfo->name_bt2_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "'Dense' link storage creation order index v2 B-tree address:", linfo->corder_bt2_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
