/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Imodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Ipkg.h"      
#include "H5MMprivate.h" 
#include "H5Tprivate.h"  
#include "H5VLprivate.h" 

#define H5I_MAKE(g, i) ((((hid_t)(g) & TYPE_MASK) << ID_BITS) | ((hid_t)(i) & ID_MASK))

typedef struct {
    const void *object;   
    H5I_type_t  obj_type; 
    hid_t       ret_id;   
} H5I_get_id_ud_t;

typedef struct {
    H5I_search_func_t user_func;  
    void             *user_udata; 
    bool              app_ref;    
    H5I_type_t        obj_type;   
} H5I_iterate_ud_t;

typedef struct {
    H5I_type_info_t *type_info; 
    bool             force;     
    bool             app_ref;   
} H5I_clear_type_ud_t;

static void  *H5I__unwrap(void *object, H5I_type_t type);
static herr_t H5I__mark_node(void *_id, void *key, void *udata);
static void  *H5I__remove_common(H5I_type_info_t *type_info, hid_t id);
static int    H5I__dec_ref(hid_t id, void **request);
static int    H5I__dec_app_ref(hid_t id, void **request);
static int    H5I__dec_app_ref_always_close(hid_t id, void **request);
static int    H5I__find_id_cb(void *_item, void *_key, void *_udata);

bool H5_PKG_INIT_VAR = false;

H5I_type_info_t *H5I_type_info_array_g[H5I_MAX_NUM_TYPES];
int              H5I_next_type_g = (int)H5I_NTYPES;

H5FL_DEFINE_STATIC(H5I_id_info_t);

static bool H5I_marking_s = false;

int
H5I_term_package(void)
{
    int in_use = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        H5I_type_info_t *type_info = NULL; 
        int              i;

        
        for (i = 0; i < H5I_next_type_g; i++)
            if ((type_info = H5I_type_info_array_g[i]) && type_info->hash_table)
                in_use++;

        
        if (0 == in_use) {
            for (i = 0; i < H5I_next_type_g; i++) {
                type_info = H5I_type_info_array_g[i];
                if (type_info) {
                    assert(NULL == type_info->hash_table);
                    type_info                = H5MM_xfree(type_info);
                    H5I_type_info_array_g[i] = NULL;
                    in_use++;
                }
            }

            
            if (0 == in_use)
                H5_PKG_INIT_VAR = false;
        }
    }

    FUNC_LEAVE_NOAPI(in_use)
} 

H5I_type_t
H5I__register_type_common(unsigned reserved, H5I_free_t free_func)
{
    H5I_class_t *cls       = NULL;      
    H5I_type_t   new_type  = H5I_BADID; 
    H5I_type_t   ret_value = H5I_BADID; 

    FUNC_ENTER_PACKAGE

    

    
    if (H5I_next_type_g < H5I_MAX_NUM_TYPES) {
        new_type = (H5I_type_t)H5I_next_type_g;
        H5I_next_type_g++;
    }
    else {
        bool done; 
        int  i;

        
        done = false;
        for (i = H5I_NTYPES; i < H5I_MAX_NUM_TYPES && done == false; i++) {
            if (NULL == H5I_type_info_array_g[i]) {
                
                new_type = (H5I_type_t)i;
                done     = true;
            }
        }

        
        if (done == false)
            HGOTO_ERROR(H5E_ID, H5E_NOSPACE, H5I_BADID, "Maximum number of ID types exceeded");
    }

    
    if (NULL == (cls = H5MM_calloc(sizeof(H5I_class_t))))
        HGOTO_ERROR(H5E_ID, H5E_CANTALLOC, H5I_BADID, "ID class allocation failed");

    
    cls->type      = new_type;
    cls->flags     = H5I_CLASS_IS_APPLICATION;
    cls->reserved  = reserved;
    cls->free_func = free_func;

    
    if (H5I_register_type(cls) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, H5I_BADID, "can't initialize ID class");

    
    ret_value = new_type;

done:
    
    if (ret_value == H5I_BADID)
        if (cls)
            cls = H5MM_xfree(cls);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5I_register_type(const H5I_class_t *cls)
{
    H5I_type_info_t *type_info = NULL;    
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cls);
    assert(cls->type > 0 && (int)cls->type < H5I_MAX_NUM_TYPES);

    
    if (NULL == H5I_type_info_array_g[cls->type]) {
        
        if (NULL == (type_info = (H5I_type_info_t *)H5MM_calloc(sizeof(H5I_type_info_t))))
            HGOTO_ERROR(H5E_ID, H5E_CANTALLOC, FAIL, "ID type allocation failed");
        H5I_type_info_array_g[cls->type] = type_info;
    }
    else {
        
        type_info = H5I_type_info_array_g[cls->type];
    }

    
    if (type_info->init_count == 0) {
        type_info->cls          = cls;
        type_info->id_count     = 0;
        type_info->nextid       = cls->reserved;
        type_info->last_id_info = NULL;
        type_info->hash_table   = NULL;
    }

    
    type_info->init_count++;

done:
    
    if (ret_value < 0)
        if (type_info)
            H5MM_free(type_info);

    FUNC_LEAVE_NOAPI(ret_value)
} 

int64_t
H5I_nmembers(H5I_type_t type)
{
    H5I_type_info_t *type_info = NULL; 
    int64_t          ret_value = 0;    

    FUNC_ENTER_NOAPI((-1))

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");
    if (NULL == (type_info = H5I_type_info_array_g[type]) || type_info->init_count <= 0)
        HGOTO_DONE(0);

    
    H5_CHECKED_ASSIGN(ret_value, int64_t, type_info->id_count, uint64_t);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5I__unwrap(void *object, H5I_type_t type)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(object);

    
    if (H5I_FILE == type || H5I_GROUP == type || H5I_DATASET == type || H5I_ATTR == type) {
        const H5VL_object_t *vol_obj;

        vol_obj   = (const H5VL_object_t *)object;
        ret_value = H5VL_object_data(vol_obj);
    }
    else if (H5I_DATATYPE == type) {
        H5T_t *dt = (H5T_t *)object;

        ret_value = (void *)H5T_get_actual_type(dt);
    }
    else
        ret_value = object;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5I_clear_type(H5I_type_t type, bool force, bool app_ref)
{
    H5I_clear_type_ud_t udata; 
    H5I_id_info_t      *item      = NULL;
    H5I_id_info_t      *tmp       = NULL;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");

    udata.type_info = H5I_type_info_array_g[type];
    if (udata.type_info == NULL || udata.type_info->init_count <= 0)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "invalid type");

    
    udata.force   = force;
    udata.app_ref = app_ref;

    

    
    H5I_marking_s = true;

    
    HASH_ITER(hh, udata.type_info->hash_table, item, tmp)
    {
        if (!item->marked)
            if (H5I__mark_node((void *)item, NULL, (void *)&udata) < 0)
                HGOTO_ERROR(H5E_ID, H5E_BADITER, FAIL, "iteration failed while clearing the ID type");
    }

    
    H5I_marking_s = false;

    
    HASH_ITER(hh, udata.type_info->hash_table, item, tmp)
    {
        if (item->marked) {
            HASH_DELETE(hh, udata.type_info->hash_table, item);
            item = H5FL_FREE(H5I_id_info_t, item);
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5I__mark_node(void *_info, void H5_ATTR_UNUSED *key, void *_udata)
{
    H5I_id_info_t       *info  = (H5I_id_info_t *)_info;        
    H5I_clear_type_ud_t *udata = (H5I_clear_type_ud_t *)_udata; 
    bool                 mark  = false;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(info);
    assert(udata);
    assert(udata->type_info);

    
    if (udata->force || (info->count - (!udata->app_ref * info->app_count)) <= 1) {
        
        if (info->is_future) {
            herr_t status;

            
            H5_BEFORE_USER_CB_NOCHECK
                {
                    
                    status = (info->discard_cb)(info->u.object);
                }
            H5_AFTER_USER_CB_NOCHECK
            if (status < 0) {
                if (udata->force) {
                    
                    mark = true;
                }
            }
            else {
                
                mark = true;
            }
        }
        else {
            
            if (udata->type_info->cls->free_func) {
                herr_t status;

                
                H5_BEFORE_USER_CB_NOCHECK
                    {
                        status = (udata->type_info->cls->free_func)(info->u.object, H5_REQUEST_NULL);
                    }
                H5_AFTER_USER_CB_NOCHECK
                if (status < 0) {
                    if (udata->force) {
                        
                        mark = true;
                    }
                }
                else
                    
                    mark = true;
            }
            else {
                
                mark = true;
            }
        }

        
        if (mark) {
            
            info->marked = true;

            
            udata->type_info->id_count--;
        }
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5I__destroy_type(H5I_type_t type)
{
    H5I_type_info_t *type_info = NULL;    
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");

    type_info = H5I_type_info_array_g[type];
    if (type_info == NULL || type_info->init_count <= 0)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "invalid type");

    
    if (H5I_clear_type(type, true, false) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTRELEASE, FAIL, "unable to release IDs for type");

    
    if (type_info->cls->flags & H5I_CLASS_IS_APPLICATION)
        type_info->cls = H5MM_xfree_const(type_info->cls);

    HASH_CLEAR(hh, type_info->hash_table);
    type_info->hash_table = NULL;

    type_info = H5MM_xfree(type_info);

    H5I_type_info_array_g[type] = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5I__register(H5I_type_t type, const void *object, bool app_ref, H5I_future_realize_func_t realize_cb,
              H5I_future_discard_func_t discard_cb)
{
    H5I_type_info_t *type_info = NULL;            
    H5I_id_info_t   *info      = NULL;            
    hid_t            new_id    = H5I_INVALID_HID; 
    hid_t            ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, H5I_INVALID_HID, "invalid type number");
    type_info = H5I_type_info_array_g[type];
    if ((NULL == type_info) || (type_info->init_count <= 0))
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, H5I_INVALID_HID, "invalid type");
    if (NULL == (info = H5FL_CALLOC(H5I_id_info_t)))
        HGOTO_ERROR(H5E_ID, H5E_NOSPACE, H5I_INVALID_HID, "memory allocation failed");

    
    new_id           = H5I_MAKE(type, type_info->nextid);
    info->id         = new_id;
    info->count      = 1; 
    info->app_count  = !!app_ref;
    info->u.c_object = object;
    info->is_future  = (NULL != realize_cb);
    info->realize_cb = realize_cb;
    info->discard_cb = discard_cb;
    info->marked     = false;

    
    HASH_ADD(hh, type_info->hash_table, id, sizeof(hid_t), info);
    type_info->id_count++;
    type_info->nextid++;

    
    assert(type_info->nextid <= ID_MASK);

    
    type_info->last_id_info = info;

    
    ret_value = new_id;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5I_register(H5I_type_t type, const void *object, bool app_ref)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    assert(type >= H5I_FILE && type < H5I_NTYPES);
    assert(object);

    
    if (H5I_INVALID_HID == (ret_value = H5I__register(type, object, app_ref, NULL, NULL)))
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5I_register_using_existing_id(H5I_type_t type, void *object, bool app_ref, hid_t existing_id)
{
    H5I_type_info_t *type_info = NULL;    
    H5I_id_info_t   *info      = NULL;    
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(object);

    
    if (NULL != (info = H5I__find_id(existing_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADRANGE, FAIL, "ID already in use");

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");

    
    type_info = H5I_type_info_array_g[type];

    if (NULL == type_info || type_info->init_count <= 0)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "invalid type");

    
    if (H5I_TYPE(existing_id) != type)
        HGOTO_ERROR(H5E_ID, H5E_BADRANGE, FAIL, "invalid type for provided ID");

    
    if (NULL == (info = H5FL_CALLOC(H5I_id_info_t)))
        HGOTO_ERROR(H5E_ID, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    info->id        = existing_id;
    info->count     = 1; 
    info->app_count = !!app_ref;
    info->u.object  = object;
    
    info->is_future  = false;
    info->realize_cb = NULL;
    info->discard_cb = NULL;
    info->marked     = false;

    
    HASH_ADD(hh, type_info->hash_table, id, sizeof(hid_t), info);
    type_info->id_count++;

    
    type_info->last_id_info = info;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5I_subst(hid_t id, const void *new_object)
{
    H5I_id_info_t *info      = NULL; 
    void          *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    if (NULL == (info = H5I__find_id(id)))
        HGOTO_ERROR(H5E_ID, H5E_NOTFOUND, NULL, "can't get ID ref count");

    
    ret_value = info->u.object;

    
    info->u.c_object = new_object;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5I_object(hid_t id)
{
    H5I_id_info_t *info      = NULL; 
    void          *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOERR

    
    if (NULL != (info = H5I__find_id(id)))
        
        ret_value = info->u.object;

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5I_object_verify(hid_t id, H5I_type_t type)
{
    H5I_id_info_t *info      = NULL; 
    void          *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOERR

    assert(type >= 1 && (int)type < H5I_next_type_g);

    
    if (type == H5I_TYPE(id) && NULL != (info = H5I__find_id(id)))
        
        ret_value = info->u.object;

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5I_type_t
H5I_get_type(hid_t id)
{
    H5I_type_t ret_value = H5I_BADID; 

    FUNC_ENTER_NOAPI_NOERR

    if (id > 0)
        ret_value = H5I_TYPE(id);

    assert(ret_value >= H5I_BADID && (int)ret_value < H5I_next_type_g);

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5I_is_file_object(hid_t id)
{
    H5I_type_t type      = H5I_get_type(id);
    htri_t     ret_value = FAIL;

    FUNC_ENTER_NOAPI(FAIL)

    
    if (type < 1 || type >= H5I_NTYPES)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "ID type out of range");

    
    if (H5I_DATASET == type || H5I_GROUP == type || H5I_MAP == type)
        ret_value = true;
    else if (H5I_DATATYPE == type) {

        H5T_t *dt = NULL;

        if (NULL == (dt = (H5T_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unable to get underlying datatype struct");

        ret_value = H5T_is_named(dt);
    }
    else
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5I__remove_verify(hid_t id, H5I_type_t type)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    

    
    if (type == H5I_TYPE(id))
        ret_value = H5I_remove(id);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5I__remove_common(H5I_type_info_t *type_info, hid_t id)
{
    H5I_id_info_t *info      = NULL; 
    void          *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(type_info);

    
    HASH_FIND(hh, type_info->hash_table, &id, sizeof(hid_t), info);
    if (info) {
        assert(!info->marked);
        if (!H5I_marking_s)
            HASH_DELETE(hh, type_info->hash_table, info);
        else
            info->marked = true;
    }
    else
        HGOTO_ERROR(H5E_ID, H5E_CANTDELETE, NULL, "can't remove ID node from hash table");

    
    if (type_info->last_id_info == info)
        type_info->last_id_info = NULL;

    ret_value = info->u.object;

    if (!H5I_marking_s)
        info = H5FL_FREE(H5I_id_info_t, info);

    
    (type_info->id_count)--;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5I_remove(hid_t id)
{
    H5I_type_info_t *type_info = NULL;      
    H5I_type_t       type      = H5I_BADID; 
    void            *ret_value = NULL;      

    FUNC_ENTER_NOAPI(NULL)

    
    type = H5I_TYPE(id);
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, NULL, "invalid type number");
    type_info = H5I_type_info_array_g[type];
    if (type_info == NULL || type_info->init_count <= 0)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, NULL, "invalid type");

    
    if (NULL == (ret_value = H5I__remove_common(type_info, id)))
        HGOTO_ERROR(H5E_ID, H5E_CANTDELETE, NULL, "can't remove ID node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5I__dec_ref(hid_t id, void **request)
{
    H5I_id_info_t *info      = NULL; 
    int            ret_value = 0;    

    FUNC_ENTER_PACKAGE

    
    assert(id >= 0);

    
    if (NULL == (info = H5I__find_id(id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "can't locate ID");

    
    if (1 == info->count) {
        H5I_type_info_t *type_info; 
        bool             remove_node = false;

        
        type_info = H5I_type_info_array_g[H5I_TYPE(id)];

        if (type_info->cls->free_func) {
            herr_t status;

            
            H5_BEFORE_USER_CB((-1))
                {
                    status = (type_info->cls->free_func)(info->u.object, request);
                }
            H5_AFTER_USER_CB((-1))

            if (status >= 0)
                remove_node = true;
        }
        else
            remove_node = true;

        if (remove_node) {
            
            if (NULL == H5I__remove_common(type_info, id))
                HGOTO_ERROR(H5E_ID, H5E_CANTDELETE, (-1), "can't remove ID node");
        } 
        else
            ret_value = -1;
    } 
    else {
        --(info->count);
        ret_value = (int)info->count;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_dec_ref(hid_t id)
{
    int ret_value = 0; 

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if ((ret_value = H5I__dec_ref(id, H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't decrement ID ref count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5I__dec_app_ref(hid_t id, void **request)
{
    int ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    assert(id >= 0);

    
    if ((ret_value = H5I__dec_ref(id, request)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't decrement ID ref count");

    
    if (ret_value > 0) {
        H5I_id_info_t *info = NULL; 

        
        if (NULL == (info = H5I__find_id(id)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "can't locate ID");

        
        --(info->app_count);
        assert(info->count >= info->app_count);

        
        ret_value = (int)info->app_count;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_dec_app_ref(hid_t id)
{
    int ret_value = 0; 

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if ((ret_value = H5I__dec_app_ref(id, H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't decrement ID ref count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_dec_app_ref_async(hid_t id, void **token)
{
    int ret_value = 0; 

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if ((ret_value = H5I__dec_app_ref(id, token)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't asynchronously decrement ID ref count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5I__dec_app_ref_always_close(hid_t id, void **request)
{
    int ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    assert(id >= 0);

    
    ret_value = H5I__dec_app_ref(id, request);

    
    if (ret_value < 0) {
        
        H5I_remove(id);

        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't decrement ID ref count");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_dec_app_ref_always_close(hid_t id)
{
    int ret_value = 0; 

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if ((ret_value = H5I__dec_app_ref_always_close(id, H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't decrement ID ref count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_dec_app_ref_always_close_async(hid_t id, void **token)
{
    int ret_value = 0; 

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if ((ret_value = H5I__dec_app_ref_always_close(id, token)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTDEC, (-1), "can't asynchronously decrement ID ref count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_inc_ref(hid_t id, bool app_ref)
{
    H5I_id_info_t *info      = NULL; 
    int            ret_value = 0;    

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if (NULL == (info = H5I__find_id(id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "can't locate ID");

    
    ++(info->count);
    if (app_ref)
        ++(info->app_count);

    
    ret_value = (int)(app_ref ? info->app_count : info->count);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_get_ref(hid_t id, bool app_ref)
{
    H5I_id_info_t *info      = NULL; 
    int            ret_value = 0;    

    FUNC_ENTER_NOAPI((-1))

    
    assert(id >= 0);

    
    if (NULL == (info = H5I__find_id(id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, (-1), "can't locate ID");

    
    ret_value = (int)(app_ref ? info->app_count : info->count);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I__inc_type_ref(H5I_type_t type)
{
    H5I_type_info_t *type_info = NULL; 
    int              ret_value = -1;   

    FUNC_ENTER_PACKAGE

    
    assert(type > 0 && (int)type < H5I_next_type_g);

    
    type_info = H5I_type_info_array_g[type];
    if (NULL == type_info)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, (-1), "invalid type");

    
    ret_value = (int)(++(type_info->init_count));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I_dec_type_ref(H5I_type_t type)
{
    H5I_type_info_t *type_info = NULL; 
    herr_t           ret_value = 0;    

    FUNC_ENTER_NOAPI((-1))

    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, (-1), "invalid type number");

    type_info = H5I_type_info_array_g[type];
    if (type_info == NULL || type_info->init_count <= 0)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, (-1), "invalid type");

    
    if (1 == type_info->init_count) {
        H5I__destroy_type(type);
        ret_value = 0;
    }
    else {
        --(type_info->init_count);
        ret_value = (herr_t)type_info->init_count;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5I__get_type_ref(H5I_type_t type)
{
    H5I_type_info_t *type_info = NULL; 
    int              ret_value = -1;   

    FUNC_ENTER_PACKAGE

    
    assert(type >= 0);

    
    type_info = H5I_type_info_array_g[type];
    if (!type_info)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, (-1), "invalid type");

    
    ret_value = (int)type_info->init_count;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5I__iterate_cb(void *_item, void H5_ATTR_UNUSED *_key, void *_udata)
{
    H5I_id_info_t    *info      = (H5I_id_info_t *)_item;     
    H5I_iterate_ud_t *udata     = (H5I_iterate_ud_t *)_udata; 
    int               ret_value = H5_ITER_CONT;               

    FUNC_ENTER_PACKAGE_NOERR

    
    if ((!udata->app_ref) || (info->app_count > 0)) {
        H5I_type_t type = udata->obj_type;
        void      *object;
        herr_t     cb_ret_val = FAIL;

        
        object = H5I__unwrap(info->u.object, type);

        
        
        H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
            {
                cb_ret_val = (*udata->user_func)((void *)object, info->id, udata->user_udata);
            }
        H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)

        
        if (cb_ret_val > 0)
            ret_value = H5_ITER_STOP; 
        else if (cb_ret_val < 0)
            ret_value = H5_ITER_ERROR; 
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5I_iterate(H5I_type_t type, H5I_search_func_t func, void *udata, bool app_ref)
{
    H5I_type_info_t *type_info = NULL;    
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "invalid type number");
    type_info = H5I_type_info_array_g[type];

    
    if (type_info && type_info->init_count > 0 && type_info->id_count > 0) {
        H5I_iterate_ud_t iter_udata; 
        H5I_id_info_t   *item = NULL;
        H5I_id_info_t   *tmp  = NULL;

        
        iter_udata.user_func  = func;
        iter_udata.user_udata = udata;
        iter_udata.app_ref    = app_ref;
        iter_udata.obj_type   = type;

        
        HASH_ITER(hh, type_info->hash_table, item, tmp)
        {
            if (!item->marked) {
                int ret = H5I__iterate_cb((void *)item, NULL, (void *)&iter_udata);
                if (H5_ITER_ERROR == ret)
                    HGOTO_ERROR(H5E_ID, H5E_BADITER, FAIL, "iteration failed");
                if (H5_ITER_STOP == ret)
                    break;
            }
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5I_id_info_t *
H5I__find_id(hid_t id)
{
    H5I_type_t       type;             
    H5I_type_info_t *type_info = NULL; 
    H5I_id_info_t   *id_info   = NULL; 
    H5I_id_info_t   *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    type = H5I_TYPE(id);
    if (type <= H5I_BADID || (int)type >= H5I_next_type_g)
        HGOTO_DONE(NULL);
    type_info = H5I_type_info_array_g[type];
    if (!type_info || type_info->init_count <= 0)
        HGOTO_DONE(NULL);

    
    if (type_info->last_id_info && type_info->last_id_info->id == id)
        id_info = type_info->last_id_info;
    else {
        HASH_FIND(hh, type_info->hash_table, &id, sizeof(hid_t), id_info);

        
        type_info->last_id_info = id_info;
    }

    
    if (id_info && id_info->is_future) {
        hid_t  actual_id = H5I_INVALID_HID; 
        void  *future_object;               
        void  *actual_object;               
        herr_t status = FAIL;

        
        H5_BEFORE_USER_CB_NOERR(NULL)
            {
                
                status = (id_info->realize_cb)(id_info->u.object, &actual_id);
            }
        H5_AFTER_USER_CB_NOERR(NULL)
        if (status < 0)
            HGOTO_DONE(NULL);

        
        if (H5I_INVALID_HID == actual_id)
            HGOTO_DONE(NULL);
        if (H5I_TYPE(id) != H5I_TYPE(actual_id))
            HGOTO_DONE(NULL);

        
        future_object = id_info->u.object;
        actual_object = H5I__remove_common(type_info, actual_id);
        assert(actual_object);
        id_info->u.object = actual_object;

        
        H5_BEFORE_USER_CB_NOERR(NULL)
            {
                
                status = (id_info->discard_cb)(future_object);
            }
        H5_AFTER_USER_CB_NOERR(NULL)
        if (status < 0)
            HGOTO_DONE(NULL);
        future_object = NULL;

        
        id_info->is_future  = false;
        id_info->realize_cb = NULL;
        id_info->discard_cb = NULL;
    }

    
    ret_value = id_info;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5I__find_id_cb(void *_item, void H5_ATTR_UNUSED *_key, void *_udata)
{
    H5I_id_info_t   *info      = (H5I_id_info_t *)_item;    
    H5I_get_id_ud_t *udata     = (H5I_get_id_ud_t *)_udata; 
    H5I_type_t       type      = udata->obj_type;
    const void      *object    = NULL;
    int              ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(info);
    assert(udata);

    
    object = H5I__unwrap(info->u.object, type);

    
    if (object == udata->object) {
        udata->ret_id = info->id;
        ret_value     = H5_ITER_STOP;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5I_find_id(const void *object, H5I_type_t type, hid_t *id)
{
    H5I_type_info_t *type_info = NULL;    
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(id);

    *id = H5I_INVALID_HID;

    type_info = H5I_type_info_array_g[type];
    if (!type_info || type_info->init_count <= 0)
        HGOTO_ERROR(H5E_ID, H5E_BADGROUP, FAIL, "invalid type");

    
    if (type_info->init_count > 0 && type_info->id_count > 0) {
        H5I_get_id_ud_t udata; 
        H5I_id_info_t  *item = NULL;
        H5I_id_info_t  *tmp  = NULL;

        
        udata.object   = object;
        udata.obj_type = type;
        udata.ret_id   = H5I_INVALID_HID;

        
        HASH_ITER(hh, type_info->hash_table, item, tmp)
        {
            int ret = H5I__find_id_cb((void *)item, NULL, (void *)&udata);
            if (H5_ITER_ERROR == ret)
                HGOTO_ERROR(H5E_ID, H5E_BADITER, FAIL, "iteration failed");
            if (H5_ITER_STOP == ret)
                break;
        }

        *id = udata.ret_id;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
