/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HLmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5HLpkg.h"     

H5FL_DEFINE_STATIC(H5HL_t);

H5HL_t *
H5HL__new(size_t sizeof_size, size_t sizeof_addr, size_t prfx_size)
{
    H5HL_t *heap      = NULL; 
    H5HL_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(sizeof_size > 0);
    assert(sizeof_addr > 0);
    assert(prfx_size > 0);

    
    if (NULL == (heap = H5FL_CALLOC(H5HL_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    heap->sizeof_size = sizeof_size;
    heap->sizeof_addr = sizeof_addr;
    heap->prfx_size   = prfx_size;

    
    ret_value = heap;

done:
    if (!ret_value && heap != NULL)
        if (NULL == (heap = H5FL_FREE(H5HL_t, heap)))
            HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, NULL, "can't free heap memory");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL__inc_rc(H5HL_t *heap)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(heap);

    
    heap->rc++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HL__dec_rc(H5HL_t *heap)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(heap);

    
    heap->rc--;

    
    if (heap->rc == 0 && FAIL == H5HL__dest(heap))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy local heap");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL__dest(H5HL_t *heap)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(heap);

    
    assert(heap->prots == 0);
    assert(heap->rc == 0);
    assert(heap->prfx == NULL);
    assert(heap->dblk == NULL);

    
    if (heap->dblk_image)
        if (NULL != (heap->dblk_image = H5FL_BLK_FREE(lheap_chunk, heap->dblk_image)))
            HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free local heap data block image");
    while (heap->freelist) {
        H5HL_free_t *fl;

        fl             = heap->freelist;
        heap->freelist = fl->next;
        if (NULL != (fl = H5FL_FREE(H5HL_free_t, fl)))
            HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free local heap free list");
    }

    if (NULL != (heap = H5FL_FREE(H5HL_t, heap)))
        HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free local heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 
