% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_fairness_metrics.R
\name{get_fairness_metrics}
\alias{get_fairness_metrics}
\title{Compute Fairness Metrics for Binary Classification}
\usage{
get_fairness_metrics(
  data,
  outcome,
  group,
  probs,
  confint = TRUE,
  cutoff = 0.5,
  bootstraps = 2500,
  alpha = 0.05,
  digits = 2
)
}
\arguments{
\item{data}{A data frame containing the outcome, group, and predicted probabilities.}

\item{outcome}{The name of the column containing the true binary outcome.}

\item{group}{The name of the column representing the binary protected attribute (e.g., race, gender).}

\item{probs}{The name of the column with predicted probabilities.}

\item{confint}{Logical indicating whether to calculate confidence intervals.}

\item{cutoff}{Numeric threshold for classification. Default is 0.5.}

\item{bootstraps}{Number of bootstrap samples. Default is 2500.}

\item{alpha}{Significance level for confidence intervals. Default is 0.05.}

\item{digits}{Number of digits to round the metrics to. Default is 2.}
}
\value{
A dataframe containing fairness assessments, the performance metrics they use and the evaluated results for each (binary) group (specified by the \code{group} parameter) along with the difference and ratio between them. If \code{confint} is  set to \code{TRUE}, then the estimated \verb{(1-alpha)*100\%} bootstrap confidence intervals are returned as well.
}
\description{
Computes a comprehensive set of fairness metrics for binary classification models, disaggregated by a binary protected attribute. The function also computes corresponding performance metrics used in the fairness calculations.
}
\details{
The results are returned as a list of two data frames:
\itemize{
\item \code{performance}: Contains performance metrics (e.g., TPR, FPR, PPV) by group.
\item \code{fairness}: Contains group-level fairness metrics (e.g., disparities or ratios), confidence intervals (if specified).
}
\subsection{Fairness Metrics Included:}{
\itemize{
\item \strong{Statistical Parity}: Difference in positive prediction rates across groups.
\item \strong{Equal Opportunity}: Difference in true positive rates (TPR) across groups.
\item \strong{Predictive Equality}: Difference in false positive rates (FPR) across groups.
\item \strong{Balance for Positive Class}: Checks whether the predicted probability distributions for
positive outcomes are similar across groups.
\item \strong{Balance for Negative Class}: Same as above, but for negative outcomes.
\item \strong{Positive Predictive Parity}: Difference in positive predictive values (precision) across groups.
\item \strong{Negative Predictive Parity}: Difference in negative predictive values across groups.
\item \strong{Brier Score Parity}: Difference in Brier scores across groups.
\item \strong{Overall Accuracy Parity}: Difference in overall accuracy across groups.
\item \strong{Treatment Equality}: Ratio of false negatives to false positives across groups.
}

\strong{NOTE:} Statistical inference from bootstrapped confidence intervals should be interpreted with caution. A confidence interval crossing 0 (for differences) or 1 (for ratios) means the evidence is inconclusive rather than proving absence of unfairness. Apparent violations may reflect sampling variability rather than systematic bias. Always complement these results with domain knowledge, sensitivity analyses, and additional fairness diagnostics before drawing strong conclusions about a specific fairness assessment.
}
}
\examples{
\donttest{
library(fairmetrics)
library(dplyr)
library(randomForest)
library(magrittr)
data("mimic_preprocessed")
set.seed(123)
train_data <- mimic_preprocessed \%>\%
  dplyr::filter(dplyr::row_number() <= 700)
# Fit a random forest model
rf_model <- randomForest::randomForest(factor(day_28_flg) ~ ., data = train_data, ntree = 1000)
# Test the model on the remaining data
test_data <- mimic_preprocessed \%>\%
  dplyr::mutate(gender = ifelse(gender_num == 1, "Male", "Female"))\%>\%
  dplyr::filter(dplyr::row_number() > 700)

test_data$pred <- predict(rf_model, newdata = test_data, type = "prob")[, 2]

# Fairness evaluation
# We will use sex as the protected attribute and day_28_flg as the outcome.
# We choose threshold = 0.41 so that the overall FPR is around 5\%.

# Get Fairness Metrics
get_fairness_metrics(
 data = test_data,
 outcome = "day_28_flg",
 group = "gender",
 probs = "pred",
 confint = TRUE,
 cutoff = 0.41,
 alpha = 0.05
)
}


}
