% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lencode.R
\name{step_lencode}
\alias{step_lencode}
\alias{tidy.step_lencode}
\title{Likelihood encoding using analytical formula}
\usage{
step_lencode(
  recipe,
  ...,
  role = NA,
  trained = FALSE,
  outcome = NULL,
  smooth = TRUE,
  mapping = NULL,
  skip = FALSE,
  id = rand_id("lencode")
)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the sequence of
operations for this recipe.}

\item{...}{One or more selector functions to choose variables. For
\code{step_lencode()}, this indicates the variables to be encoded into a
numeric format. See \code{\link[recipes:selections]{recipes::selections()}} for more details. For the
\code{tidy} method, these are not currently used.}

\item{role}{Not used by this step since no new variables are created.}

\item{trained}{A logical to indicate if the quantities for preprocessing have
been estimated.}

\item{outcome}{A call to \code{vars} to specify which variable is used as the
outcome. Only numeric and two-level factors are currently supported.}

\item{smooth}{A logical, default to \code{TRUE}, should the estimates of groups
with low counts be pulled towards the gobal estimate? Defaults to \code{TRUE}.
See Details for how this is done. This is also known as partial pooling or
shrinkage. Only works for numeric outcomes.}

\item{mapping}{A list of tibble results that define the encoding. This is
\code{NULL} until the step is trained by \code{\link[recipes:prep]{recipes::prep()}}.}

\item{skip}{A logical. Should the step be skipped when the recipe is baked by
\code{\link[recipes:bake]{recipes::bake()}}? While all operations are baked when \code{\link[recipes:prep]{recipes::prep()}} is
run, some operations may not be able to be conducted on new data (e.g.
processing the outcome variable(s)). Care should be taken when using \code{skip = TRUE} as it may affect the computations for subsequent operations}

\item{id}{A character string that is unique to this step to identify it.}
}
\value{
An updated version of \code{recipe} with the new step added to the
sequence of existing steps (if any). For the \code{tidy} method, a tibble with
columns \code{terms} (the selectors or variables for encoding), \code{level} (the
factor levels), and \code{value} (the encodings).
}
\description{
\code{step_lencode()} creates a \emph{specification} of a recipe step that will convert
a nominal (i.e. factor) predictor into a single set of scores derived
analytically.
}
\details{
Each selected nominal predictor will be replaced by a numeric predictor.
Each unique value of the nominal predictor is replaced by a numeric value.
Thse values are calculated differently depending on the type of the outcome.

For \strong{numeric} outcomes each value is the average value of the outcome
inside each of the levels of the predictor. Unseen levels of the predictor
will be using the global mean of the predictor.
If case weights are used then a weighted mean is calculated instead.

For \strong{nominal} outcomes each value is the log odds of the of the first level
of the outcome variable being present, within each level of the levels of the
predictor. Unseen levels will be replaced by the global log odds without
stratification.
If case weights are used then a weighted log odds is calculated.

If no or all occurances happens then the log odds is calculated using
\code{p = (2 * nrow(data) - 1) / (2 * nrow(data))} to avoid infinity that would
happen by taking the log of \code{0}.

For numeric outcomes where \code{smooth = TRUE}, the following adjustment is done.

\deqn{
estimate = (n / global_{var}) /
(n / global_{var} + 1 / outcome_{var}) *
 estimate +
 (1 / outcome_{var}) / (n / global_{var} + 1 / outcome_{var}) * global_{mean}
}

Where \eqn{n} is the number of observations in the group.
}
\section{Tidying}{
When you \code{\link[recipes:tidy.recipe]{tidy()}} this step, a tibble is returned
with columns \code{level}, \code{value}, \code{terms}, and \code{id}:

\describe{
\item{level}{character, the factor levels}
\item{value}{numeric, the encoding}
\item{terms}{character, the selectors or variables selected}
\item{id}{character, id of this step}
}
}

\section{Case weights}{


This step performs an supervised operation that can utilize case weights.
To use them, see the documentation in \link[recipes:case_weights]{recipes::case_weights} and the examples on
\code{tidymodels.org}.
}

\examples{
\dontshow{if (rlang::is_installed("modeldata")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(recipes)
library(dplyr)
library(modeldata)

data(grants)

set.seed(1)
grants_other <- sample_n(grants_other, 500)
reencoded <- recipe(class ~ sponsor_code, data = grants_other) |>
  step_lencode(sponsor_code, outcome = vars(class), smooth = FALSE) |>
  prep()

bake(reencoded, grants_other)

tidy(reencoded, 1)
\dontshow{\}) # examplesIf}
}
\references{
Micci-Barreca D (2001) "A preprocessing scheme for high-cardinality
categorical attributes in classification and prediction problems," ACM SIGKDD
Explorations Newsletter, 3(1), 27-32.

Zumel N and Mount J (2017) "vtreat: a data.frame Processor for Predictive
Modeling," arXiv:1611.09477
}
\concept{preprocessing encoding}
\keyword{datagen}
