% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectoryRMA.R
\encoding{UTF-8}
\name{trajectoryRMA}
\alias{trajectoryRMA}
\alias{RTMA}
\title{Relative Trajectory Movement Assessment (RTMA)}
\usage{
trajectoryRMA(x, alpha = 0.05, nperm = 999, full.output = TRUE, add = TRUE)
}
\arguments{
\item{x}{An object of class \code{\link{trajectories}}.}

\item{alpha}{The alpha level for the tests performed in RTMA. Defaults to \code{0.05}.}

\item{nperm}{Passed to function \code{\link{trajectoryCorrespondence}}. The number of permutations to be used in the dynamic correspondence test. Defaults to \code{999}.}

\item{full.output}{Flag to indicate that the full output of tests should be computed. Defaults to \code{TRUE}. Setting to \code{FALSE} will improve computation speed but yield incomplete outputs (see details).}

\item{add}{Passed to function \code{\link{trajectoryConvergence}}. Flag to indicate that constant values should be added (local transformation) to correct triplets of distance values that do not fulfill the triangle inequality.}
}
\value{
Function \code{trajectoryRMA} returns an object of classes \code{\link{list}} and \code{RTMA} containing:
\itemize{
\item{\code{dynamic_relationships_taxonomy}: a data-frame containing the names of the relative movement relationships recognized by RTMA as well as corresponding relationship groups. This part of \code{trajectoryRMA} output is independent of the trajectories used as input and is primarily a bestiary (see details). It can be used to transform the \code{dynamic_relationships} matrix (see below) to focus on chosen relationship groups. }
\item{\code{dynamic_relationships}: a matrix containing the relative movement relationships attributed to each pair of trajectories.}
\item{\code{symmetric_convergence}: a list containing the results of the symmetric convergence test.}
\item{\code{asymmetric_convergence}: a list containing the results of the two asymmetric convergence tests.}
\item{\code{correspondence}: a matrix containing the results of the the dynamic correspondence tests (partial if \code{full.out = FALSE}).}
\item{\code{parameters}: a vector containing the parameters \code{alpha}, the \enc{Šidák}{Sidak} \code{corrected_alpha}, and \code{nperm}}.
}
In addition to the relationships recognized by RTMA, matrix \code{dynamic_relationships} provides the role of each trajectory in asymmetric relationships.
The role is provided in parenthesis and applies to the trajectory of the ROW index. For example, \code{"approaching (approacher)"} means
that the trajectory of the corresponding row is approaching the trajectory of the corresponding column, which will have \code{"approaching (target)"}.
In symmetric relationships, the wording \code{(symmetric)} is added to indicate that there is no distinction of roles.
}
\description{
Relative Trajectory Movement Assessment (RTMA) is a method for testing and qualifying of the relative movements of ecological trajectories (e.g. "convergence", "parallel" etc., see details) as described in Djeghri et al. (in prep). It is implemented in function \code{trajectoryRMA()}.
}
\details{
Function \code{trajectoryRMA} attributes a dynamic relationship to pairs of ecological trajectories A and B describing their relative movement. It does so by combining four tests:
\itemize{
\item{Three convergence tests performed through internal callings of function \code{\link{trajectoryConvergence}}:
\itemize{
\item{The symmetric convergence test between trajectories A and B.}
\item{The asymmetric convergence test assessing if trajectory A approaches trajectory B.}
\item{The asymmetric convergence test assessing if trajectory B approaches trajectory A.}
}
}
\item{One dynamic correspondence test performed through internal callings to function \code{\link{trajectoryCorrespondence}}.}
}
To account for multiple testing, \code{trajectoryRMA} performs internally a \enc{Šidák}{Sidak} (1967) correction on the alpha level provided in parameter \code{alpha}.

The results of the four tests (p-values and sign of statistic) are used to assign to each trajectory pair a relationship describing their relative movements. RTMA recognizes a total of 10 relationships, some existing in "weak" variations.
The following five dynamic relationships are \emph{symmetric}, i.e. applying to the two trajectories without distinction of roles:
\itemize{
\item{\code{"convergence"} - The two trajectories converge. Exists in a weak version.}
\item{\code{"divergence"} - The two trajectories diverge. Exists in a weak version.}
\item{\code{"parallel"} - The two trajectories travel side by side with broadly similar movements.}
\item{\code{"antiparallel"} - As in \code{"parallel"} but the two trajectories travel in opposite directions.}
\item{\code{"neutral"} - The two trajectories have no particular movements relative to each other (effectively the null hypothesis for RTMA).}
}
The following five dynamic relationships are \emph{asymmetric} (e.g. in \code{"pursuit"} there is a leading and a following trajectory). In these asymmetric relationships the output of function \code{trajectoryRMA} gives the role of each trajectory (see Value section). A more general interpretation of asymmetry is to consider that the relationship is \emph{oriented} (see below, relationship groups).
\itemize{
\item{\code{"approaching"} - One trajectory approaches the other. Exists in a weak version.}
\item{\code{"departing"} - One trajectory moves away from the other. Exists in a weak version.}
\item{\code{"pursuit"} - The two trajectories follow each other.}
\item{\code{"catch-up"} - As in \code{"pursuit"} but the following trajectory moves faster.}
\item{\code{"escape"} - As in \code{"pursuit"} but the leading trajectory is faster.}
}

In rare cases, unlikely relationships (labelled \code{"unknown"}, with a short description in brackets) may occur. These involve contradictory patterns hard to interpret.

RELATIONSHIP GROUPS: It is possible to further sort trajectory relationships in broad \emph{relationship groups} (not always mutually exclusive). Three such groups are recognized in RTMA:
\itemize{
\item{The \code{"convergence group"}, includes relationships that display convergence in the broadest sense with a trend of diminishing distance between the two trajectories. Formally this group includes relationships of
\code{"convergence"} and \code{"approaching"} and their weak versions, as well as \code{"catch-up"}.}
\item{The \code{"divergence group"}, includes relationships that display divergence in the broadest sense with a trend of increasing distance between the two trajectories. Formally this group includes relationships of
\code{"divergence"} and  \code{"departing"} and their weak versions, as well as \code{"escape"}.}
\item{The \code{"oriented group"}, includes relationships that have, broadly speaking, a trajectory \emph{in front} and a trajectory \emph{in the back} implying an orientation to their relationship. This group includes all asymmetric relationships, formally:
\code{"approaching"} and \code{"departing"} and their weak versions, \code{"catch-up"}, \code{"escape"} and \code{"pursuit"}.}
}
Note that a given relationship may belong to two groups (either convergence or divergence group + oriented group) and that \code{"parallel"},\code{"antiparallel"} and \code{"neutral"} relationships stand on their own, not belonging to any groups.
In our experience, relationship groups have proven a useful conceptual tool to reveal large scale patterns particularly when adressing many trajectory relationships (see Djeghri et al. in prep).

LIMITATIONS: RTMA has some limitations, in particular it uses trend tests not well suited to study trajectories pairs with changing relative movements (e.g. if two trajectories cross each other, they are first converging then diverging).
We advise users to not only rely on RTMA but to also visualize trajectories using function \code{\link{trajectoryPCoA}} for ecological interpretations. See Djeghri et al. (in prep) for more details.
Note also that, because RTMA incorporates a correction for multiple testing, it needs somewhat long trajectories to operate (minimum number of survey = 6 at alpha = 0.05).

COMPUTATION TIME: The dynamic correspondence tests performed in RTMA are computationally costly permutation tests only used when all three convergence tests are non-significant.
Function \code{trajectoryRMA} performs by default all tests but it is possible to only perform the tests useful for RTMA by setting \code{full.output = FALSE}.
This will reduce computation time but the details of the output of RTMA will not contain the information on all possible dynamic correspondence tests, only on relevant ones.

PLOTTING: Functions \code{\link{trajectoryConvergencePlot}} and \code{\link{trajectoryRMAPlot}} provide options to plot the results of RTMA.
}
\examples{
#Obtain and format some trajectories
data("avoca")
avoca_D_man <- vegclust::vegdiststruct(avoca_strat, 
                                       method ="manhattan", 
                                       transform = function(x){log(x+1)})
years <- c(1971, 1974, 1978, 1983, 1987, 1993, 1999, 2004, 2009)
avoca_times <- years[avoca_surveys]
avoca_x <- defineTrajectories(d = avoca_D_man,  
                              sites = avoca_sites, 
                              times = avoca_times)
                              
#Visualize the trajectories
trajectoryPCoA(avoca_x,traj.colors = RColorBrewer::brewer.pal(8,"Accent"),length=0.1,lwd=2)
legend("bottomleft",bty="n",legend=1:8,col=RColorBrewer::brewer.pal(8,"Accent"),lwd=2,ncol=2)

#Perform RTMA
trajectoryRMA(avoca_x)


}
\references{
Djeghri et al. (in preparation) Uncovering the relative movements of ecological trajectories.

\enc{Šidák}{Sidak}, Z. (1967) Rectangular confidence regions for the means of multivariate normal distributions. Journal of the American Statistical Association 62:648-633.
}
\seealso{
\code{\link{trajectoryConvergence}}, \code{\link{trajectoryCorrespondence}}, \code{\link{trajectoryConvergencePlot}}, \code{\link{trajectoryRMAPlot}}
}
\author{
Nicolas Djeghri, UBO

Miquel De \enc{Cáceres}{Caceres}, CREAF
}
