% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_qgraph.R
\name{plot_qgraph}
\alias{plot_qgraph}
\title{Plot a Network Graph from a Distance Matrix}
\usage{
plot_qgraph(
  dist_mat,
  group = NULL,
  max_nodes = 100,
  label_size = 2,
  edge_threshold = 0.1,
  layout = "spring",
  seed = 123,
  main_title = NULL
)
}
\arguments{
\item{dist_mat}{A square distance matrix or a \code{dist} object. Distances are automatically
normalized to [0,1] and converted to similarity via \code{1 - distance}.}

\item{group}{Optional factor or vector indicating group membership for nodes, used for coloring.}

\item{max_nodes}{Integer. Maximum number of nodes to plot. If the number of observations exceeds
this, stratified sampling is performed to reduce the node count.}

\item{label_size}{Numeric. Size of the node labels.}

\item{edge_threshold}{Numeric between 0 and 1. Edges with similarity below this threshold are removed.}

\item{layout}{Character string specifying the layout algorithm for \code{qgraph}. Default is \code{"spring"}.}

\item{seed}{Integer. Random seed used for reproducibility during sampling and layout.}

\item{main_title}{Optional character string specifying the main title of the plot.}
}
\value{
Invisibly returns \code{NULL}. The plot is drawn as a side effect.
}
\description{
This internal function visualizes a network graph representation of a distance matrix,
where nodes represent observations and edges represent similarity. Groups can be specified
for node coloring. A maximum number of nodes can be set to avoid overcrowding, and weak edges
are thresholded.
}
\details{
This function is internal and not intended for direct use. It is called by
\code{\link{visualize_distances}()} to display network graphs of robust distances.

Features:
\itemize{
  \item Converts \code{dist} objects to matrices automatically.
  \item Downsamples nodes if the number of observations exceeds \code{max_nodes}, using stratified sampling by group.
  \item Normalizes the distance matrix to [0,1] and converts it to similarity (1 - distance).
  \item Removes weak edges below \code{edge_threshold}.
  \item Colors nodes according to group membership.
  \item Adds a main title using \code{title()} after plotting with \code{qgraph}.
}
}
\examples{
# --------------------------------------
# Network Graph Example from Robust Distances
# --------------------------------------
data("Data_HC_contamination", package = "dbrobust")
# Subset small dataset
Data_small <- Data_HC_contamination[1:20, ]

cont_vars <- c("V1", "V2", "V3", "V4")
cat_vars  <- c("V5", "V6", "V7")
bin_vars  <- c("V8", "V9")
w <- Data_small$w_loop

# Compute GGower robust distances
dist_sq_ggower <- dbrobust::robust_distances(
  data = Data_small,
  cont_vars = cont_vars,
  bin_vars  = bin_vars,
  cat_vars  = cat_vars,
  w = w,
  alpha = 0.10,
  method = "ggower"
)

# Create factor indicating Normal vs Outlier
n_obs <- nrow(dist_sq_ggower)
group_vec <- rep("Normal", n_obs)
group_vec[attr(dist_sq_ggower, "outlier_idx")] <- "Outlier"
group_factor <- factor(group_vec, levels = c("Normal", "Outlier"))

# Plot network graph (small, for CRAN)
dbrobust::plot_qgraph(
  dist_mat = sqrt(dist_sq_ggower),
  group = group_factor,
  max_nodes = 10,
  label_size = 2,
  edge_threshold = 0.1,
  layout = "spring",
  seed = 123,
  main_title = "GGower Network Graph with Outliers"
)

}
\keyword{internal}
