% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale_process_mirai.R
\name{par_hierarchy_mirai}
\alias{par_hierarchy_mirai}
\title{Parallelize spatial computation by hierarchy in input data}
\usage{
par_hierarchy_mirai(
  regions,
  regions_id = NULL,
  length_left = NULL,
  pad = 0,
  pad_y = FALSE,
  fun_dist,
  ...,
  .debug = TRUE
)
}
\arguments{
\item{regions}{\code{sf}/\code{SpatVector} object.
Computational regions. Only polygons are accepted.}

\item{regions_id}{character(1). Name of unique ID field in \code{regions}.
The regions will be split by the common level value.}

\item{length_left}{integer(1). Length of the first characters of
the \code{regions_id} values. Default is NULL, which will not manipulate
the \code{regions_id} values. If the number of characters is not
consistent (for example, numerics), the function will alert the user.}

\item{pad}{numeric(1). Padding distance for each subregion defined
by \code{regions_id} or trimmed \code{regions_id} values.
in linear unit of coordinate system. Default is 0, which means
each subregion is used as is. If the value is greater than 0,
the subregion will be buffered by the value. The padding distance will
be applied to \code{x} (\code{pad_y = FALSE}) or \code{y} (\code{pad_y = TRUE}) to filter
the data.}

\item{pad_y}{logical(1). Whether to filter y with the padded grid.
Should be TRUE when x is where the values are calculated.
Default is \code{FALSE}. In the reverse case, like \code{terra::extent} or
\code{exactextractr::exact_extract}, the raster (x) should be scoped
with the padded grid.}

\item{fun_dist}{\code{sf}, \code{terra}, or \code{chopin} functions.
This function should have \code{x} and \code{y} arguments.}

\item{...}{Arguments passed to the argument \code{fun_dist}.}

\item{.debug}{logical(1). Default is \code{FALSE}
If a unit computation fails, the error message and the \code{regions_id}
value where the error occurred will be included in the output.}
}
\value{
a data.frame object with computation results.
For entries of the results, consult the function used in
\code{fun_dist} argument.
}
\description{
"Hierarchy" refers to a system,
which divides the entire study region into multiple subregions.
It is usually reflected in an area code system
(e.g., FIPS for US Census geographies and
Nomenclature of Territorial Units for Statistics (NUTS), etc.).
\link[mirai:daemons]{mirai::daemons} will set the parallel backend then \link[mirai:mirai_map]{mirai::mirai_map}
will the work by splitting lower level features into
several higher level feature group. For details of the terminology
in \code{mirai} package, refer to \code{\link[mirai:mirai]{mirai::mirai}}.
Each thread will process the number of lower level features
in each higher level feature. Be advised that
accessing the same file simultaneously with
multiple processes may result in errors.
}
\details{
In dynamic dots (\code{...}), \code{fun_dist} arguments should include
x and y where sf/terra class objects or file paths are accepted.
Hierarchy is interpreted by the \code{regions_id} argument first.
\code{regions_id} is assumed to be a field name in the \code{x} or \code{y} argument
object. It is expected that \code{regions} represents the higher level
boundaries and \code{x} or \code{y} in \code{fun_dist} is the lower level boundaries.
However, if that is not the case, with \code{trim} argument, the function
will generate the higher level codes from \code{regions_id} by extracting
the code from the left end (controlled by \code{length_left}).
Whether \code{x} or \code{y} is searched is determined by \code{pad_y} value.
\code{pad_y = TRUE} will make the function attempt to find \code{regions_id}
in \code{x}, whereas \code{pad_y = FALSE} will look for \code{regions_id} at
\code{y}. If the \code{regions_id} doesn't exist in \code{x} or \code{y}, the function
will utilize spatial relationship (intersects) to filter the data.
Note that dispatching computation by subregions based on the spatial
relationship may lead to a slight discrepancy in the result. For
example, if the higher and lower level features are not perfectly
aligned, there may be some features that are not included or duplicated
in the subregions. The function will alert the user if spatial relation-
ship is used to filter the data.
}
\note{
Virtually any sf/terra functions that accept two arguments
can be put in \code{fun_dist}; however, be advised that
some spatial operations do not necessarily give the
exact result from what would have been done with one thread.
For example, distance calculated through this function may return the
lower value than actual because the computational region was reduced.
This would be the case especially where the target features
are spatially sparsely distributed.
}
\examples{
\donttest{
lastpar <- par(mfrow = c(1, 1))
library(terra)
library(sf)
library(mirai)
options(sf_use_s2 = FALSE)
mirai::daemons(4)

nccnty <- sf::st_read(
  system.file("shape/nc.shp", package = "sf")
)
nccnty <- sf::st_transform(nccnty, "EPSG:5070")

nccntygrid <- sf::st_make_grid(nccnty, n = c(200, 100))
nccntygrid <- sf::st_as_sf(nccntygrid)
nccntygrid$GEOID <- sprintf("\%05d", seq_len(nrow(nccntygrid)))
nccntygrid <- sf::st_intersection(nccntygrid, nccnty)

rrast <- terra::rast(nccnty, nrow = 600, ncol = 1320)
terra::values(rrast) <- rgamma(7.92e5, 4, 2)

# Using raster path
rastpath <- file.path(tempdir(), "ncelev.tif")
terra::writeRaster(rrast, rastpath, overwrite = TRUE)

ncsamp <-
  sf::st_sample(
    nccnty,
    size = 1e4L
  )
# sfc to sf
ncsamp <- sf::st_as_sf(ncsamp)
# assign ID
ncsamp$kid <- sprintf("K-\%05d", seq_len(nrow(ncsamp)))
res <-
  par_hierarchy_mirai(
    regions = nccnty,
    regions_id = "FIPS",
    fun_dist = extract_at,
    y = nccntygrid,
    x = rastpath,
    id = "GEOID",
    func = "mean",
    .debug = TRUE
  )
mirai::daemons(0L)
par(lastpar)
}
}
\seealso{
\code{\link[mirai:mirai_map]{mirai::mirai_map}}, \code{\link[mirai:daemons]{mirai::daemons}}, \code{\link{par_convert_f}}

Other Parallelization: 
\code{\link{par_cut_coords}()},
\code{\link{par_grid}()},
\code{\link{par_grid_mirai}()},
\code{\link{par_hierarchy}()},
\code{\link{par_make_dggrid}()},
\code{\link{par_make_grid}()},
\code{\link{par_make_h3}()},
\code{\link{par_merge_grid}()},
\code{\link{par_multirasters}()},
\code{\link{par_multirasters_mirai}()},
\code{\link{par_pad_balanced}()},
\code{\link{par_pad_grid}()},
\code{\link{par_split_list}()}
}
\author{
Insang Song \email{geoissong@gmail.com}
}
\concept{Parallelization}
