% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aoe.R
\name{aoe}
\alias{aoe}
\title{Classify and Prune Points by Area of Effect}
\usage{
aoe(
  points,
  support = NULL,
  scale = NULL,
  area = NULL,
  method = c("buffer", "stamp"),
  reference = NULL,
  mask = NULL,
  largest_polygon = TRUE,
  coords = NULL
)
}
\arguments{
\item{points}{An \code{sf} object with POINT geometries.}

\item{support}{One of:
\itemize{
\item \code{sf} object with POLYGON/MULTIPOLYGON geometries
\item Country name or ISO code: \code{"France"}, \code{"FR"}, \code{"FRA"}
\item Vector of countries: \code{c("France", "Germany")}
\item Missing: auto-detects countries containing the points
}}

\item{scale}{Numeric scale factor (default \code{sqrt(2) - 1}, approximately 0.414).
Controls the size of the halo relative to the core:
\itemize{
\item \code{sqrt(2) - 1} (default): equal core/halo areas, ratio 1:1
\item \code{1}: area ratio 1:3 (halo is 3x core area)
}

For \code{method = "buffer"}, determines the target halo area as
\code{original_area * ((1 + scale)^2 - 1)}.

For \code{method = "stamp"}, the multiplier \code{1 + scale} is applied to distances
from the reference point.

Cannot be used together with \code{area}.}

\item{area}{Numeric area proportion (alternative to \code{scale}).
Specifies the target halo area as a proportion of the original support area.
For example, \code{area = 1} means halo area equals the original support area.

Unlike \code{scale}, this parameter accounts for masking: the function finds
the scale that produces the target halo area \emph{after} mask intersection.
This is useful when you need a specific effective area regardless of
how much gets clipped by coastlines or borders.

Cannot be used together with \code{scale}.}

\item{method}{Method for computing the area of effect:
\itemize{
\item \code{"buffer"} (default): Uniform buffer around the support boundary.
Robust for any polygon shape. Buffer distance is calculated to achieve
the target halo area.
\item \code{"stamp"}: Scale vertices outward from the centroid (or reference point).
Preserves shape proportions but only guarantees containment for
star-shaped polygons. May leave small gaps for highly concave shapes.
}}

\item{reference}{Optional \code{sf} object with a single POINT geometry.

If \code{NULL} (default), the centroid of each support is used.
Only valid when \code{support} has a single row and \code{method = "stamp"}.}

\item{mask}{Optional mask for clipping the area of effect. Can be:
\itemize{
\item \code{sf} object with POLYGON or MULTIPOLYGON geometry
\item \code{"land"}: use the bundled global land mask to exclude sea areas
If provided, each area of effect is intersected with this mask.
}}

\item{largest_polygon}{Logical (default \code{TRUE}). When the support contains
multiple polygons (e.g., mainland plus islands), use only the largest
polygon by area. This is typically the mainland. Points near dropped
polygons will be pruned entirely (not classified). Set to \code{FALSE} to
include all polygons, in which case \code{area = "equal"} uses total area
with redistribution across all polygons.}

\item{coords}{Column names for coordinates when \code{points} is a data.frame,
e.g. \code{c("lon", "lat")}. If \code{NULL}, auto-detects common names.}
}
\value{
An \code{aoe_result} object (extends \code{sf}) containing only the supported
points, with columns:
\describe{
\item{point_id}{Original point identifier (row name or index)}
\item{support_id}{Identifier for which support the classification refers to}
\item{aoe_class}{Classification: \code{"core"} or \code{"halo"}}
}
When multiple supports are provided, points may appear multiple times
(once per support whose AoE contains them).

The result has S3 methods for \code{print()}, \code{summary()}, and \code{plot()}.
Use \code{aoe_geometry()} to extract the AoE polygons.
}
\description{
Given a set of points and one or more support polygons, \code{aoe()} classifies
points as "core" (inside original support) or "halo" (inside the area of
effect but outside original support), pruning all points outside.
}
\details{
By default, the area of effect is computed using a buffer that produces
equal core and halo areas. This means the AoE has twice the area of the
original support, split evenly between core (inside) and halo (outside).
\subsection{Buffer method (default)}{

Computes a uniform buffer distance \eqn{d} such that the buffered area
equals the target. The buffer distance is found by solving:
\deqn{\pi d^2 + P \cdot d = A_{target}}
where \eqn{P} is the perimeter and \eqn{A_{target}} is the desired halo area.
}

\subsection{Stamp method}{

Applies an affine transformation to each vertex:
\deqn{p' = r + (1 + s)(p - r)}
where \eqn{r} is the reference point (centroid), \eqn{p} is each vertex,
and \eqn{s} is the scale factor. This method preserves shape proportions
but only guarantees the AoE contains the original for star-shaped polygons
(where the centroid can "see" all boundary points).

Points exactly on the original support boundary are classified as "core".

The support geometry is validated internally using \code{\link[sf:valid]{sf::st_make_valid()}}.
}
}
\examples{
library(sf)

# Single support
support <- st_as_sf(
  data.frame(id = 1),
  geometry = st_sfc(st_polygon(list(
    cbind(c(0, 10, 10, 0, 0), c(0, 0, 10, 10, 0))
  ))),
  crs = 32631
)

pts <- st_as_sf(
  data.frame(id = 1:4),
  geometry = st_sfc(
    st_point(c(5, 5)),
    st_point(c(2, 2)),
    st_point(c(15, 5)),
    st_point(c(30, 30))
  ),
  crs = 32631
)

result <- aoe(pts, support)

# Multiple supports (e.g., admin regions)
supports <- st_as_sf(
  data.frame(region = c("A", "B")),
  geometry = st_sfc(
    st_polygon(list(cbind(c(0, 10, 10, 0, 0), c(0, 0, 10, 10, 0)))),
    st_polygon(list(cbind(c(8, 18, 18, 8, 8), c(0, 0, 10, 10, 0))))
  ),
  crs = 32631
)

result <- aoe(pts, supports)
# Points near the boundary may appear in both regions' AoE

}
