% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spp_strategy.R
\name{spp_strategy}
\alias{spp_strategy}
\title{Classify species ecological strategies}
\usage{
spp_strategy(spp_trend_result, sig_level = 0.05, responses = responses)
}
\arguments{
\item{spp_trend_result}{A data frame containing trend indicators per species, typically generated by the \code{spp_trend} function. It should include columns such as:
\itemize{
\item \code{species}: Name of the analyzed species.
\item \code{responses}: Name of the analyzed variable.
\item \code{trend}: Estimated slope of the linear model.
\item \code{t}: t-statistic for the species-specific trend.
\item \code{pvalue}: Statistical significance of the species-specific trend.
\item \code{dif_t}: t-statistic of the interaction term, indicating the magnitude of the difference between the species trend and the Overall Trend (OT).
\item \code{dif_pvalue}: p-values of the interaction term. A low value indicates a significant deviation from the general trend.
\item \code{n}: Total number of occurrence records (sample size) for the specific species.
\item \code{hemisphere}: Geographical subset (\code{North}, \code{South}, or \code{Global}) used to ensure latitudinal symmetry in the analysis.
}}

\item{sig_level}{The \code{numeric} significance level to use for classifying trends as significant. Defaults to \code{0.05}. See Bonferroni correction \code{0.05/length(species)}.}

\item{responses}{A \verb{character vector} of response variable names to analyze (\code{c("lat", "lon", "tme", "ele")}).
The function will create classification columns for responses present in this vector and in the \code{responses} column of \code{spp_trend_result}.}
}
\value{
A \verb{data frame} summarizing the ecological strategy of each species for each analyzed response variable.
The table includes:
\itemize{
\item Species name
\item Hemisphere
\item Sample size
\item Classification columns for:
\itemize{
\item Spatial (latitude, longitude and elevation if present) responses. Spatial Adaptation \code{SA}, Spatial Discordance \code{SD}, Spatial Conformity \code{SC}
\item Thermal (temperature if present) responses. Thermal Tolerance \code{TT}, Thermal Adjustment \code{TA}, Thermal Conformity \code{TC}
}
}

Classification for spatial responses (\code{lat}, \code{lon}, \code{ele}) are classified as \code{Spatial_lat}, \code{Spatial_lon} and \code{Spatial_ele}.
Thermal responses (\code{tme}) are classified as \code{Thermal_tme}.
}
\description{
This function analyses the outputs of \code{spp_trend()} to classify species into distinct spatial or thermal response categories based on the direction and statistical
significance of their species-specific trends relative to the overall trend.
The function incorporates hemisphere-specific logic to correctly interpret poleward shifts in latitude and can also be applied to classify elevational trends.
}
\details{
This function takes the trend analysis results from \code{spp_trend} and classifies each species' response based on the
significance of its trend and how it differs from the general trend.
Applied Bonferroni correction to avoid false positives (Type I errors) due to multiple comparisons when analyzing many species.
The classification identifying three possible spatial responses and three thermal responses:
\itemize{
\item \strong{Spatial Responses:}
\itemize{
\item \strong{Spatial Adaptation (SA):} A significant positive temporal trend in the spatial position of species occurrences.
In the context of climate change, this pattern is commonly associated with a poleward shift, corresponding to a northward displacement
(towards higher latitude values) in the Northern Hemisphere and southward displacement (towards lower latitude values) in the Southern Hemisphere, as species expand into newly suitable areas.
\item \strong{Spatial Discordance (SD):} A significant negative temporal trend in the spatial position of species occurrences.
In the context of climate change, this pattern is often associated with an equatorward shift and may arise when other ecological and anthropogenic
factors influence species distributions independently of, or in opposition to, climate-driven range shifts.
\item \strong{Spatial Conformity (SC):} A spatial response pattern in which the species-specific temporal trend does not differ significantly from the overall trend.
Species showing spatial conformance share the same bias structure as the complete dataset, preventing the inference of a distinct, species-specific response to climate change at the scale of analysis.
}
\item \strong{Thermal Responses:}
\itemize{
\item \strong{Thermal Tolerance (TT):} A thermal response pattern characterised by a significant positive temporal trend in the temperature conditions under which species are observed, relative to the overall trend.
This pattern suggest an increased likelihood of occurrence under warmer conditions and an apparent capacity to tolerate rising temperatures through physiological, behavioural, and evolutionary mechanisms.
\item \strong{Thermal Adjustment (TA):} A thermal response characterised by a significant negative temporal trend in the temperature conditions associated with species occurrences, relative to the overall trend.
This indicates and increasing association with cooler temperature conditions over time, potentially reflecting microevolutionary change or phenotypic adjustment.
\item \strong{Thermal Conformity (TC):} A thermal response pattern in which species-specific temperature trends do not differ significantly from the overall trend.
Species showing thermal conformance share the same background thermal signal as the complete dataset, preventing the formulation of specific hypotheses regarding climate-driven thermal responses.
}
}

\emph{Note: The interpretation of longitude trends assumes that if transformation was applied in \code{spp_trend}, it used the Antimeridian as 0.}
}
\examples{

# Assuming spp_trends_results is a data frame generated by spp_trend()

spp_trends_results <- data.frame(
  species = paste0("spp_", 1:10),
  responses = rep(c("lat", "lon", "tme"), length.out = 30),
  trend = runif(30, -0.5, 0.5),
  t = runif(30, -2, 2),
  pvalue = runif(30, 0, 1),
  dif_t = runif(30, -1, 1.5),
  dif_pvalue = runif(30, 0.001, 0.9),
  n = round(runif(30, 40, 60)),
  hemisphere = sample(c("North", "South", "Global"), 30, replace = TRUE)
)

spp <- unique(spp_trends_results$species)
sig_level <- 0.05 / length(spp) # Bonferroni correction
responses_to_analyze <- c("lat", "lon", "tme")

spp_strategy_results <- spp_strategy(spp_trends_results,
                                     sig_level = sig_level,
                                     responses = responses_to_analyze)

print(spp_strategy_results)

}
