# LogisExpPower.R
#' Logistic-Exponential Power (LEP) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Exponential Power distribution.
#'
#' The Logistic-Exponential Power distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Exponential Power distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{1+\left\{\exp \left(e^{\lambda x^\beta}-1\right)-1\right\}^\alpha}   ; x \geq 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dlogis.exp.power()} — Density function
#'   \item \code{plogis.exp.power()} — Distribution function
#'   \item \code{qlogis.exp.power()} — Quantile function
#'   \item \code{rlogis.exp.power()} — Random generation
#'   \item \code{hlogis.exp.power()} — Hazard function
#' }
#'
#' @usage
#' dlogis.exp.power(x, alpha, beta, lambda, log = FALSE)
#' plogis.exp.power(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.exp.power(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.exp.power(n, alpha, beta, lambda)
#' hlogis.exp.power(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.exp.power}: numeric vector of (log-)densities
#'   \item \code{plogis.exp.power}: numeric vector of probabilities
#'   \item \code{qlogis.exp.power}: numeric vector of quantiles
#'   \item \code{rlogis.exp.power}: numeric vector of random variates
#'   \item \code{hlogis.exp.power}: numeric vector of hazard values
#' }
#'
#' @references
#' Joshi, R. K., Sapkota, L.P., & Kumar, V. (2020). 
#' The Logistic-Exponential Power Distribution with Statistical Properties 
#' and Applications. \emph{International Journal of Emerging Technologies 
#' and Innovative Research}, \bold{7(12)}, 629--641.  
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.1)
#' dlogis.exp.power(x, 1.5, 1.5, 0.1)
#' plogis.exp.power(x, 1.5, 1.5, 0.1)
#' qlogis.exp.power(0.5, 1.5, 1.5, 0.1)
#' rlogis.exp.power(10, 2.0, 5.0, 0.1)
#' hlogis.exp.power(x, 1.5, 1.5, 0.1)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=1.8940, beta=1.2276, lambda=0.1650)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.exp.power, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.exp.power, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.exp.power, pfun=plogis.exp.power, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisExpPower
#' @aliases dlogis.exp.power plogis.exp.power qlogis.exp.power rlogis.exp.power hlogis.exp.power
#' @export
NULL

#' @rdname LogisExpPower
#' @usage NULL
#' @export   
dlogis.exp.power <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- exp(lambda *(x[valid] ^ beta))
    num <- alpha * beta *lambda *(x[valid] ^(beta - 1)) * u * exp(u - 1) * ((exp(u - 1)-1)^(alpha - 1.0))
    deno <- (1.0 + (exp(u - 1.0)-1) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname LogisExpPower
#' @usage NULL
#' @export  
plogis.exp.power <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(exp(lambda *(q[valid] ^ beta)) -1)
	tmp <- 1 + ((u - 1) ^ alpha)
    cdf[valid] <- 1 - (1/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisExpPower
#' @usage NULL
#' @export
qlogis.exp.power <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
	qtl[valid] <- ((1/lambda) *log(log(1 + ((p[valid]/(1 - p[valid])) ^ (1/alpha))) + 1))^ (1/beta) 
    }       
    return(qtl)   
}

#' @rdname LogisExpPower
#' @usage NULL
#' @export
rlogis.exp.power <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.exp.power(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisExpPower
#' @usage NULL
#' @export
hlogis.exp.power <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dlogis.exp.power(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.exp.power(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}  


