# LindleyInvWeibull.R
#' Lindley-Inverse Weibull Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley-Inverse Weibull distribution.
#'
#' The Lindley-Inverse Weibull distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley-Inverse Weibull distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \theta) =
#' \quad 1 - {\left( {1 - {e^{ - \alpha {x^{ - \beta }}}}} \right)^\theta }
#' \left\{ {1 - \left( {\frac{\theta }{{\theta  + 1}}} \right)\ln 
#' \left( {1 - {e^{ - \alpha {x^{ - \beta }}}}} \right)} \right\}  \, ; \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\theta} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dlindley.inv.weib()} — Density function
#'   \item \code{plindley.inv.weib()} — Distribution function
#'   \item \code{qlindley.inv.weib()} — Quantile function
#'   \item \code{rlindley.inv.weib()} — Random generation
#'   \item \code{hlindley.inv.weib()} — Hazard function
#' }
#'
#' @usage
#' dlindley.inv.weib(x, alpha, beta, theta, log = FALSE)
#' plindley.inv.weib(q, alpha, beta, theta, lower.tail = TRUE, log.p = FALSE)
#' qlindley.inv.weib(p, alpha, beta, theta, lower.tail = TRUE, log.p = FALSE)
#' rlindley.inv.weib(n, alpha, beta, theta)
#' hlindley.inv.weib(x, alpha, beta, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlindley.inv.weib}: numeric vector of (log-)densities
#'   \item \code{plindley.inv.weib}: numeric vector of probabilities
#'   \item \code{qlindley.inv.weib}: numeric vector of quantiles
#'   \item \code{rlindley.inv.weib}: numeric vector of random variates
#'   \item \code{hlindley.inv.weib}: numeric vector of hazard values
#' }
#'
#' @references 
#' Joshi, R. K., & Kumar, V. (2020). 
#' Lindley inverse Weibull distribution: Theory and Applications.
#' \emph{Bull. Math. & Stat. Res.}, \bold{8(3)}, 32--46. 
#'  
#' @examples
#' x <- seq(0.1, 1, 0.1)
#' dlindley.inv.weib(x, 1.5, 2.0, 0.5)
#' plindley.inv.weib(x, 1.5, 2.0, 0.5)
#' qlindley.inv.weib(0.5, 2.0, 5.0, 0.1)
#' rlindley.inv.weib(10, 1.5, 2.0, 0.5)
#' hlindley.inv.weib(x, 1.5, 2.0, 0.5)
#' 
#' # Data
#' x <- waiting
#' # ML estimates
#' params = list(alpha=9.3340, beta=0.3010, theta=104.4248)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plindley.inv.weib, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlindley.inv.weib, fit.line=FALSE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlindley.inv.weib, pfun=plindley.inv.weib, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyInvWeibull
#' @aliases dlindley.inv.weib plindley.inv.weib qlindley.inv.weib rlindley.inv.weib hlindley.inv.weib
#' @export
NULL

#' @rdname LindleyInvWeibull
#' @usage NULL
#' @export   
dlindley.inv.weib <- function (x, alpha, beta, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, beta, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, beta, theta) <= 0)) {
        stop("Parameters 'alpha', 'beta' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- alpha * (x[valid] ^ -beta)	
	z <- 1 - exp(-u) 	
    pdf[valid] <- (theta*theta/(1+theta)) * alpha * beta * (x[valid]^-(beta+1)) * exp(-u) * (z ^ (theta-1)) * (1 - log(z))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyInvWeibull
#' @usage NULL
#' @export
plindley.inv.weib <- function (q, alpha, beta, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, theta) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- alpha * (q[valid] ^ -beta)	
	z <- 1 - exp(-u)
    cdf[valid] <- 1.0 - (z ^ theta) * (1 - (theta/(1+theta)) * log(z))  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LindleyInvWeibull
#' @usage NULL
#' @export
qlindley.inv.weib <- function (p, alpha, beta, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, theta) <= 0))
		stop("Parameters 'alpha', 'beta', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	   theta*log(1-exp(-alpha*(x^-beta)))-log(1-wi) 
	      + log(1-(theta/(1+theta)) *log(1-exp(-alpha*(x^-beta))))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname LindleyInvWeibull
#' @usage NULL
#' @export
rlindley.inv.weib <- function(n, alpha, beta, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, theta), is.numeric)))
		stop("'alpha', 'beta', and 'theta' must be numeric.")
	if (any(c(alpha, beta, theta) <= 0))
		stop("Invalid parameters: alpha, 'beta', and theta must be positive.")
 
    u <- runif(n)
    q <- qlindley.inv.weib(u, alpha, beta, theta)
    return(q)
}

#' @rdname LindleyInvWeibull
#' @usage NULL
#' @export
hlindley.inv.weib <- function (x, alpha, beta, theta)
{
	if (!all(sapply(list(x, alpha, beta, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
		
	nume <- dlindley.inv.weib(x, alpha, beta, theta)
	surv <- 1.0 - plindley.inv.weib(x, alpha, beta, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
	
    return(hrf)   
} 


