# InvUBD.R
#' Inverse Upside Down Bathtub-shaped Hazard Function Distribution
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverse Upside Down Bathtub-shaped Hazard Function distribution.
#'
#' The Inverse Upside Down Bathtub-shaped Hazard Function distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Inverse Upside Down Bathtub-shaped Hazard Function distribution has CDF:
#'
#' \deqn{
#' F(x;\,\alpha,\beta,\lambda) = \, \exp \left[ {1 - {{\left( {1 + \lambda {x^{ - \beta }}} 
#' \right)}^\alpha }} \right], \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dinv.ubd()} — Density function
#'   \item \code{pinv.ubd()} — Distribution function
#'   \item \code{qinv.ubd()} — Quantile function
#'   \item \code{rinv.ubd()} — Random generation
#'   \item \code{hinv.ubd()} — Hazard function
#' }
#'
#' @usage
#' dinv.ubd(x, alpha, beta, lambda, log = FALSE)
#' pinv.ubd(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qinv.ubd(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rinv.ubd(n, alpha, beta, lambda)
#' hinv.ubd(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.ubd}: numeric vector of (log-)densities
#'   \item \code{pinv.ubd}: numeric vector of probabilities
#'   \item \code{qinv.ubd}: numeric vector of quantiles
#'   \item \code{rinv.ubd}: numeric vector of random variates
#'   \item \code{hinv.ubd}: numeric vector of hazard values
#' }
#'
#' @references
#'   
#' Dimitrakopoulou, T., Adamidis, K., & Loukas, S.(2007). 
#' A liftime distribution with an upside down bathtub-shaped hazard function, 
#' \emph{IEEE Trans. on Reliab.}, \bold{56(2)}, 308--311. 
#'
#' Joshi, R.K., & Kumar, V. (2018). 
#' Inverse Upside Down Bathtub-Shaped Hazard Function distribution: Theory
#' and Applications. \emph{Journal of National Academy of Mathematics}, \bold{32}, 6--20.
#' 
#' @examples
#' x <- seq(0.1, 1, 0.1)
#' dinv.ubd(x, 1.5, 0.8, 2)
#' pinv.ubd(x, 1.5, 0.8, 2)
#' qinv.ubd(0.5, 1.5, 0.8, 2)
#' rinv.ubd(10, 1.5, 0.8, 2)
#' hinv.ubd(x, 1.5, 0.8, 2)
#'
#' #Data
#' x <- rainfall
#' #ML Estimates    
#' params = list(alpha =0.1804, beta=4.3216, lambda=85.13)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.ubd, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.ubd, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dinv.ubd, pfun = pinv.ubd, plot=FALSE)
#' print.gofic(out)
#' 
#' @name InvUBD
#' @aliases dinv.ubd pinv.ubd qinv.ubd rinv.ubd hinv.ubd
#' @export
NULL

#' @rdname InvUBD
#' @usage NULL
#' @export 
dinv.ubd <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- 1 + lambda * (x[valid] ^ -beta)
	z <- exp(1 - u ^ alpha)
    pdf[valid] <- alpha * beta * lambda * (x[valid] ^ -(beta+1)) *(u ^(alpha-1)) * z
    }
	if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname InvUBD
#' @usage NULL
#' @export 
pinv.ubd <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
    u <- 1 + lambda * (q[valid] ^ -beta)
    cdf[valid] <- exp(1 - u ^ alpha)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname InvUBD
#' @usage NULL
#' @export
qinv.ubd <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {		
		z <- (1/lambda) * (((1 - log(p[valid])) ^ (1/alpha))-1) 		
		qtl[valid] <- z ^ (-1/beta)
	}
    return(qtl)   
}

#' @rdname InvUBD
#' @usage NULL
#' @export 
rinv.ubd <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
 
    u <- runif(n)
    q <- qinv.ubd(u, alpha, beta, lambda)
    return(q)
}

#' @rdname InvUBD
#' @usage NULL
#' @export 
hinv.ubd <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
    nume <- dinv.ubd(x, alpha, beta, lambda, log = FALSE)
	surv <- 1.0 - pinv.ubd(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}
