# HCGenExp.R
#' Half-Cauchy Generalized Exponential(HCGE) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy Generalized Exponential(HCGE) distribution.
#'
#' The HCGE distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The HCGE distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \quad 1 - \frac{2}{\pi }\arctan \left[ { - \frac{\alpha }{\theta }\ln 
#' \left( {1 - {e^{ - \lambda x}}} \right)} \right] \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dhc.gen.exp()} — Density function
#'   \item \code{phc.gen.exp()} — Distribution function
#'   \item \code{qhc.gen.exp()} — Quantile function
#'   \item \code{rhc.gen.exp()} — Random generation
#'   \item \code{hhc.gen.exp()} — Hazard function
#' }
#'
#' @usage
#' dhc.gen.exp(x, alpha, lambda, theta, log = FALSE)
#' phc.gen.exp(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.gen.exp(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.gen.exp(n, alpha, lambda, theta)
#' hhc.gen.exp(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.gen.exp}: numeric vector of (log-)densities
#'   \item \code{phc.gen.exp}: numeric vector of probabilities
#'   \item \code{qhc.gen.exp}: numeric vector of quantiles
#'   \item \code{rhc.gen.exp}: numeric vector of random variates
#'   \item \code{hhc.gen.exp}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary, A.K., Sapkota, L.P. &  Kumar, V. (2022). 
#' Half-Cauchy Generalized Exponential Distribution:Theory and Application. 
#' \emph{Journal of Nepal Mathematical Society (JNMS)}, \bold{5(2)}, 1--10.  
#' \doi{10.3126/jnms.v5i2.50018}
#' 
#' Gupta, R. D.,  & Kundu, D. (1999). 
#' Generalized exponential distributions. 
#' \emph{Australian and New Zealand Journal of Statistics}, \bold{41(2)}, 173--188.
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dhc.gen.exp(x, 2.0, 0.5, 0.1)
#' phc.gen.exp(x, 2.0, 0.5, 0.1)
#' qhc.gen.exp(0.5, 2.0, 0.5, 0.1)
#' rhc.gen.exp(10, 2.0, 0.5, 0.1)
#' hhc.gen.exp(x, 2.0, 0.5, 0.1)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(alpha=6.6141, lambda=0.9352, theta=0.0103)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.gen.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.gen.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' res <- gofic(x, params = params,
#'              dfun = dhc.gen.exp, pfun=phc.gen.exp, plot=FALSE)
#' print.gofic(res)
#' 
#' @name HCGenExp
#' @aliases dhc.gen.exp phc.gen.exp qhc.gen.exp rhc.gen.exp hhc.gen.exp
#' @export
NULL

#' @rdname HCGenExp
#' @usage NULL
#' @export   
dhc.gen.exp <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- exp(-lambda*x[valid])	
	z <- 1 - u	
    pdf[valid] <- (2/pi)*(alpha*lambda/theta)*u *(1/z) * (1/(1+(-alpha*log(z)/theta)^2))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCGenExp
#' @usage NULL
#' @export 
phc.gen.exp <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -alpha * log(1- exp(-lambda*q[valid]))/theta	
    cdf[valid] <- 1- (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCGenExp
#' @usage NULL
#' @export 
qhc.gen.exp <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <-  -(1/lambda) * log(1.0- exp(-(theta/alpha)*tan((1-p[valid]) * pi *0.5)))	   
    }    
    return(qtl)  
 } 

#' @rdname HCGenExp
#' @usage NULL
#' @export 
rhc.gen.exp <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.gen.exp(u, alpha, lambda, theta)
    return(q)
}

#' @rdname HCGenExp
#' @usage NULL
#' @export 
hhc.gen.exp <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dhc.gen.exp(x, alpha, lambda, theta) 
	surv <- 1.0 - phc.gen.exp(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 
