
#' Filtering for Microbial Features of Low Abundance and Low Prevalence.
#'
#' @description
#' This function filteres an OTU/ASV table based on overall counts and prevalence thresholds, and
#'     optionally applies a logarithmic transformation. When grouping variables are provided,
#'     the function performs abundance and prevalence filtering within each group separately.
#'
#'
#' @param Data A data frame or a list object which contains the selected biomarker count table
#'     (generated by \code{\link[MicrobTiSDA]{Data.rf.classifier}}), where rows represent OTUs/ASVs
#'     and columns represent samples.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param OTU_counts_filter_value An integer, indicating the sum of the minimum abundances of OTUs/ASVs in all samples. If the sum
#'     of the abundances that OTU is below the given positive integer threshold, the OTU is excluded, and vice versa, it is retained.
#'     The default is 1000. Note: if the input \code{Data} is the important OTU table that produced via sample classification,
#'     this argument should be \code{NA}, As the low abundance OTUs/ASVs might be filtered out during the classification progress by
#'     \code{\link[MicrobTiSDA]{Data.rf.classifier}}.
#' @param OTU_filter_value Numeric between 0 and 1. This specifies the minimum prevalence rate of an OTU/ASV across all samples within
#'     each group or individual. OTUs/ASVs with a prevalence rate below the given threshold will be removed.
#' @param log_base This argument specifies the base of the logarithm. When the dataset is not very large, the default is \code{NA},
#'     and no logarithmic transformation is applied. For large datasets, the logarithm base can be 2, "e", or 10.
#' @param Group_var A string or a vector. This specifies the grouping variables, which should match the column names in
#'     the \code{metadata} used to designate sample groups, and for pre-processing OTU data of each group or individual separately.
#'     For instance, to split the OTU table based on the \code{Group} variable, set \code{Group_var = "Group"};
#'     to split the data based on the \code{Group} and \code{Diet} (if in \code{metadata})categorical variables to study the
#'     interaction between different grouping variables, set \code{Group_var = c("Group","Diet")}.
#'
#' @details
#' The function executes several key steps:
#' \enumerate{
#'   \item \strong{Input Validation:}
#'         It first checks whether the input \code{Data} is a data frame or a list generated by function \code{\link[MicrobTiSDA]{Data.rf.classifier}}.
#'         If \code{Data} is a list but not a data frame, the first element is extracted. Otherwise, if \code{Data} is neither a data frame nor an appropriate
#'         list, the function stops with an error.
#'
#'   \item \strong{OTU Count Filtering:}
#'         If an \code{OTU_counts_filter_value} is provided (i.e., not \code{NA}), OTUs with total counts (across all samples) less than or equal to this value are removed.
#'
#'   \item \strong{Logarithmic Transformation:}
#'         If a \code{log_base} is specified (allowed values are 10, 2, or e), a log transformation (with an offset of 1 to avoid log(0)) is applied to the data.
#'         If \code{log_base} is \code{NA}, the data remains untransformed.
#'
#'   \item \strong{Prevalence Filtering without Grouping:}
#'         When \code{Group_var} is not provided (\code{NULL}), if an \code{OTU_filter_value} is specified, the function filters out OTUs whose prevalence (the proportion of samples
#'         with a non-zero count) is less than the threshold. If \code{OTU_filter_value} is not provided, a warning is issued and no prevalence filtering is applied.
#'
#'   \item \strong{Group-based Prevalence Filtering:}
#'         If one or more grouping variables are specified in \code{Group_var}, the function first checks that these variables exist in \code{metadata}.
#'         For each group (or combination of groups if multiple variables are provided), the prevalence of each OTU is calculated, and OTUs are retained if they
#'         meet the prevalence threshold in at least one group. The filtered OTU table is then returned.
#' }
#'
#'
#' @return A list of class \code{FilteredData} containing:
#' \describe{
#'   \item{filtered_table}{The filtered OTU count table, optionally log-transformed.}
#'   \item{parameters}{A list of the filtering parameters used.}
#'   \item{metadata}{The input metadata, possibly augmented with a combined grouping variable if multiple \code{Group}s were provided.}
#' }
#' @export
#' @author Shijia Li
#'
#' @examples
#' # Example OTU table
#' set.seed(123)
#' otu_table <- as.data.frame(matrix(sample(0:100, 100, replace = TRUE), nrow = 10))
#' rownames(otu_table) <- paste0("OTU", 1:10)
#' colnames(otu_table) <- paste0("Sample", 1:10)
#'
#'
#' # Example metadata
#' metadata <- data.frame(
#'   Group = rep(c("A", "B"), each = 5),
#'   row.names = paste0("Sample", 1:10)
#' )
#'
#' # Filter OTU table without grouping
#' filtered_data <- Data.filter(
#'   Data = otu_table,
#'   metadata = metadata,
#'   OTU_counts_filter_value = 50,
#'   OTU_filter_value = 0.2
#' )
#'
#' # Filter OTU table with grouping
#' filtered_data_grouped <- Data.filter(
#'   Data = otu_table,
#'   metadata = metadata,
#'   OTU_filter_value = 0.5,
#'   Group_var = "Group"
#' )
#'
#'
Data.filter <- function(Data,
                        metadata,
                        OTU_counts_filter_value = 1000,
                        OTU_filter_value = NA,
                        log_base = NA,
                        Group_var = NULL) {
  if (is.list(Data) == TRUE && is.data.frame(Data) != TRUE) {
    Data <- Data[[1]]
  } else if (is.list(Data) == TRUE && is.data.frame(Data) == TRUE) {
    Data <- Data
  } else {
    stop("The input 'Data' should either be a dataframe or the output of the function 'OTU_classifier'.")
  }

  if (!is.na(OTU_filter_value) && (OTU_filter_value < 0 || OTU_filter_value > 1)) {
    stop("The 'OTU_filter_value' should be between 0 and 1.")
  }

  if (!is.na(OTU_counts_filter_value)) {
    Data <- Data[rowSums(Data) > OTU_counts_filter_value, ]
  }

  if (!is.na(log_base)) {
    log_table <- switch(log_base,
                        `10` = apply(Data, 2, function(x) log10(x + 1)),
                        `2` = apply(Data, 2, function(x) log2(x + 1)),
                        `e` = apply(Data, 2, function(x) log(x + 1)),
                        stop("The log_base should only be 10, 2, or e"))
  } else {
    log_table <- Data
  }

  if (is.null(Group_var)) {
    if (is.na(OTU_filter_value)) {
      warning("No OTU_filter_value specified, returning unfiltered data.")
      log_table_filtered <- log_table
    } else {
      log_table_filtered <- log_table[rowMeans(log_table > 0) >= OTU_filter_value, ]
    }
  } else {
    if (!all(Group_var %in% colnames(metadata))) {
      stop("Some grouping variables in 'Group_var' do not exist in 'metadata'.")
    }

    if (length(Group_var) == 1) {
      groups <- unique(metadata[[Group_var]])
      group_OTU <- lapply(groups, function(group) {
        samples_in_group <- rownames(subset(metadata, metadata[[Group_var]] == group))
        log_table[, samples_in_group]
      })
    } else {
      metadata$Group_combo <- interaction(metadata[, Group_var, drop = FALSE])
      groups <- unique(metadata$Group_combo)
      group_OTU <- lapply(groups, function(group) {
        samples_in_group <- rownames(subset(metadata, metadata$Group_combo == group))
        log_table[, samples_in_group]
      })
    }

    OTU_frequencies <- lapply(group_OTU, function(group) rowMeans(group > 0))
    selected_OTU <- lapply(OTU_frequencies, function(frequencies) {
      names(frequencies[frequencies >= OTU_filter_value])
    })

    common_OTU <- unique(unlist(selected_OTU))
    log_table_filtered <- log_table[common_OTU, , drop = FALSE]
  }

  result <- list(
    filtered_table = log_table_filtered,
    parameters = list(
      OTU_counts_filter_value = OTU_counts_filter_value,
      OTU_filter_value = OTU_filter_value,
      log_base = log_base,
      Group_var = Group_var
    ),
    metadata = metadata
  )

  class(result) <- "FilteredData"
  return(result)
}




#' Interpolate Time-Series Data Based on Sample Time
#'
#' @description
#' This function performs interpolation on a data frame or matrix (e.g., OTU/ASV counts or other time-series measurements)
#'     using corresponding metadata time points. For each unique subject (as defined by a subject ID), the function constructs
#'     a full time series between the minimum and maximum time points and applies interpolation (defaulting to cubic interpolation)
#'     to generate data for missing time points. The function returns both the interpolated time-series data and the associated
#'     updated metadata.
#'
#' @details
#' This function processes the input data and metadata by interating over each unique subject ID defined in \code{Sample_ID}. For
#'     each subject, it subsets and sorts the metadata by \code{Sample_Time} and constructs a complete time series from the minimum
#'     to maximum time values with a step of 1. It then extracts the corresponding data columns and performs interpolation (Using
#'     the specified \code{interp_method}, with \code{cubic} as the default) on each feature across the full time series. Simultaneously,
#'     updated metadata is generated for the interpolated time points, preserving the subject ID and group information as indicated by
#'     \code{Group_var}. The function returns a list object containing the interpolated data matrix and the corresponding
#'     updated metadata.
#'
#' @param Data A data frame where rows represent OTUs/ASVs and columns represent samples Or the output of the function
#'     \code{\link[MicrobTiSDA]{Data.filter}}.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param Sample_Time A character string specifying the column name in \code{metadata} that contains time information.
#' @param Sample_ID A character string specifying the column name in \code{metadata} that identifies unique samples of each subject.
#' @param interp_method A character string specifying the interpolation method to be used by \code{\link[pracma]{interp1}}.
#'                      Default is \code{'cubic'}. Other methods accepted by \code{interp1} (e.g., \code{'linear'})
#'                      can also be used.
#' @param Group_var A character string specifying the column name in \code{metadata} that indicates group membership.
#'
#' @return An object of class \code{"MicrobTiSDA.interpolate"} containing:
#'   \item{Interpolated_Data}{A data frame of interpolated abundance data.}
#'   \item{Interpolated_Data_metadata}{A data frame of corresponding interpolated metadata.}
#'
#' @importFrom pracma interp1
#' @author Shijia Li
#' @export
#'
#' @examples
#' \donttest{
#' # Example data: 5 features across 8 samples with time points from two subjects.
#' set.seed(123)
#' Data <- matrix(sample(1:100, 40, replace = TRUE), nrow = 5)
#' rownames(Data) <- paste0("Feature", 1:5)
#' colnames(Data) <- paste0("Sample", 1:8)
#'
#' # Create metadata with time points, sample IDs, and group assignments.
#' metadata <- data.frame(
#'   Time = c(1, 3, 5, 7, 2, 4, 6, 8),
#'   ID = c(rep("Subject1", 4), rep("Subject2", 4)),
#'   Group = c(rep("A", 4), rep("B", 4)),
#'   row.names = paste0("Sample", 1:8)
#' )
#'
#' # Interpolate the data using cubic interpolation.
#' interp_results <- Data.interpolate(Data = Data,
#'                                    metadata = metadata,
#'                                    Sample_Time = "Time",
#'                                    Sample_ID = "ID",
#'                                    interp_method = "cubic",
#'                                    Group_var = "Group")
#' }
Data.interpolate <- function(Data,
                             metadata,
                             Sample_Time,
                             Sample_ID,
                             interp_method = 'cubic',
                             Group_var) {

  if (is.object(Data)) {
    Data <- Data$filtered_table
  } else {
    Data <- Data
  }

  sample_id <- unique(metadata[[Sample_ID]])
  interpolated_data <- data.frame(matrix(ncol = 0, nrow = nrow(Data)))
  interpolated_meta <- data.frame()

  for (i in sample_id) {
    meta_id <- subset(metadata, metadata[[Sample_ID]] == i)
    sorted_meta_id <- meta_id[order(meta_id[[Sample_Time]]), ]
    Group <- unique(sorted_meta_id[[Group_var]])
    full_time_series <- seq(min(sorted_meta_id[, Sample_Time]), max(sorted_meta_id[, Sample_Time]), 1)
    data_id <- Data[, rownames(sorted_meta_id)]

    intp_data <- as.data.frame(matrix(0, nrow = nrow(data_id), ncol = length(full_time_series)))
    rownames(intp_data) <- rownames(data_id)
    colnames(intp_data) <- paste(i, "_T_", full_time_series, sep = "")

    intp_meta <- as.data.frame(matrix(0, nrow = ncol(intp_data), ncol = 0))
    rownames(intp_meta) <- colnames(intp_data)
    intp_meta$Time <- full_time_series
    intp_meta$ID <- rep(i, nrow(intp_meta))
    intp_meta$Group <- rep(Group, nrow(intp_meta))

    for (j in rownames(data_id)) {
      intp_data[j, ] <- pracma::interp1(
        x = as.numeric(sorted_meta_id[, Sample_Time]),
        y = as.numeric(data_id[j, ]),
        xi = full_time_series,
        method = interp_method
      )
    }

    interpolated_data <- cbind(interpolated_data, intp_data)
    interpolated_meta <- rbind(interpolated_meta, intp_meta)
  }

  results <- list(
    Interpolated_Data = interpolated_data,
    Interpolated_Data_metadata = interpolated_meta,
    Method = interp_method
  )
 class(results) <- "MicrobTiSDA.interpolate"
 return(results)
}



#' @title Modified Centered Log-Ratio (MCLR) Transformation
#' @description
#' Applies a modified centered log-ratio (MCLR) transformation to compositional data. This transformation is
#'     particularly useful in microbiome and compositional data analysis, as it normalizes the data by
#'     comparing each value to the geometric mean of the positive values in its row.
#'
#' @details
#' The MCLR method calculates the geometric mean of each sample from positive proportions only, normalized and
#'     log-transformation all non-zero components in the dataset. Specifically, let \eqn{x_{nt} \in \Omega^I} denotes
#'     the compositional vector for the sample from subject \eqn{\textit{n}} at timepoint \eqn{\textit{t}}, where \eqn{\Omega^I} represents the collection of \eqn{\textit{I}}
#'     microbial features.For simplicity of illustration, assume that the first \eqn{\textit{q}}
#'     elements of \eqn{x_{nt}} are zero while the remaining elements are non-zero. Then itcan be expressed as:
#'     \deqn{mclr_\epsilon (x_{nt}) = [0, \dots, 0, \ln{\left(\frac{x_{nt(q+1)}}{\tilde{g}(x_{nt})}\right)} + \epsilon, \dots, \ln{\left(\frac{x_{ntI}}{\tilde{g}(x_{nt})}\right)} + \epsilon]}
#'     where \eqn{\tilde{g}(x_{nt}) = \left(\prod_{i=q+1}^{p} x_{nti}\right)^{\frac{1}{I-q}}} is the geometric mean of the
#'     non-zero elements of \eqn{x_{nt}}. When \eqn{\varepsilon = 0}, \eqn{\text{mclr}_0} corresponds to the centered
#'     log-ratio transform applied to non-zero proportions only. When \eqn{\varepsilon > 0}, \eqn{\text{mclr}_\varepsilon}
#'     applies a positive shift to all non-zero compositions. To make all non-zero values strictly positive, by
#'     default \eqn{\varepsilon = 0.1}. The MCLR transformation is invariant to the addition of extra zero components,
#'     preserves the original zero measurements, and is overal rank preserving. For more details, see Yoon et al. (2019).
#' @references Yoon, Grace, Irina Gaynanova, and Christian L. Müller. "Microbial networks in SPRING-Semi-parametric rank-based correlation and partial correlation estimation for quantitative microbiome data." Frontiers in Genetics 10 (2019).
#' @param Z A numeric matrix or data frame containing the compositional data to be transformed.
#' @param base A numeric value specifying the logarithmic base to use (default is `exp(1)`, i.e., the natural logarithm).
#' @param eps A small positive constant added to the transformed data to ensure positivity and avoid zeros (default is `0.1`).
#'
#' @return A data matrix of the same size as \code{Z} after the modified centered log-ratio transformation.
#' @export
#'
#' @examples
#' \donttest{
#' # Example compositional data matrix
#' Z <- matrix(c(1, 2, 0, 4, 5, 6, 0, 8, 9), nrow = 3, byrow = TRUE)
#' transformed_Z <- mclr.transform(Z, base = 10, eps = 0.1)
#' }
mclr.transform <- function(Z, base = exp(1), eps = 0.1) {
  Z = as.matrix(Z)
  gm_mean_pos <- function(x) {
    y <- x[x > 0 & !is.na(x)]
    if (length(y) == 0) {
      return(0)
    } else {
      exp(sum(log(y)) / length(y))
    }
  }
  index <- which(Z > 0 & !is.na(Z))

  a <- apply(Z, 1, function(x) {
    y <- x[x > 0 & !is.na(x)]
    gm_mean = gm_mean_pos(x)
    if (gm_mean == 0) {
      x[] <- 0
    } else {
      y <- log(y / gm_mean, base)
      x[x > 0 & !is.na(x)] <- y
    }
    x
  })
  a <- t(a)
  if (length(index) > 0) {
    Z[index] <- a[index] + abs(min(a)) + eps
  }
  return(Z)
}



#' @title Transform Microbial Composition Data.
#'
#' @description
#' This function applies the modified centered log-ratio (MCLR) transformation function \code{\link[MicrobTiSDA]{mclr.transform}}
#'     to a data matrix (e.g., OTU/ASV counts). When a grouping variable is provided the transformation is applied separately for
#'     each group defined in the metadata.
#' @details
#' The function transforms the input data using the MCLR method. If no grouping variable is provided (i.e. \code{Group_var}
#'     is \code{NULL}), the transformation is applied to the entire dataset. If a single grouping variable is specified,
#'     the data is partitioned into subsets corresponding to the unique groups in the metadata, and the transformation
#'     is applied to each subset separately; the results are then combined using row binding. For multiple grouping variables,
#'     a composite grouping factor is created using the interaction of the specified variables, and the transformation is
#'     applied to each composite group in a similar manner.
#'
#' @param Data A data frame or matrix of microbial compositional data, with rows representing microbial features (OTUs/ASVs)
#'     and columns representing samples. If the function \code{Data.interpolate} is preformed, the first element of the output
#'     object of \code{Data.interpolate} should be extract as the input of this function.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param Group_var A string or a vector. This specifies the grouping variables, which should match the column names in
#'     the \code{metadata} used to designate sample groups, and for pre-processing OTU data of each group or individual separately.
#'     For instance, to split the OTU table based on the \code{"Group"} variable, set \code{Group_var = "Group"};
#'     to split the data based on the \code{"Group"} and \code{"Diet"} (if in \code{metadata})categorical variables to study the
#'     interaction between different grouping variables, set \code{Group_var = c("Group","Diet")}.
#'
#' @return An object of class \code{"TransformedData"} containing the transformed count table.
#' @author Shijia Li
#' @export
#'
#' @examples
#' \donttest{
#' # Create example data matrix (5 features x 10 samples)
#' set.seed(123)
#' Data <- matrix(sample(1:100, 50, replace = TRUE), nrow = 5)
#' rownames(Data) <- paste0("Feature", 1:5)
#' colnames(Data) <- paste0("Sample", 1:10)
#'
#' # Create example metadata with a grouping variable
#' metadata <- data.frame(Group = rep(c("A", "B"), each = 5))
#' rownames(metadata) <- paste0("Sample", 1:10)
#'
#' # Apply MCLR transformation to the entire dataset
#' transformed_data <- Data.trans(Data, metadata, Group_var = NULL)
#'
#' # Apply MCLR transformation separately for each group
#' transformed_data_by_group <- Data.trans(Data, metadata, Group_var = "Group")
#' }
#'
Data.trans <- function(Data,metadata,Group_var) {

  if (inherits(Data, "FilteredData")) {
    Data <- Data[[1]]
  } else if (inherits(Data,"RfBiomarker")) {
    Data <- Data[[1]]
  }

  if (is.null(Group_var)) {

    transformed_data <- as.data.frame(t(mclr.transform(Data)))

  } else if (length(Group_var) < 2) {

    groups <- unique(metadata[,Group_var])
    group_OTU <- list()
    for (group in groups) {
      samples_in_group <- rownames(subset(metadata, metadata[,Group_var] == group))
      group_data <- Data[,samples_in_group]
      transformed_data <- as.data.frame(t(mclr.transform(group_data)))
      group_OTU[[group]] <- transformed_data
    }
    transformed_data <- dplyr::bind_rows(group_OTU)

  } else {

    groups_combo <- metadata[, Group_var, drop = FALSE]
    metadata$Group_combo <- interaction(metadata[,colnames(groups_combo)])

    groups <- unique(metadata[,"Group_combo"])
    group_OTU <- list()
    for (group in groups) {
      samples_in_group <- rownames(subset(metadata, metadata[,Group_combo] == group))
      group_data <- Data[,samples_in_group]

      transformed_data <- as.data.frame(t(mclr.transform(group_data)))
      group_OTU[[group]] <- transformed_data
    }
    transformed_data <- dplyr::bind_rows(group_OTU)
  }
  transformed_data <- transformed_data[match(rownames(metadata), rownames(transformed_data)), ]

  results <- list(
    transformed_data = transformed_data
  )
  class(results) <- "TransformedData"

  return(results)
}



#' @title Species Interaction Inferrences
#' @description
#' This function describes interspecies interactions based on the discrete-time Lotka-Volterral model.
#' @details
#' This function implements the discrete-time Lotka-Volterra model to characterize species interactions in microbiome time-series data.
#'     The model describes the abundance (MCLR transformed) \eqn{x_{ni}} of species \eqn{i} for subject \eqn{n} at time \eqn{t+\Delta t} as:
#'     \deqn{x_{ni} (t+\Delta t) = \eta_{ni} (t) x_{ni} (t) \exp\left(\Delta t \sum_j c_{nij} (x_{nj} (t) - <x_{nj}>) \right)}
#'     where \eqn{<x_{nj}>} represents the equilibrium abundance of species \eqn{j}, typically defined as the
#'     median abundance across samples from the same subject; \eqn{c_{nij}} denotes the interaction coefficient of species
#'     \eqn{j} on species \eqn{i}; and \eqn{\eta_{ni} (t)} accounts for log-normally distributed stochastic effects. For
#'     computational simplicity, stochastic effects are ignored, \eqn{\Delta t} is set to 1. Taking the natural logarithm yealds:
#'     \deqn{\ln x_{ni} (t+1) - \ln x_{ni} (t) = \sum_j c_{nij} (x_{nj} (t) - <x_{nj}>)}
#'     To improve sparsity and interpretability, the LIMITS algorithm is applied, incorporating stepwise regression and bagging.
#'     First, 50% of the samples are randomly selected as the training set while the rest serve as the test set. An initial regression
#'     model includes only the self-interaction term:
#'     \deqn{\ln x_{ni} (t+1) - \ln x_{ni} (t) = c_{nii} (x_{ni} (t) - <x_{ni}>)}
#'     Stepwise regression then iteratively adds species interaction terms from a candidate set \eqn{S}, forming:
#'     \deqn{\ln x_{ni} (t+1) - \ln x_{ni} (t) = c_{nii} (x_{ni} (t) - <x_{ni}>) + \sum_{j \in S} c_{nij} (x_{nj} (t) - <x_{nj}>)}
#'     The inclusion of a new term is determined based on the improvement in mean squared error (MSE) on the test set:
#'     \deqn{\theta = \frac{\text{MSE}_{\text{before}} - \text{MSE}_{\text{after}}}{\text{MSE}_{\text{before}}}}
#'     If \eqn{\theta} exceeds a predefined threshold (default \eqn{10^{-3}}), the species is included. Bagging is performed over \eqn{B}
#'     iterations by repeating the random splitting and stepwise regression, enhancing robustness. The final interaction coefficient matrix is computed as:
#'     \deqn{c_{nij} = \text{median}(c_{nij}^{(1)}, c_{nij}^{(2)}, ..., c_{nij}^{(B)})}
#'     This approach refines the inferred species interactions while ensuring sparsity.
#'
#'
#' @param Data A matrix or data frame of the transformed species abundance data.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param Group_var A character string specifying the column name in \code{metadata} that defines the groups for analysis.
#' @param abund_centered_method A character string indicating the method to compute species equilibrium abundance.
#'     Accepted values are \code{median} (default) and \code{mean}.
#' @param num_iterations An integer specifying the number of bagging iterations for the iterative variable selection process. Default is 10.
#' @param error_threshold A numeric value representing the relative error improvement threshold for adding new predictors during bagging iteration.
#'     Default is 1e-3.
#' @param pre_error A numeric value specifying the initial (large) error used for comparison in the iterative procedure. Default is 10000.
#' @param seed Random seed, default by \code{NULL}.
#'
#' @return A S3 object with an element for each group defined by \code{Group_var}. Each element is a list containing:
#' \describe{
#'   \item{interaction_matrices}{A three-dimensional array of estimated interaction coefficients with dimensions corresponding to features
#'   \eqn{\times} features \eqn{\times} iterations.}
#'   \item{final_interaction_matrix}{A two-dimensional matrix of interaction coefficients obtained by taking the median over the iterations.}
#' }
#' @export
#' @author Shijia Li
#' @examples
#' \donttest{
#' # Example usage:
#' set.seed(123)
#' Data <- matrix(sample(1:100, 50, replace = TRUE), nrow = 5)
#' rownames(Data) <- paste0("Feature", 1:5)
#' colnames(Data) <- paste0("Sample", 1:10)
#'
#' # Create example metadata with a grouping variable
#' metadata <- data.frame(Group = rep(c("A", "B"), each = 5))
#' rownames(metadata) <- paste0("Sample", 1:10)
#' metadata$Time = rep(c(1,2,3,4,5),2)
#' metadata$ID = paste("ID",seq(1:10),"")
#'
#' results <- Spec.interact(Data = as.data.frame(t(Data)),
#'                          metadata = metadata,
#'                          Group_var = "Group",
#'                          abund_centered_method = "median",
#'                          num_iterations = 5,
#'                          error_threshold = 1e-3,
#'                          pre_error = 10000)
#' }
Spec.interact <- function(Data, metadata, Group_var, abund_centered_method = 'median',
                          num_iterations = 10, error_threshold = 1e-3, pre_error = 10000,
                          seed = NULL) {

  new_microbTiSDA <- function(results, params = list()) {
    structure(list(
      results = results,
      params = params
    ), class = "microbTiSDA")
  }

  if (inherits(Data,"TransformedData")) {
    Data <- Data[[1]]
  }

  results <- list()
  Data <- as.data.frame(t(Data))

  for (g in unique(metadata[[Group_var]])) {
    group_info <- metadata[metadata[[Group_var]] == g, ]
    otu_table <- as.data.frame(t(Data[, rownames(group_info)] + 1))

    equilibrium_abundance <- if (abund_centered_method == "median") {
      apply(otu_table, 2, median)
    } else if (abund_centered_method == "mean") {
      apply(otu_table, 2, mean)
    } else {
      stop("abund_centered_method must be 'median' or 'mean'")
    }

    design_matrix_list <- list()
    data_vector_list <- list()
    for (i in seq_along(otu_table)) {
      log_diff_vector <- diff(log(otu_table[[i]]))
      design_matrix <- as.matrix(otu_table[-nrow(otu_table), ] - equilibrium_abundance)
      design_matrix_list[[i]] <- design_matrix
      data_vector_list[[i]] <- log_diff_vector
    }

    interaction_matrices <- array(0, dim = c(ncol(otu_table), ncol(otu_table), num_iterations))

    for (iter in 1:num_iterations) {
      if (!is.null(seed)) {
        set.seed(seed)
      } else {set.seed(iter)}
      sample_indices <- sample(1:(nrow(otu_table) - 1), size = floor((nrow(otu_table) - 1) / 2))
      training_data <- otu_table[sample_indices, ]
      test_data <- otu_table[-sample_indices, ]

      for (i in seq_along(otu_table)) {
        active_set <- c(i)
        inactive_set <- setdiff(seq_len(ncol(otu_table)), i)
        data_vector <- data_vector_list[[i]][sample_indices]
        test_data_vector <- data_vector_list[[i]][-sample_indices]

        prev_error <- pre_error
        while (length(inactive_set) > 0) {
          candidate_errors <- sapply(inactive_set, function(j) {
            candidate_set <- c(active_set, j)
            design_matrix <- as.matrix(training_data[, candidate_set] - equilibrium_abundance[candidate_set])
            if (nrow(design_matrix) < ncol(design_matrix)) return(Inf)

            candidate_coefs <- MASS::ginv(design_matrix) %*% data_vector
            if (any(is.na(candidate_coefs) | is.infinite(candidate_coefs))) return(Inf)

            pred_values <- as.matrix(test_data[, candidate_set] - equilibrium_abundance[candidate_set]) %*% candidate_coefs
            error <- suppressWarnings(mean((pred_values - test_data_vector)^2))

            if (is.na(error) || is.infinite(error)) return(Inf)
            return(error)
          })

          if (length(candidate_errors) == 0 || all(is.infinite(candidate_errors))) break
          best_j <- inactive_set[which.min(candidate_errors)]
          best_error <- min(candidate_errors)

          if (!is.na(prev_error) && !is.na(best_error) && prev_error > 0 && best_error >= 0) {
            if ((prev_error - best_error) / prev_error > error_threshold) {
              active_set <- c(active_set, best_j)
              inactive_set <- setdiff(inactive_set, best_j)
              prev_error <- best_error
            } else {
              break
            }
          } else {
            break
          }
        }

        final_design_matrix <- as.matrix(training_data[, active_set] - equilibrium_abundance[active_set])
        final_data_vector <- data_vector_list[[i]][sample_indices]
        final_coefs <- MASS::ginv(final_design_matrix) %*% final_data_vector
        interaction_matrices[active_set, i, iter] <- final_coefs
      }
    }

    final_interaction_matrix_median <- apply(interaction_matrices, c(1, 2), median, na.rm = TRUE)
    colnames(final_interaction_matrix_median) <- colnames(otu_table)
    rownames(final_interaction_matrix_median) <- colnames(otu_table)
    final_interaction_matrix_median <- round(final_interaction_matrix_median, 4)

    results[[g]] <- list(
      interaction_matrices = interaction_matrices,
      final_interaction_matrix = final_interaction_matrix_median
    )
  }

  new_microbTiSDA(
    results = results,
    params = list(
      Group_var = Group_var,
      abund_centered_method = abund_centered_method,
      num_iterations = num_iterations,
      error_threshold = error_threshold,
      pre_error = pre_error
    )
  )
}



#' @title Dynamic Visualization of Microbial Interaction Networks
#' @description
#' This function visualizes microbial species interactions based on the given inter-species interaction results by \code{\link[MicrobTiSDA]{Spec.interact}}.
#'
#' @param Interact_data A list containing the inferred species interaction results generated by \code{\link[MicrobTiSDA]{Spec.interact}}.
#' @param threshold Indicates the minimum interaction coefficient used to visualize species interactions. The default value is 1e-6.
#' @param core_arrow_num Indicates the number of species pairs involved in species interactions that identify keystone species. The default value is 4,
#'     meaning that the screened keystone species must interact with three other species populations in addition to its own population.
#' @param Taxa A data frame providing taxonomic annotations for microbial species.
#' @param fontsize Indicates the size of the text font. The default is 15.
#'
#' @return An S3 objects of species interact information or plots.
#' @details
#' The function processes microbial interaction matrices from \code{\link[MicrobTiSDA]{Spec.interact}} to construct species interaction networks.
#' It selects edges with absolute interaction strengths above \code{threshold}, assigns interaction types (positive or negative).
#' The function supports onpional taxonomic annotations for improved interpretability.
#'
#' The nodes represent microbial species,. Edge colors indicate interaction types:
#' blue for negative and orange for positive interactions. Edge widths are scaled by interaction strength. If taxonomic annotations are provided,
#' node labels reflect species-level taxonomy; otherwise, OTU/ASV IDs are used.
#'
#' @export
#' @author Shijia Li
#' @import dplyr tidygraph ggraph scales
#' @importFrom ggplot2 ggplot
#' @importFrom tidyr pivot_longer
#' @importFrom dplyr mutate
#' @importFrom stats na.omit
#' @importFrom stats median
#' @importFrom tibble rownames_to_column
#' @importFrom visNetwork visEdges
#' @importFrom visNetwork visNodes
#' @importFrom visNetwork visOptions
#' @importFrom visNetwork visPhysics
#' @importFrom visNetwork visLayout
#'

Interact.dyvis <- function(Interact_data, threshold, core_arrow_num, Taxa = NULL, fontsize = 15) {

  if (!inherits(Interact_data, "microbTiSDA")) {
    stop("Interact_data must be a 'microbTiSDA' object")
  }

  dynamic_vis <- list()
  edges_nodes <- list()

  for (g in names(Interact_data$results)) {
    g_interact <- as.data.frame(Interact_data$results[[g]][['final_interaction_matrix']])

    edge_list <- which(abs(g_interact) > threshold, arr.ind = TRUE)
    edges <- data.frame(
      from = rownames(g_interact)[edge_list[, 1]],
      to = colnames(g_interact)[edge_list[, 2]],
      value = g_interact[edge_list],
      stringsAsFactors = FALSE
    )
    edges$color <- ifelse(edges$value > 0, "tomato", "steelblue")
    edges$width <- scales::rescale(abs(edges$value), to = c(1, 8))
    edges$arrows <- "to"

    node_ids <- unique(c(edges$from, edges$to))
    nodes <- data.frame(
      id = node_ids,
      label = node_ids,
      title = node_ids,
      stringsAsFactors = FALSE
    )

    degree_table <- table(c(edges$from, edges$to)) - 1
    degree <- stats::setNames(as.numeric(degree_table), names(degree_table))
    nodes$degree <- degree[as.character(nodes$id)]
    nodes$degree[is.na(nodes$degree)] <- 0
    nodes$core_microbe <- nodes$degree >= core_arrow_num
    nodes$color.background <- ifelse(nodes$core_microbe, "gold", "lightgray")
    nodes$color.border <- ifelse(nodes$core_microbe, "orange", "gray")
    nodes$size <- ifelse(nodes$core_microbe, 30, 20)
    nodes$title <- paste0(nodes$label, "<br>Degree: ", nodes$degree)

    if (!is.null(Taxa)) {
      Annotate <- Taxa[match(nodes$id, rownames(Taxa)), ]
      Annotate <- Annotate %>%
        dplyr::mutate(across(where(is.factor), as.character)) %>%
        dplyr::mutate(across(everything(), ~na_if(., "")))
      Taxon <- apply(Annotate, 1, function(x) { tail(na.omit(x), 1) })
      Taxon <- data.frame(Species_annotation = Taxon)
      nodes$label <- Taxon[, 1]
    }

    edges_nodes[[g]] <- list(edges = edges, nodes = nodes)

    dynamic_vis[[g]] <- visNetwork::visNetwork(nodes, edges, width = "100%", height = "800px") %>%
      visNetwork::visEdges(smooth = TRUE, arrows = "to") %>%
      visNetwork::visNodes(font = list(size = fontsize)) %>%
      visNetwork::visOptions(
        highlightNearest = list(enabled = TRUE, degree = 1),
        nodesIdSelection = TRUE
      ) %>%
      visNetwork::visPhysics(stabilization = TRUE) %>%
      visNetwork::visLayout(randomSeed = 123)
  }

  dyvis_object <- list(
    visualization = dynamic_vis,
    data = edges_nodes,
    params = list(
      threshold = threshold,
      core_arrow_num = core_arrow_num,
      fontsize = fontsize
    )
  )

  class(dyvis_object) <- "microbTiSDA_dynamic_vis"

  return(dyvis_object)
}



#' @title Create Design Matrix for Regression Analysis
#' @description
#' \code{Design} creates the design matrix of dummies for fitting regression models of microbiota in time.
#'
#' @details
#' The main functionality of \code{Design} is to add user-selected sample information to the pre-processed OTU/ASV table
#'     as independent variables for fitting the OTU time series regression models. One necessary independent variable
#'     for fitting is \code{Time}, so the default output of this function is the transformed OTU/ASV table with added
#'     sample Time information. If the user also inputs other qualitative variables such grouping, gender, etc., the
#'     function will define dummy variables to distinguish each group based on the number of qualitative variables
#'     entered by the user and the grouping situation of samples based on qualitative variables. Moreover, the subject ID
#'     of each sample will be added as a column to the generated design matrix.
#'
#' @param metadata A data frame containing information for all samples, which should be identical to the \code{metadata}
#'     received by other functions in \code{MicrobTiSDA}.
#' @param Group_var A string or a vector. Same as the \code{Group_var} in \code{\link[MicrobTiSDA]{Data.trans}}.
#' @param Pre_processed_Data The transformed data output from the \code{\link[MicrobTiSDA]{Data.trans}} function. A
#'     pre-processed OTU data frame with sample IDs as row names and OTU IDs as column names.
#' @param Sample_Time A character string indicating the column name in \code{metadata} that contains sample time information.
#' @param Sample_ID A character string indicating the column name in \code{metadata} that contains sample identifiers.
#'
#' @return An object of class \code{Design}, which contains:
#' \item{data}{A data frame ready for regression modeling.}
#' \item{params}{A list of parameters used to construct the design.}
#' @export
#' @author Shijia Li
#'
#' @examples
#' \donttest{
#' # Example metadata with grouping variables
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' # Create design data without grouping variables
#' design_data_no_group <- Design(metadata, Group_var = NULL, Pre_processed_Data,
#'                                Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#' }
Design <- function(metadata, Group_var = NULL, Pre_processed_Data, Sample_Time, Sample_ID) {

  if (inherits(Pre_processed_Data,"TransformedData")) {
    Pre_processed_Data <- Pre_processed_Data[[1]]
  }

  colnames(metadata)[colnames(metadata) == Sample_Time] <- 'Time'
  colnames(metadata)[colnames(metadata) == Sample_ID] <- 'ID'

  Time <- metadata$Time
  ID <- metadata$ID

  if (!is.null(Group_var)) {
    selected_columns <- metadata[, Group_var, drop = FALSE]

    selected_columns <- lapply(selected_columns, factor)

    interaction_terms <- interaction(selected_columns, drop = TRUE)

    design_matrix <- stats::model.matrix(~interaction_terms - 1)

    colnames(design_matrix) <- levels(interaction_terms)

    Data_for_Reg <- as.data.frame(design_matrix) %>%
      cbind(Time, .) %>%
      cbind(Pre_processed_Data, .)
    Data_for_Reg$ID <- ID
  } else {
    Data_for_Reg <- cbind(Pre_processed_Data, Time)
  }

  structure(
    list(
      data = Data_for_Reg,
      params = list(
        Group_var = Group_var,
        Sample_Time = Sample_Time,
        Sample_ID = Sample_ID
      )
    ),
    class = "Design"
  )
}



#' @title Fit Spline Regression Models with Knot Selection
#' @description
#' The \code{Reg.SPLR} function fits natural spline regression models to time-series OTU data for each subgroup
#'     defined by design dummy variables. It uses \code{\link[mgcv]{gam}} to model the relationship between OTU
#'     abundance and time, incorporating a z-score based outlier filtering step (optional) and selecting the optimal number
#'     of knots via the Generalized Cross-Validation (GCV) criterion when not provided by the user.
#' @details
#' \code{Reg_SPLR} utilize Generalised Additive Model (GAM,see \code{\link[mgcv]{gam}}) to fit natural splines. Here, let \eqn{x_{ni}(t)} denote
#'     the transformed abundance of microbial feature \eqn{i} at time point \eqn{t}. To explain the evolution of microbial abundance along the
#'     time (\eqn{t}), the following generalized additive model is considered
#'     \deqn{x_{ni}(t) = \beta_{n0} + \sum_{k=1}^K \beta_{nk} \cdot N_{nk}(t) + \epsilon_{ni}}
#'     where \eqn{\beta_{n0}} is the intercept term, \eqn{\beta_{nk}} denotes the regression coefficients for the k-th natural spline basis function
#'     \eqn{N_{nk}(t)}, and \eqn{\epsilon_{ni}} is the error term. The parameter \eqn{K} refers to the number of basis functions, which is equal to the
#'     number of knots plus one. (i.e., \eqn{K = number of knots + 1}).
#'
#' If a z-score threshold is specified via \code{z_score}, observations with absolute z-scores exceeding this threshold are removed prior to model
#'     fitting. When the OTU has more unique values than specified by \code{unique_values}, the function searches over a range of knot numbers
#'     (from 1 up to \code{max_Knots}) to select the optimal model based on the GCV criterion; if \code{Knots} is provided, it is used directly
#'     in constructing the natural spline basis. The fitted models and their corresponding knots information are stored in a nested list
#'     structure and returned.
#'
#'
#' @param Data_for_Reg The data frame output from the \code{\link[MicrobTiSDA]{Design}}.
#' @param pre_processed_data The transformed data output from the \code{\link[MicrobTiSDA]{Data.trans}} function. A
#'     pre-processed OTU data frame with sample IDs as row names and OTU IDs as column names.
#' @param z_score A numeric value specifying the z-score threshold for outlier filtering; if \code{NA}, no
#'     outlier filtering is performed (default: \code{NA}).
#' @param unique_values An integer specifying the minimum number of unique values required in an OTU for spline fitting (default: 5).
#' @param Knots An optional numeric vector specifying the knots to use in the spline regression. If \code{NULL}, the optimal number of
#'     knots is determined by minimizing the GCV criterion (default: \code{NULL}).
#' @param max_Knots An integer indicating the maximum number of knots to consider when selecting the optimal spline model (default: 5).
#' @param seed Random seed.
#'
#' @return An object of class \code{"MicrobTiSDA_spline_regression"} containing:
#'   \item{fitted_model}{A nested list of the fitted spline regression models for each design dummy
#'     variable and OTU.}
#'   \item{knots_info}{Contains the corresponding knots information for each model.}
#'
#' @export
#' @import mgcv splines
#'
#' @examples
#' \donttest{
#' # Example metadata with grouping variables
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' result <- Reg.SPLR(design_data,
#'                   Pre_processed_Data,
#'                   z_score = 2,
#'                   unique_values = 5,
#'                   Knots = NULL,
#'                   max_Knots = 5)
#' # Access the fitted model for a particular design dummy variable and OTU:
#' fitted_model_example <- result$fitted_model[['Group1']][['OTU_name']]
#' }
Reg.SPLR <- function(Data_for_Reg,
                     pre_processed_data,
                     z_score = NA,
                     unique_values = 5,
                     Knots = NULL,
                     max_Knots = 5,
                     seed = 123) {

  if (is.object(Data_for_Reg)) {
    Data_for_Reg <- Data_for_Reg[[1]]
  }

  if (is.object(pre_processed_data)) {
    pre_processed_data <- pre_processed_data[[1]]
  }

  inde_vars <- setdiff(colnames(Data_for_Reg), colnames(pre_processed_data))
  inde_vars_data <- Data_for_Reg[inde_vars]
  fit_data <- list()
  knots_info <- list()

  for (i in colnames(inde_vars_data[,-c(1,ncol(inde_vars_data))])) {
    if (!is.null(seed)) {
      set.seed(seed)
    }
    GCV <- rep(0, max_Knots)
    Sub_Data <- cbind(pre_processed_data, inde_vars_data[1], inde_vars_data[i])
    Sub_Data <- Sub_Data[Sub_Data[,i] == 1,]

    for (j in colnames(pre_processed_data)) {
      Sub_Data_for_fit <- cbind(Sub_Data[j], Sub_Data[ncol(Sub_Data)-1])

      if (!is.na(z_score)) {
        z_scores <- scale(Sub_Data_for_fit[1])
        outliers_idx <- which(abs(z_scores) > z_score)
        Sub_data_filout <- Sub_Data_for_fit[-outliers_idx,]
      } else {
        Sub_data_filout <- Sub_Data_for_fit
      }

      if (length(unique(Sub_data_filout[,j])) > unique_values) {

        if (is.null(Knots)) {
          for (k in 1:max_Knots) {
            formula <- stats::as.formula(paste(j, " ~ splines::ns(Time, df = ", 1+k, ")", sep = ""))
            sp_model <- mgcv::gam(formula = formula, data = Sub_data_filout)
            GCV[k] <- summary(sp_model)$sp.criterion[[1]]
          }
          best_knots <- c(1:max_Knots)[which.min(GCV)]
          knots <- attr(splines::ns(Sub_data_filout$Time, df = 1+best_knots), "knots")
          knots_info[[i]][[j]] <- knots
          best_formula <- stats::as.formula(paste(j, " ~ splines::ns(Time, df = ", 1+best_knots, ")", sep = ""))
          fit_data[[i]][[j]] <- mgcv::gam(formula = best_formula, data = Sub_data_filout)
        } else {
          Knots_str <- paste(Knots, collapse = ",")
          formula <- stats::as.formula(paste(j, " ~ splines::ns(Time, knots = c(", Knots_str, "))", sep = ''))
          fit_data[[i]][[j]] <- mgcv::gam(formula = formula, data = Sub_data_filout)
          knots_info[[i]][[j]] <- Knots
        }

      } else {
        fit_data[[i]][[j]] <- NULL
        knots_info[[i]][[j]] <- NULL
      }
    }
  }

  RegSPLR_object <- function(fit_data, knots_info, params = list()) {
    obj <- list(
      fitted_model = fit_data,
      knots_info_each_model = knots_info,
      params = params
    )
    class(obj) <- "MicrobTiSDA_spline_regression"
    return(obj)
  }

  fit_result <- RegSPLR_object(
    fit_data = fit_data,
    knots_info = knots_info,
    params = list(
      z_score = z_score,
      unique_values = unique_values,
      Knots = Knots,
      max_Knots = max_Knots
    )
  )

  return(fit_result)
}



#' @title Predict Time-Series Data from Fitted Spline Models
#' @description
#' This function generates predicted OTU abundances at new time points using previously fitted spline regression models.
#'     For each subject defined in the fitted models, it extracts the time range from the metadata, creates a new sequence of time points
#'     with a specified interval, and uses the corresponding fitted models to predict values at these new time points.
#'
#' @details
#' The function accepts a list of fitted models (typically the output from \code{\link[MicrobTiSDA]{Reg.SPLR}}) and sample metadata.
#'     For each subject (as defined by the \code{Group} parameter), it extracts the subset of metadata corresponding to that subject. The minimum and maximum time
#'     values are determined, and a new sequence of time points is generated using the provided \code{time_step}. For each OTU model of the
#'     subject, predictions are obtained via the \code{predict} function applied to the new time points, and the results are compiled into a data
#'     frame with an additional column \code{Predicted_Time} indicating the prediction time. The output is a list of data frames, each corresponding
#'     to a subject, containing the predicted OTU abundances.
#'
#' @param Fitted_models A list object generated by \code{\link[MicrobTiSDA]{Reg.SPLR}}.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param Group A character string indicating the column name in \code{metadata} that defines subject information for prediction.
#' @param time_step A numeric value specifying the interval between new time points in the prediction sequence.
#' @param Sample_Time A character string indicating the column name in \code{metadata} that contains sample time information.
#'
#' @return An object of class \code{PredictedData} which contains the predict microbial feature abundances at the new time points, with rows
#'     labeled by the group and time (formatted as \code{Group_T_Time}) and an additional column \code{Predicted_Time}.
#' @export
#' @author Shijia Li
#'
#' @examples
#' \donttest{
#' # Example metadata with grouping variables
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' reg <- Reg.SPLR(design_data,
#'                   Pre_processed_Data,
#'                   z_score = 2,
#'                   unique_values = 5,
#'                   Knots = NULL,
#'                   max_Knots = 5)
#' predictions <- Pred.data(reg,
#'                         metadata,
#'                         Group = "GroupA",
#'                         time_step = 1,
#'                         Sample_Time = "TimePoint")
#' }
Pred.data <- function(Fitted_models, metadata, Group, time_step, Sample_Time) {

  predicted_data <- list()
  fitted_models <- Fitted_models$fitted_model

  for (i in names(fitted_models)) {
    colnames(metadata)[colnames(metadata) == Sample_Time] <- 'Time'
    meta_group <- metadata[rownames(metadata[metadata[Group] == i,]),]
    start_time <- min(meta_group['Time'])
    end_time <- max(meta_group['Time'])
    New_time <- seq(start_time,end_time,time_step)

    pred_data <- as.data.frame(matrix(0,length(New_time),
                                      length(fitted_models[[i]])))
    rownames(pred_data) <- paste(i,"_T_",New_time,sep = "")
    colnames(pred_data) <- names(fitted_models[[i]])

    for (j in 1:length(fitted_models[[i]])) {
      pred_data[,j] <- stats::predict(fitted_models[[i]][[j]],
                               data.frame(Time = New_time))
    }

    pred_data <- as.data.frame(pred_data) %>%
      mutate(.,Predicted_Time = New_time)

    predicted_data[[i]] <- pred_data
  }
  class(predicted_data) <- "PredictedData"

  return(predicted_data)
}



#' @title Cluster OTU Time-Series Data Based on Regression Model prediction and Generate Dendrogram Plots
#'
#' @description
#' This function performs hierarchical clustering on predicted OTU time-series data for different groups
#'     and generates corresponding dendrogram plots. For each group in the input list, the function computes a
#'     correlation-based distance matrix, performs hierarchical clustering using the specified clustering method
#'     (e.g. \code{average}), and then converts the result into a dendrogram.
#'
#' @details
#' For each group in the input \code{predicted_data}, the function first extracts the predicted OTU data (excluding the
#'     last column, which is assumed to contain time information) and computes a correlation matrix, which is converted
#'     into a distance matrix via
#'     \deqn{d_{\text{corr}}(x,y) = 1-\frac{{\sum_{i=1}^{n}(x_i-\bar{y})}}{{\sqrt{{\sum_{i=1}^{n}(x_i-\bar{x})^2}} \sqrt{{\sum_{i=1}^{n}(y_i-\bar{y})^2}}}}}
#'     where \eqn{x} and \eqn{y} represent the two OTU time series being compared, \eqn{n} denotes the total number of time points, and
#'     \eqn{\bar{x}} and \eqn{\bar{y}} denote the means of the respective time series. Hierarchical clustering is
#'     performed on the above distance matrix using the method specified in \code{clust_method}.
#'
#'
#'
#' @param predicted_data The output data frame from the \code{\link[MicrobTiSDA]{Pred.data}}.
#' @param clust_method A string, the agglomeration method to be used. This argument should be one of "ward.D", "ward.D2", "single",
#'     "complete", "average", "mcquitty", "median", "centroid". Detail see \code{\link[stats]{hclust}}.
#' @param font_size A numeric value specifying the font size for text labels in the dendrogram plots (default: \code{0.2}).
#' @param dend_title_size A numeric value specifying the font size of the dendrogram plot title (default: \code{15}).
#'
#' @return An object of \code{MicrobTiSDA.cluster} with three elements:
#' \describe{
#'   \item{predicted_data}{The original input list of predicted data.}
#'   \item{cluster_results}{A list of hierarchical clustering objects (one per group).}
#'   \item{cluster_figures}{A list of ggplot2 objects containing the dendrogram plots for each group.}
#' }
#'
#' @export
#' @author Shijia Li
#' @importFrom ggplot2 ggplot
#' @importFrom cluster silhouette
#'
#' @examples
#' \donttest{
#' # Example metadata with grouping variables
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' reg <- Reg.SPLR(design_data,
#'                   Pre_processed_Data,
#'                   z_score = 2,
#'                   unique_values = 5,
#'                   Knots = NULL,
#'                   max_Knots = 5)
#' predictions <- Pred.data(reg,
#'                         metadata,
#'                         Group = "GroupA",
#'                         time_step = 1,
#'                         Sample_Time = "TimePoint")
#' result <- Data.cluster(predicted_data = predictions,
#'                        clust_method = "average",
#'                        font_size = 0.2,
#'                        dend_title_size = 15)
#'
#' }
Data.cluster <- function(predicted_data,
                         clust_method = "complete",
                         font_size = 0.2,
                         dend_title_size = 15) {

  if (inherits(predicted_data,"PredictedDataMESR")) {
    predicted_data <- predicted_data[[1]]
  }

  cluster_results <- list()
  cluster_figures <- list()

  for (i in names(predicted_data)) {

    pred_data <- as.data.frame(predicted_data[[i]][, -ncol(predicted_data[[i]])])

    if (is.null(pred_data) || (!is.data.frame(pred_data) && !is.matrix(pred_data)) || ncol(pred_data) < 3) {
      cluster_results[[i]] <- NULL
      message("Group ", i, " was excluded due to having fewer than 3 Features, making it unsuitable for clustering.")
    } else {
      cor_matrix <- cor(pred_data)
      pred_data_dist <- as.dist((1 - cor_matrix))

      pred_data_dist_hc <- stats::hclust(pred_data_dist, clust_method)

      dend <- stats::as.dendrogram(pred_data_dist_hc)
      cluster_figures[[i]] <- ggplot() +
        geom_segment(data = ggdendro::dendro_data(dend, type = "rectangle")$segments,
                     aes(x = .data$x, y = .data$y, xend = .data$xend, yend = .data$yend)) +
        geom_text(data = ggdendro::dendro_data(dend, type = "rectangle")$labels,
                  aes(x = .data$x, y = .data$y, label = label),
                  size = font_size, hjust = 0) +
        coord_flip() + scale_y_reverse(expand = c(0.2, 0)) +
        labs(title = paste("Feature profiles clustering plot for group", i)) +
        theme(axis.line.y = element_blank(),
              axis.ticks.y = element_blank(),
              axis.title.x = element_blank(),
              axis.text.y = element_blank(),
              axis.title.y = element_blank(),
              panel.background = element_rect(fill = "white"),
              panel.grid = element_blank(),
              plot.title = element_text(size = dend_title_size))

      cluster_results[[i]] <- pred_data_dist_hc
    }
  }

  cluster_data <- list(
    predicted_data = predicted_data,
    cluster_results = cluster_results,
    cluster_figures = cluster_figures
  )
  class(cluster_data) <- "MicrobTiSDA.cluster"
  return(cluster_data)
}



#' @title Cut and Annotate Dendrogram Based on a Specified Cut-Off Height
#' @description
#' This function processes clustering outputs from \code{\link[MicrobTiSDA]{Data.cluster}} to update dendrogram plots.
#'     Depending on the user’s preference, it either automatically determines the optimal number of clusters via
#'     silhouette analysis or uses a user-specified cut-off height.
#'
#' @details
#' The function takes as input a list containing predicted data and clustering results (typically generated by another function) from
#'     \code{\link[MicrobTiSDA]{Data.cluster}}, and then computes a correlation-based distance matrix for each group. If \code{auto_cutree}
#'     is \code{TRUE}, the function performs a repeated k-fold cross-validation by iterating over a range of potential cluster numbers
#'     and computing the average silhouette width, thereby determining the optimal number of clusters. The dendrogram is then cut accordingly,
#'     and the resulting clusters are used to annotate the dendrogram plot with different colors for each cluster.
#'
#' If \code{auto_cutree} is \code{FALSE}, the function uses the provided \code{cut_height} to cut the dendrogram. It then assigns cluster
#'     emberships based on this cut-off and updates the dendrogram plot by adding a horizontal dashed line at the specified cut-off and annotating
#'     the plot with the cut-off value. In both cases, the function prints the dendrogram plot for each group and returns a list containing the
#'     clustering results and the corresponding ggplot2 objects of the dendrograms.
#'
#'
#' @param cluster_outputs The output object of \code{\link[MicrobTiSDA]{Data.cluster}}.
#' @param cut_height A numeric value specifying the cut-off height for cutting the dendrogram when \code{auto_cutree} is \code{FALSE}.
#' @param cut_height_dist A numeric value used to adjust the vertical distance of the cut-off line annotation in the dendrogram
#'     plot (default: \code{0.2}).
#' @param auto_cutree Logical; if \code{TRUE}, the function automatically determines the optimal number of clusters based on
#'     silhouette width (default: \code{FALSE}).
#' @param font_size A numeric value specifying the font size for text labels in the dendrogram plots (default: \code{0.2}).
#'
#' @return A object of \code{MicrobTiSDA.clusterCut} with two elements:
#' \describe{
#'   \item{cluster_results}{A list of clustering objects for each group.}
#'   \item{cluster_figures}{A list of ggplot2 objects containing the annotated dendrogram plots for each group.}
#' }
#'
#' @importFrom stats as.dist cor hclust as.dendrogram
#' @import ggplot2
#' @import ggdendro
#' @importFrom utils read.table
#' @export
#' @author Shijia Li
#'
#' @examples
#' \donttest{
#' # Example metadata with grouping variables
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' reg <- Reg.SPLR(design_data,
#'                   Pre_processed_Data,
#'                   z_score = 2,
#'                   unique_values = 5,
#'                   Knots = NULL,
#'                   max_Knots = 5)
#' predictions <- Pred.data(reg,
#'                         metadata,
#'                         Group = "GroupA",
#'                         time_step = 1,
#'                         Sample_Time = "TimePoint")
#' result <- Data.cluster(predicted_data = predictions,
#'                        clust_method = "average",
#'                        font_size = 0.2,
#'                        dend_title_size = 15)
#'
#' result <- Data.cluster.cut(cluster_outputs = result,
#'                           cut_height = 0.3,
#'                           cut_height_dist = 0.2,
#'                           auto_cutree = FALSE)
#'
#' # To automatically determine the optimal number of clusters:
#' result_auto <- Data.cluster.cut(cluster_outputs = result, auto_cutree = TRUE)
#' }
Data.cluster.cut <- function(cluster_outputs,
                             cut_height,
                             cut_height_dist = 0.2,
                             font_size = 0.2,
                             auto_cutree = FALSE) {

  cluster_results <- cluster_outputs$cluster_results
  predict_data <- cluster_outputs$predicted_data
  cluster_figures <- list()
  cut_assignments <- list()

  for (i in names(predict_data)) {

    pred_data <- as.data.frame(predict_data[[i]][, -ncol(predict_data[[i]])])
    cor_matrix <- cor(pred_data)
    pred_data_dist <- as.dist((1 - cor_matrix))

    if (auto_cutree) {
      max_k <- ncol(pred_data) - 1
      sil_width <- numeric(max_k)
      for (k in 2:max_k) {
        cutree_k <- stats::cutree(cluster_results[[i]], k)
        sil <- cluster::silhouette(cutree_k, pred_data_dist)
        sil_width[k] <- mean(sil[, 3])
      }
      optimal_k <- which.max(sil_width)
      message("Number of optimal clusters for group ", i, ": ", optimal_k)

      dend <- stats::as.dendrogram(cluster_results[[i]])
      clusters <- stats::cutree(cluster_results[[i]], optimal_k)[stats::order.dendrogram(dend)]
      clusters.df <- data.frame(label = names(clusters), cluster = factor(clusters))
      cut_assignments[[i]] <- clusters.df

      dend_data <- ggdendro::dendro_data(dend, type = "rectangle")
      dend_data[["labels"]] <- merge(dend_data[["labels"]], clusters.df, by = "label")

      cluster_figures[[i]] <- ggplot() +
        geom_segment(data = dend_data$segments,
                     aes(x = .data$x, y = .data$y, xend = .data$xend, yend = .data$yend)) +
        geom_text(data = dend_data$labels,
                  aes(x = .data$x, y = .data$y, label = label, color = cluster),
                  size = font_size, hjust = 0) +
        coord_flip() + scale_y_reverse(expand = c(0.2, 0)) +
        labs(title = paste("Feature profiles clustering plot for group", i,
                           "\nSuggested optimal cluster:", optimal_k)) +
        theme(axis.line.y = element_blank(),
              axis.ticks.y = element_blank(),
              axis.title.x = element_blank(),
              axis.text.y = element_blank(),
              axis.title.y = element_blank(),
              panel.background = element_rect(fill = "white"),
              panel.grid = element_blank(),
              plot.title = element_text(hjust = 0.5)) +
        scale_color_discrete(name = "Cluster")

    } else {
      dend <- stats::as.dendrogram(cluster_results[[i]])
      cut_height <- as.numeric(cut_height)

      clusters <- stats::cutree(cluster_results[[i]], h = cut_height)
      clusters_ordered <- clusters[stats::order.dendrogram(dend)]
      clusters_df <- data.frame(label = names(clusters_ordered),
                                cluster = factor(clusters_ordered))
      cut_assignments[[i]] <- clusters_df

      dend_data <- ggdendro::dendro_data(dend, type = "rectangle")
      dend_data[["labels"]] <- merge(dend_data[["labels"]], clusters_df, by = "label", all.x = TRUE)

      cluster_figures[[i]] <- ggplot() +
        geom_segment(data = dend_data$segments,
                     aes(x = x, y = y, xend = xend, yend = yend)) +
        geom_text(data = dend_data$labels,
                  aes(x = x, y = y, label = label, color = cluster),
                  size = font_size, hjust = 0) +
        coord_flip() + scale_y_reverse(expand = c(0.2, 0)) +
        labs(title = paste("Feature profiles clustering plot for group", i)) +
        theme(axis.line.y = element_blank(),
              axis.ticks.y = element_blank(),
              axis.title.x = element_blank(),
              axis.text.y = element_blank(),
              axis.title.y = element_blank(),
              panel.background = element_rect(fill = "white"),
              panel.grid = element_blank(),
              plot.title = element_text(hjust = 0.5)) +
        scale_color_discrete(name = "Cluster") +
        geom_hline(yintercept = cut_height, color = "blue", linetype = "dashed") +
        annotate("text", x = 3, y = cut_height + cut_height_dist,
                 label = paste("Cut-off line:", cut_height), color = "blue")
    }
  }

  cluster_cut <- list(
    cluster_results = cluster_results,
    cut_assignments = cut_assignments,
    cluster_figures = cluster_figures,
    auto_cutree = auto_cutree
  )
  class(cluster_cut) <- "MicrobTiSDA.clusterCut"
  return(cluster_cut)
}



#' @title Visualize Temporal OTU Profiles from Clustered Predicted Data
#' @description
#' The \code{Data.visual} function generates visualizations of temporal profiles for OTUs by integrating clustering results,
#' predicted time-series data, and design information. It produces ggplot2 figures for each group and for each cluster branch,
#' displaying smoothed curves of predicted OTU abundances over time. Optionally, the function overlays raw data points and fits linear
#' models to assess temporal trends, annotating the plots with model statistics when certain criteria are met.
#'
#' @details
#' This function uses hierarchical clustering results (obtained from a dendrogram) to cut the tree either by a specified height or
#'     by a user specified number of branches of each dendrogram in \code{cluster_results}. For each group in \code{cluster_results},
#'     the function extracts the corresponding predicted OTU data and raw design data. Temporal profiles are visualized by plotting
#'     smooth curves (using \code{stat_smooth}) for each cluster branch. When \code{plot_dots} is set to \code{TRUE}, the function
#'     overlays raw data points. Additionally, if \code{plot_lm} is \code{TRUE}, a linear model is fitted to the predicted data,
#'     and if the model meets specified thresholds for R-squared (\code{lm_R2}) and absolute slope (\code{lm_abs_slope})
#'     (i.e., R2 > 0.1 and absolute slope > 0.05), a dashed regression line is added along with an annotation of
#'     the R-squared and slope values. The resulting list of ggplot2 objects can be used to visually inspect the temporal dynamics of
#'     OTUs across different clusters and groups.
#'
#'
#' @param cluster_results A list object output from the \code{\link[MicrobTiSDA]{Data.cluster}}).
#' @param cutree_by A character string specifying the method to cut the dendrogram, either by \code{"height"} or by \code{"branches"}.
#' @param cluster_height A numeric vector specifying the cut-off height for each group when \code{cutree_by = "height"}.
#' @param cluster_branches A numeric vector specifying the number of clusters for each group when \code{cutree_by = "branches"}.
#' @param predicted_data The output data frame from the \code{\link[MicrobTiSDA]{Pred.data}}).
#' @param Design_data The output data from the \code{\link[MicrobTiSDA]{Design}}).
#' @param pre_processed_data The transformed data output from the \code{\link[MicrobTiSDA]{Data.trans}} function. A
#'     pre-processed OTU data frame with sample IDs as row names and OTU IDs as column names.
#' @param Taxa A data frame providing taxonomic annotations for microbial species.
#' @param plot_dots Logical; if \code{TRUE}, raw data points are overlaid on the temporal curves (default: \code{TRUE}).
#' @param figure_x_scale A numeric value specifying the interval for x-axis breaks in the figures (default: \code{5}).
#' @param plot_lm Logical; if \code{TRUE}, a linear model is fitted to the predicted data to detect trends, and the regression line
#'     is added (default: \code{FALSE}).
#' @param lm_R2 A numeric threshold for the minimum R-squared value required to annotate the linear model (default: \code{0.01}).
#' @param lm_abs_slope A numeric threshold for the minimum absolute slope required to annotate the linear model (default: \code{0.005}).
#' @param title_size A numeric value specifying the font size for the plot title (default: \code{10}).
#' @param axis_title_size A numeric value specifying the font size for the axis titles (default: \code{8}).
#' @param axis_y_size A numeric value specifying the font size for the y-axis text (default: \code{5}).
#' @param axis_x_size A numeric value specifying the font size for the x-axis text (default: \code{5}).
#' @param lm_sig_size A numeric value specifying the font size for linear model annotation text (default: \code{5}).
#' @param legend_title_size A numeric value specifying the font size for legend titles (default: \code{5}).
#' @param legend_text_size A numeric value specifying the font size for legend text (default: \code{5}).
#' @param dots_size A numeric value specifying the size of the overlaid raw data points (default: \code{0.7}).
#' @importFrom ggplot2 ggplot
#' @return An object of class \code{MicrobTiSDA.visual} which contains the list of visualizations of clustered microbial features.
#' @export
#'
#' @examples
#' \donttest{
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' reg <- Reg.SPLR(design_data,
#'                   Pre_processed_Data,
#'                   z_score = 2,
#'                   unique_values = 5,
#'                   Knots = NULL,
#'                   max_Knots = 5)
#' predictions <- Pred.data(reg,
#'                         metadata,
#'                         Group = "GroupA",
#'                         time_step = 1,
#'                         Sample_Time = "TimePoint")
#' result <- Data.cluster(predicted_data = predictions,
#'                        clust_method = "average",
#'                        font_size = 0.2,
#'                        dend_title_size = 15)
#'
#' result <- Data.cluster.cut(cluster_outputs = result,
#'                           cut_height = 0.3,
#'                           cut_height_dist = 0.2,
#'                           auto_cutree = FALSE)
#'
#' curves <- Data.visual(cluster_results = result,
#'                       cutree_by = "height",
#'                       cluster_height = c(0.2,0.2),
#'                       cluster_branches = NA,
#'                       predicted_data = predictions,
#'                       Design_data = design_data,
#'                       pre_processed_data = Pre_processed_Data,
#'                       Taxa = NULL,
#'                       plot_dots = TRUE)
#' }
Data.visual <- function(cluster_results,
                        cutree_by = "height",
                        cluster_height = NA,
                        cluster_branches = NA,
                        predicted_data,
                        Design_data,
                        pre_processed_data,
                        Taxa = NULL,
                        plot_dots = TRUE,
                        figure_x_scale = 5,
                        plot_lm = FALSE,
                        lm_R2 = 0.01,
                        lm_abs_slope = 0.005,
                        title_size = 10,
                        axis_title_size = 8,
                        axis_y_size = 5,
                        axis_x_size = 5,
                        lm_sig_size = 5,
                        legend_title_size = 5,
                        legend_text_size = 5,
                        dots_size = 0.7) {

  curves_plot <- list()
  if (inherits(cluster_results,"MicrobTiSDA.clusterCut")) {
    cluster_results <- cluster_results[[1]]
  }

  if (inherits(Design_data,"Design")) {
    Design_data <- Design_data[[1]]
  }

  if (inherits(pre_processed_data,"TransformedData")) {
    pre_processed_data <- pre_processed_data[[1]]
  }

  n <- 0
  for (i in names(cluster_results)) {
    n <- n + 1
    if (cutree_by == 'height') {
      cutrees <- stats::cutree(cluster_results[[i]], h = cluster_height[n])
    } else if (cutree_by == 'branches') {
      cutrees <- stats::cutree(cluster_results[[i]], k = cluster_branches[n])
    } else {
      stop("cutree_by must be either 'height' or 'branches'")
    }

    rawdata <- Design_data[Design_data[,i] == 1, ]

    ylim_min <- min(min(pre_processed_data),
                    min(predicted_data[[i]][,-ncol(predicted_data[[i]])]))
    ylim_max <- max(max(pre_processed_data),
                    max(predicted_data[[i]][,-ncol(predicted_data[[i]])]))

    curves_plot[[i]] <- list()

    Pred_data_for_plot <- predicted_data[[i]]
    for (j in as.numeric(levels(as.factor(cutrees)))) {
      cols_to_remove <- names(which(cutrees != j))
      data_for_plot <- Pred_data_for_plot[,!colnames(Pred_data_for_plot) %in% cols_to_remove]

      if(!is.null(Taxa)) {
        Annotate <- Taxa[match(colnames(data_for_plot)[-ncol(data_for_plot)], rownames(Taxa)), ]
        Annotate <- Annotate %>% mutate(across(where(is.factor), as.character)) %>% mutate(across(everything(), ~na_if(.,"")))
        Taxon <- apply(Annotate, 1, function(x) { tail(na.omit(x), 1) })
        Taxon <- data.frame(Species_annotation = Taxon)
        colnames(data_for_plot) <- c(paste(colnames(data_for_plot)[-ncol(data_for_plot)],
                                           " (", Taxon[, 1], ")", sep = ""),
                                     "Predicted_Time")
      }

      data_for_plot_long <- data_for_plot %>%
        tidyr::pivot_longer(cols = -Predicted_Time,names_to = "Features",values_to = "Abundance")

      curves_plot[[i]][[j]] <- ggplot(data_for_plot_long, aes(x = Predicted_Time, y = Abundance, color = Features)) +
        stat_smooth(se = FALSE) +
        labs(title = paste(i,"_cluster_",j,sep = ""),
             x = "Time",y = "MCLR transformed Abundance")+
        theme_bw()+
        theme(title = element_text(size = title_size),
              axis.title = element_text(size = axis_title_size),
              axis.text.y = element_text(size = axis_y_size),
              axis.text.x = element_text(size = axis_x_size),
              legend.title = element_text(size = legend_title_size),
              legend.text = element_text(size = legend_text_size))+
        scale_x_continuous(breaks = seq(1,max(data_for_plot_long$Predicted_Time),figure_x_scale),
                           labels = seq(1,max(data_for_plot_long$Predicted_Time),figure_x_scale))+
        ylim(ylim_min-2,ylim_max+2)

      if (plot_dots) {
        cols_to_reserve <- names(which(cutrees == j))
        point_data_for_plot <- rawdata[colnames(rawdata) %in% cols_to_reserve]
        point_data_for_plot$Time <- rawdata$Time

        if (!is.null(Taxa)) {
          Annotate <- Taxa[match(colnames(point_data_for_plot)[-ncol(point_data_for_plot)],
                                 rownames(Taxa)),]
          Annotate <- Annotate %>% mutate(across(where(is.factor), as.character)) %>% mutate(across(everything(), ~na_if(.,"")))
          Taxon <- apply(Annotate,1,function(x) {tail(na.omit(x),1)})
          Taxon <- data.frame(Species_annotation = Taxon)
          colnames(point_data_for_plot) <- c(paste(colnames(point_data_for_plot)[-ncol(point_data_for_plot)],
                                                   " (",Taxon[,1],")",sep = ""),
                                             "Time")
        }

        point_data_for_plot_long <- point_data_for_plot %>%
          tidyr::pivot_longer(cols = -Time,names_to = "Features",values_to = "Abundance")

        curves_plot[[i]][[j]] <- curves_plot[[i]][[j]] +
          geom_point(data = point_data_for_plot_long,
                     aes(x = Time, y = Abundance, color = Features),
                     size = dots_size)
      }

      if (plot_lm) {
        lm_model <- stats:lm(Abundance ~ Predicted_Time,data = data_for_plot_long)
        lm_model_summary <- summary(lm_model)
        r_squared <- lm_model_summary$r.squared
        slope <- as.numeric(lm_model$coefficients[2])
        p_value <- lm_model_summary$coefficients[, "Pr(>|t|)"]
        slope_p <- as.numeric(p_value[2])
        if (slope_p < 0.05 && r_squared > lm_R2 && abs(slope) > lm_abs_slope) {
          curves_plot[[i]][[j]] <- curves_plot[[i]][[j]]+
            stat_smooth(data = data_for_plot_long,aes(x = Predicted_Time,y = Abundance),
                        method = 'lm',linetype = 'dashed',se = TRUE,color = 'darkgrey')+
            annotate("text",x=-Inf,y=Inf,label = paste("Slope=",round(slope,4)),
                     hjust = -1.1, vjust = 1.5,size = lm_sig_size,color = 'darkgray')
        }
      }
    }
  }

  result <- list(
    plots = curves_plot,
    params = list(
      cutree_by = cutree_by,
      cluster_height = cluster_height,
      cluster_branches = cluster_branches,
      plot_dots = plot_dots,
      plot_lm = plot_lm
    )
  )
  class(result) <- "MicrobTiSDA.visual"

  return(result)
}



#' @title Visualize Opposing Temporal profiles among Microbial features.
#' @description
#' This function identifies and visualizes OTUs/ASVs that exhibit opposing temporal trends based on a correlation threshold.
#'     It computes the correlation matrix of predicted OTU/ASV time-series data, selects those OTUs with correlations below a specified threshold,
#'     and generates smoothed temporal profile plots. Optionally, raw data points are overlaid and taxonomic annotations are added if provided.
#'
#' @details
#' For each group in the \code{predicted_data} list, the function first removes the time column and computes a correlation matrix
#' from the predicted OTU data. It then extracts, for each OTU, the subset of OTUs that show a correlation lower than the specified threshold
#' (\eqn{\text{ng\_cor\_thres}}). If an OTU does not have any opposing trends (i.e., all correlations exceed the threshold), it is skipped.
#' For those OTUs meeting the criteria, the function restructures the data into long format and plots the temporal profiles using
#' \code{geom_smooth} to display the primary trend (solid line) and opposing trends (dashed lines). If \code{plot_dots} is \code{TRUE},
#' the raw data points extracted from the design data are also overlaid. When taxonomic annotations are provided via \code{Taxa},
#' OTU labels are augmented with species information. The x-axis is scaled according to \code{figure_x_scale}, and plot aesthetics
#' (titles, axis text, legends, and dot sizes) can be customized using the respective parameters.
#'
#' @param predicted_data The output data frame from the \code{\link[MicrobTiSDA]{Pred.data}}).
#' @param pre_processed_data The transformed data output from the \code{\link[MicrobTiSDA]{Data.trans}} function. A
#'     pre-processed OTU data frame with sample IDs as row names and OTU IDs as column names.
#' @param Design_data The output data from the \code{\link[MicrobTiSDA]{Design}}).
#' @param ng_cor_thres A numeric value specifying the correlation threshold below which OTUs are considered to exhibit opposing trends.
#' @param Taxa A data frame providing taxonomic annotations for microbial species.
#' @param plot_dots Logical; if \code{TRUE}, raw data points are overlaid on the temporal curves (default: \code{TRUE}).
#' @param figure_x_scale A numeric value specifying the interval for x-axis breaks in the figures (default: \code{5}).
#' @param title_size A numeric value specifying the font size for the plot title (default: \code{10}).
#' @param axis_title_size A numeric value specifying the font size for the axis titles (default: \code{8}).
#' @param axis_text_y_size A numeric value specifying the font size for the y-axis text (default: \code{5}).
#' @param axis_text_x_size A numeric value specifying the font size for the x-axis text (default: \code{5}).
#' @param legend_title_size A numeric value specifying the font size for legend titles (default: \code{5}).
#' @param legend_text_size A numeric value specifying the font size for legend text (default: \code{5}).
#' @param dots_size A numeric value specifying the size of the overlaid raw data points (default: \code{0.7}).
#'
#' @author Shijia Li
#' @return An object of class \code{DataOppCorVis} which contains the list of each targeted microbial feature.
#' @export
#'
Data.opp.cor.vis <- function(predicted_data,
                             pre_processed_data,
                             Design_data,
                             ng_cor_thres,
                             Taxa = NULL,
                             plot_dots = TRUE,
                             figure_x_scale = 5,
                             title_size = 10,
                             axis_title_size = 10,
                             axis_text_y_size = 8,
                             axis_text_x_size = 8,
                             legend_title_size = 10,
                             legend_text_size = 8,
                             dots_size = 0.6){

  if (inherits(pre_processed_data,"TransformedData")) {
    pre_processed_data <- pre_processed_data[[1]]
  }

  if (inherits(Design_data,"Design")) {
    Design_data <- Design_data[[1]]
  }

  curves_plot <- list()

  for (i in names(predicted_data)) {
    pred_data <- as.data.frame(predicted_data[[i]])
    rawdata <- Design_data[Design_data[,i] == 1, ]
    curves_plot[[i]] <- list()

    ylim_min <- min(min(pre_processed_data),
                    min(predicted_data[[i]][,-ncol(predicted_data[[i]])]))
    ylim_max <- max(max(pre_processed_data),
                    max(predicted_data[[i]][,-ncol(predicted_data[[i]])]))

    pred_data <- pred_data[,-ncol(pred_data)]
    cor_matrix <- cor(pred_data)
    oppo_cor <- as.data.frame(cor_matrix)

    for (j in colnames(oppo_cor)) {
      oppo_cor_otu <- oppo_cor[j]
      if (min(oppo_cor_otu) > ng_cor_thres) {
        curves_plot[[i]][[j]] = NULL
        next
      } else {
        oppo_otu <- subset(oppo_cor_otu,oppo_cor_otu[j] < ng_cor_thres)
        oppo_otu_vec <- c(colnames(oppo_otu),rownames(oppo_otu))

        data_for_plot <- pred_data[,oppo_otu_vec]
        data_for_plot$Predicted_Time <- predicted_data[[i]]$Predicted_Time

        if(!is.null(Taxa)) {
          Annotate <- Taxa[match(colnames(data_for_plot)[-ncol(data_for_plot)], rownames(Taxa)), ]
          Annotate <- Annotate %>%
            mutate(across(where(is.factor), as.character)) %>%
            mutate(across(everything(), ~na_if(.,"")))
          Taxon <- apply(Annotate, 1, function(x) { tail(na.omit(x), 1) })
          Taxon <- data.frame(Species_annotation = Taxon)
          colnames(data_for_plot) <- c(paste(colnames(data_for_plot)[-ncol(data_for_plot)],
                                             " (", Taxon[, 1], ")", sep = ""),
                                       "Predicted_Time")
        }

        data_for_plot_long <- data_for_plot %>%
          tidyr::pivot_longer(cols = -Predicted_Time,names_to = "Features",values_to = "Abundance")

        data_for_plot_long_1 <- subset(data_for_plot_long,
                                       data_for_plot_long$Features == colnames(data_for_plot[1]))
        data_for_plog_long_2 <- subset(data_for_plot_long,
                                       data_for_plot_long$Features != colnames(data_for_plot[1]))

        curves_plot[[i]][[j]] <- ggplot(data_for_plot_long_1,
                                        aes(x = Predicted_Time, y = Abundance, color = Features)) +
          geom_smooth(se = FALSE) +
          geom_smooth(data = data_for_plog_long_2,
                      aes(x = Predicted_Time, y = Abundance, color = Features),
                      se = FALSE, linetype = 'dashed') +
          labs(title = paste('Features in ', i, ' with opposite trends to ', j, sep = ""),
               x = "Time", y = "Predicted values") +
          theme_bw()+
          theme(plot.title = element_text(size = title_size),
                axis.title = element_text(size = axis_title_size),
                axis.text.y = element_text(size = axis_text_y_size),
                axis.text.x = element_text(size = axis_text_x_size),
                legend.title = element_text(size = legend_title_size),
                legend.text = element_text(size = legend_text_size)) +
          scale_x_continuous(breaks = seq(1, max(data_for_plot_long$Predicted_Time), figure_x_scale),
                             labels = seq(1, max(data_for_plot_long$Predicted_Time), figure_x_scale)) +
          ylim(ylim_min - 2, ylim_max + 2)

        if (plot_dots == TRUE) {
          point_data_for_plot <- rawdata[colnames(rawdata) %in% oppo_otu_vec]
          point_data_for_plot$Time <- rawdata$Time

          if(!is.null(Taxa)) {
            Annotate <- Taxa[match(colnames(point_data_for_plot)[-ncol(point_data_for_plot)], rownames(Taxa)), ]
            Annotate <- Annotate %>%
              mutate(across(where(is.factor), as.character)) %>%
              mutate(across(everything(), ~na_if(.,"")))
            Taxon <- apply(Annotate, 1, function(x) { tail(na.omit(x), 1) })
            Taxon <- data.frame(Species_annotation = Taxon)
            colnames(point_data_for_plot) <- c(paste(colnames(point_data_for_plot)[-ncol(point_data_for_plot)],
                                                     " (", Taxon[, 1], ")", sep = ""),
                                               "Time")
          }
          point_data_for_plot_long <- point_data_for_plot %>%
            tidyr::pivot_longer(cols = -Time,names_to = "Features",values_to = "Abundance")
          curves_plot[[i]][[j]] = curves_plot[[i]][[j]]+
            geom_point(data = point_data_for_plot_long,
                       aes(x = Time,
                           y = Abundance,
                           color = Features),size = dots_size)
        }
      }
    }

  }

  structure(list(curves_plot = curves_plot),
            class = "DataOppCorVis")
}



#' @title Random Forest classification for OTU/ASV Data
#'
#' @description
#' This function implements a random forest classification model tailored for OTU/ASV datasets.
#'     It performs data filtering, model training, performance evaluation, cross-validation, and
#'     biomarker (important microbial features) selection based on Mean Decrease Accuracy.
#' #' @details
#' The function processes the input OTU count data and corresponding metadata in several steps:
#' \enumerate{
#'   \item \strong{Data Filtering and Preparation:} If a minimum count threshold (\code{OTU_counts_filter_value}) is provided,
#'   OTUs with total counts below this value are removed. The OTU table is then transposed and merged with the metadata,
#'   where a specific column (specified by \code{Group}) indicates the group labels.
#'   \item \strong{Data Partitioning:} The combined dataset is split into training and testing subsets based on the proportion
#'   specified by \code{train_p}.
#'   \item \strong{Model Training:} A random forest classifier is trained on the training data. The function computes the
#'   margin scores for the training samples, which are plotted to visualize the model’s confidence.
#'   \item \strong{Performance Evaluation:} Predictions are made on both training and testing datasets. Confusion matrices
#'   are generated to compare the actual versus predicted classes.
#'   \item \strong{Feature Importance and Cross-Validation:} OTU importance is assessed using Mean Decrease Accuracy.
#'   Repeated k-fold cross-validation (default 10-fold repeated \code{reps} times) is performed to determine the optimal number
#'   of OTUs (biomarkers). A cross-validation error curve is plotted, and the user is prompted to input the best number
#'   of OTUs based on the plot.
#' }
#'
#' @param raw_data A numeric matrix or data frame of counts data with OTUs/ASVs as rows and samples as columns.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param train_p A positive decimal. Indicating the percentage of data that goes to training. For example, when
#'     \code{train_p = 0.7}, 70% samples were randomly selected as training dataset. More information see
#'     \code{\link[randomForest]{rfcv}}.
#' @param Group A string that specifies the columns in the \code{metadata} for grouping the temporal series samples.
#' @param OTU_counts_filter_value An integer, indicating the sum of the minimum abundances of OTUs/ASVs in all samples. If the sum
#'     of the abundances that OTU/ASV is below the given positive integer threshold, the OTU/ASV is excluded, and vice versa, it
#'     is retained. The default is \code{NA}.
#' @param reps An integer. The number of replications for cross-validation. By default, \code{reps = 5}. More details see
#'     \code{\link[randomForest]{rfcv}}.
#' @param cv_fold An integer. Number of folds in the cross-validation. By default, \code{cv_fold = 10}. see \code{\link[randomForest]{rfcv}}
#' @param title_size Numeric value for the font size of plot titles. Defaults to 10.
#' @param axis_title_size Numeric value for the font size of axis titles. Defaults to 8.
#' @param legend_title_size Numeric value for the font size of legend titles. Defaults to 8.
#' @param legend_text_size Numeric value for the font size of legend text. Defaults to 6.
#' @param seed Random seed.
#'
#' @return An object of class \code{DataRFClassifier} with the following elements:
#' \describe{
#'   \item{Input_data}{The transposed and (optionally) filtered OTU table.}
#'   \item{Predicted_results_on_train_set}{A vector of predicted group labels for the training set.}
#'   \item{Predicted_results_on_test_set}{A vector of predicted group labels for the test set.}
#'   \item{Traindata_confusion_matrix}{A confusion matrix comparing actual vs. predicted group labels for the training set.}
#'   \item{Testdata_confusion_matrix}{A confusion matrix comparing actual vs. predicted group labels for the test set.}
#'   \item{Margin_scores_train}{A ggplot object displaying the margin scores of the training set samples.}
#'   \item{OTU_importance}{A data frame of OTU importance metrics, sorted by Mean Decrease Accuracy.}
#'   \item{Classifier}{A random forest classifier object trained on the training set.}
#'   \item{cross_validation}{A ggplot object showing the cross-validation error curve as a function of the number of features.}
#' }
#'
#' @importFrom caret createDataPartition
#' @importFrom randomForest randomForest
#' @importFrom randomForest rfcv
#' @importFrom ggplot2 ggplot
#' @importFrom reshape2 melt
#' @importFrom splines ns
#'
#' @export
#' @author Shijia Li
#' @examples
#' \donttest{
#' # Example OTU count data (20 OTUs x 10 samples)
#' set.seed(123)
#' otu_data <- matrix(sample(0:100, 200, replace = TRUE), nrow = 20)
#' colnames(otu_data) <- paste0("Sample", 1:10)
#' rownames(otu_data) <- paste0("OTU", 1:20)
#'
#' # Example metadata with group labels
#' metadata <- data.frame(Group = rep(c("Control", "Treatment"), each = 5))
#'
#' # Run the classifier
#' result <- Data.rf.classifier(raw_data = otu_data,
#'                              metadata = metadata,
#'                              train_p = 0.7,
#'                              Group = "Group",
#'                              OTU_counts_filter_value = 50)
#' }
Data.rf.classifier <- function(raw_data,
                               metadata,
                               train_p,
                               Group,
                               OTU_counts_filter_value = NA,
                               reps = 5,
                               cv_fold = 10,
                               title_size = 10,
                               axis_title_size = 8,
                               legend_title_size = 8,
                               legend_text_size = 6,
                               seed = 123) {

  otu <- raw_data

  if (!is.na(OTU_counts_filter_value)) {
    otu <- otu[which(rowSums(otu) >= OTU_counts_filter_value),]
  } else {
    otu <- otu
  }
  otu <- as.data.frame(t(otu))
  meta <- metadata

  otu$group <- meta[,Group]
  otu$group <- as.factor(otu$group)

  if (!is.null(seed)) {
    set.seed(seed)
  }
  select_train <- caret::createDataPartition(otu$group,p = train_p,list = FALSE)
  otu_train <- otu[select_train,]
  otu_train$group <- as.factor(otu_train$group)
  otu_test <- otu[-select_train,]

  if (!is.null(seed)) {
    set.seed(seed)
  }
  otu_train.forest <- randomForest::randomForest(group ~ ., data = otu_train, importance = TRUE)
  margin_scores <- randomForest::margin(otu_train.forest, otu_train$group)
  margin_df <- data.frame(
    Group = names(margin_scores),
    Margin_Score = as.vector(margin_scores)
  ) %>% arrange(Margin_Score) %>% mutate(Sample = row_number())
  oob_error <- round(otu_train.forest$err.rate[nrow(otu_train.forest$err.rate),"OOB"],3)
  p_margin <- ggplot(margin_df, aes(x = Sample, y = Margin_Score,color = Group)) +
    geom_point() +
    labs(title = 'Margin Scores in Train-Dataset',
         x = 'Number of Samples',
         y = 'Margin Score') +
    theme(panel.grid = element_blank(),
          panel.background = element_rect(color = 'black', fill = 'transparent'),
          title = element_text(size = title_size),
          axis.title = element_text(size = axis_title_size),
          legend.title = element_text(size = legend_title_size),
          legend.text = element_text(size = legend_text_size))+
    annotate("text",x = Inf,y = -Inf,hjust = 1.1,vjust = -0.5,
             label = paste("Out of Bag error = ",oob_error,sep = ""),
             size = 4,color = "black")

  train_predict <- stats::predict(otu_train.forest, otu_train)
  compare_train <- table(train_predict,otu_train$group, dnn = c('Actural','Predicted'))

  test_predict <- stats::predict(otu_train.forest,otu_test)
  compare_test <- table(otu_test$group, test_predict, dnn = c('Actural','Predicted'))

  importance_otu <- otu_train.forest$importance
  importance_otu <- as.data.frame(importance_otu)
  importance_otu <- importance_otu[order(importance_otu$MeanDecreaseAccuracy,
                                         decreasing = TRUE),]

  if (!is.null(seed)) {
    set.seed(seed)
  }
  otu_train.cv <- replicate(reps, randomForest::rfcv(otu_train[-ncol(otu_train)],
                                                     otu_train$group,cv.fold = cv_fold,step = 1.5),
                            simplify = FALSE)

  otu_train.cv <- data.frame(sapply(otu_train.cv,'[[','error.cv'))
  otu_train.cv$otus <- rownames(otu_train.cv)
  otu_train.cv <- reshape2::melt(otu_train.cv,id = 'otus')
  otu_train.cv$otus <- as.numeric(as.character(otu_train.cv$otus))

  p <- ggplot(otu_train.cv,aes(otus,value)) +
    geom_smooth(se = FALSE, method = 'glm', formula = y~splines::ns(x,6))+
    theme(panel.grid = element_blank(),
          panel.background = element_rect(color = 'black', fill = 'transparent'),
          title = element_text(size = title_size),
          axis.title = element_text(size = axis_title_size),
          legend.title = element_text(size = legend_title_size),
          legend.text = element_text(size = legend_text_size)) +
    labs(title = 'Cross-validation curve',
         x = 'Number of Features',
         y = 'Cross-validation error')

  output_result <- list(otu,
                        train_predict,
                        test_predict,
                        compare_train,
                        compare_test,
                        p_margin,
                        importance_otu,
                        otu_train.forest,
                        p)
  names(output_result) <- c('Input_data',
                            'Predicted_results_on_train_set',
                            'Predicted_results_on_test_set',
                            'Traindata_confusion_matrix',
                            'Testdata_confusion_matrix',
                            'Margin_scores_train',
                            'OTU_importance',
                            'Classifier',
                            'cross_validation')

  structure(output_result,class = 'DataRFClassifier')
}



#' @title Select Biomarkers Based on Random Forest Cross-Validation Results
#'
#' @description
#' This function extracts the top biomarkers from a random forest classification result based on cross-validation
#'     and user specified number of microbial features. It updates the cross-validation plot by adding a vertical
#'     dashed line at the specified number of features, and then selects the top features (biomarkers) based on
#'     their importance ranking.
#'
#' @details
#' The function takes an object (usually the output from \code{\link[MicrobTiSDA]{Data.rf.classifier}}, which includes
#'     a cross-validation plot, an OTU importance table, and the original input data) and a user-specified
#'     number of features to select. It then updates the cross-validation plot by adding a vertical dashed line
#'     at the position corresponding to the number of selected features. Next, it extracts the top features from the
#'     OTU importance table (ordered by Mean Decrease Accuracy) and creates q table of these features
#'     from the original microbial feature table. The function returns a list that includes both the transposed biomarker
#'     table and the modified cross-validation plot.
#'
#'
#' @param rf A list containing the results of the random forest classification. Default to \code{\link[MicrobTiSDA]{Data.rf.classifier}}.
#' @param feature_select_num A numeric value specifying the number of top features (biomarkers) to select. Typically, the numer of
#'     specified biomarkers needs to be determined by the user based on the cross-validation result plot output by
#'     \code{\link[MicrobTiSDA]{Data.rf.classifier}}.
#'
#' @return An object of class \code{RfBiomarker} with two elements:
#'   \describe{
#'     \item{OTU_importance}{A data frame of the selected biomarkers (transposed feature table).}
#'     \item{cross_validation_fig}{A ggplot object of the cross-validation plot with a vertical dashed line indicating the feature selection cutoff.}
#'   }
#' @export
#' @importFrom ggplot2 ggplot
#' @author Shijia Li
#'
#' @examples
#' \donttest{
#' # Example OTU count data (20 OTUs x 10 samples)
#' set.seed(123)
#' otu_data <- matrix(sample(0:100, 200, replace = TRUE), nrow = 20)
#' colnames(otu_data) <- paste0("Sample", 1:10)
#' rownames(otu_data) <- paste0("OTU", 1:20)
#'
#' # Example metadata with group labels
#' metadata <- data.frame(Group = rep(c("Control", "Treatment"), each = 5))
#'
#' # Run the classifier
#' rf_result <- Data.rf.classifier(raw_data = otu_data,
#'                              metadata = metadata,
#'                              train_p = 0.7,
#'                              Group = "Group",
#'                              OTU_counts_filter_value = 50)
#' # If you wish to select the top 5 features:
#' result <- Rf.biomarkers(rf = rf_result, feature_select_num = 5)
#' # View the biomarker table
#' print(result$OTU_importance)
#' # View the updated cross-validation plot
#' print(result$cross_validation_fig)
#' }
Rf.biomarkers <- function(rf = rf_results,feature_select_num) {
  p <- rf$cross_validation +
    geom_vline(xintercept = as.numeric(feature_select_num),linetype = 'dashed')+
    scale_x_continuous(breaks = as.numeric(feature_select_num))

  train_predict <- rf$Predicted_results_on_train_set
  test_predict <- rf$Predicted_results_on_test_set

  selected_biomarker <- rownames(rf$OTU_importance)[1:as.numeric(feature_select_num)]
  feature_table <- rf$Input_data
  important_otu_table <- feature_table[selected_biomarker]

  output_biomarker_table <- as.data.frame(t(important_otu_table))

  output_results <- list(output_biomarker_table,train_predict,test_predict,p)
  names(output_results) <- c('OTU_importance',
                             'Predicted_results_on_train_set',
                             'Predicted_results_on_test_set',
                             'cross_validation_fig')

  structure(output_results,class = 'RfBiomarker')
}



#' Visualize Microbial Feature Classification Results Using NMDS
#'
#' @description
#' This function generates a non-metric multidimensional scaling (NMDS) plot to visualize OTU/ASV classification
#'     results. It combines an OTU/ASV table with predicted group labels to produce a scatter plot in NMDS space,
#'     where each sample is colored according to its predicted group.
#'
#' @details
#' The function expects as input the results of output of the function \code{\link[MicrobTiSDA]{Data.rf.classifier}}.
#'     It conputes NMDS coordinates based on the OTU/ASV data using the specified distance method (defaulting to
#'     Bray-Curtis) and then maps the predicted group labels onto the NMDS coordinates. The visualization is produced
#'     using \code{ggplot2}, with polygons delineating the convex hull of each predicted group.
#'
#'
#' @param classified_results A list object of \code{\link[MicrobTiSDA]{Data.rf.classifier}} output.
#' @param dist_method Dissimilarity index, defaults to \code{bray}. For other options, including
#'     \code{manhattan}, \code{euclidean}, \code{canberra} etc., see \code{\link[vegan]{vegdist}}.
#' @param fig_title A character string specifying the title of the NMDS plot. Default is \code{'NMDS visualization'}.
#' @param legd_title A character string for the legend title representing the predicted groups. Default is \code{'Predicted Groups'}.
#' @param points_size A numeric value specifying the size of the points in the plot. Default is \code{1}.
#' @param legend_title_size A numeric value specifying the font size of the legend title. Default is \code{8}.
#' @param legend_text_size A numeric value specifying the font size of the legend text. Default is \code{6}.
#' @param axis_title_size A numeric value specifying the font size of the axis titles. Default is \code{6}.
#' @param axis_text_size A numeric value specifying the font size of the axis text. Default is \code{6}.
#'
#' @return A \code{ggplot2} object displaying the NMDS plot with samples colored by their predicted group and convex hulls
#'         outlining each group.
#'
#' @importFrom vegan metaMDS
#' @importFrom plyr ddply
#' @importFrom ggplot2 ggplot .data
#' @author Shijia Li
#' @export
#'
#' @examples
#' \donttest{
#' # Example OTU count data (20 OTUs x 10 samples)
#' set.seed(123)
#' otu_data <- matrix(sample(0:100, 200, replace = TRUE), nrow = 20)
#' colnames(otu_data) <- paste0("Sample", 1:10)
#' rownames(otu_data) <- paste0("OTU", 1:20)
#'
#' # Example metadata with group labels
#' metadata <- data.frame(Group = rep(c("Control", "Treatment"), each = 5))
#'
#' # Run the classifier
#' rf_result <- Data.rf.classifier(raw_data = otu_data,
#'                              metadata = metadata,
#'                              train_p = 0.7,
#'                              Group = "Group",
#'                              OTU_counts_filter_value = 50)
#' # If you wish to select the top 5 features:
#' result <- Rf.biomarkers(rf = rf_result, feature_select_num = 5)
#'
#' nm_plot <- Classify.vis(classified_results = result,
#'                         dist_method = 'bray',
#'                         fig_title = 'NMDS Plot',
#'                         legd_title = 'Predicted Groups',
#'                         points_size = 1.5)
#' }
#'
Classify.vis = function(classified_results,
                        dist_method = 'bray',
                        fig_title = 'NMDS visualization',
                        legd_title = 'Predicted Groups',
                        points_size = 1,
                        legend_title_size = 8,
                        legend_text_size = 6,
                        axis_title_size = 6,
                        axis_text_size = 6) {

  otu <- classified_results[[1]]
  nmds <- vegan::metaMDS(t(otu),distance = dist_method)
  result <- nmds$points
  result <- as.data.frame(cbind(result, rownames(result)))

  predict_group <- c(classified_results[[2]],classified_results[[3]])
  predict_group <- as.character(predict_group[rownames(result)])

  colnames(result)[1:3] <- c('NMDS1','NMDS2','samples')
  result$NMDS1 <- as.numeric(as.character(result$NMDS1))
  result$NMDS2 <- as.numeric(as.character(result$NMDS2))
  result$samples <- as.character(result$samples)
  result <- cbind(result, predict_group)

  p <- ggplot2::ggplot(result,aes(.data$NMDS1,.data$NMDS2,color = predict_group)) +
    geom_polygon(data = plyr::ddply(result,'predict_group',
                                    function(df) df[grDevices::chull(df[[1]], df[[2]]),]),fill = NA) +
    labs(title = fig_title,
         color = legd_title) +
    theme(legend.title = element_text(size = legend_title_size),
          legend.text = element_text(size = legend_text_size),
          axis.title = element_text(size = axis_title_size),
          axis.text = element_text(size = axis_text_size))+
    geom_point(size = points_size)+
    theme(panel.grid = element_blank(),
          panel.background = element_rect(color = 'black', fill = 'transparent'))
  return(p)
}




#' @title Fit Mixed-Effects Spline Regression Models with GAM for Microbial Features on Group-Level
#' @description
#' This function fits mixed-effect spline regression models using GAM to capture nonlinear temporal trends in microbial
#'     community data at the group level. It incorporates random effects for individual IDs within each group and uses
#'     natural splines to model the relationship between time and microbial feature abundances.
#' @details
#' This function fits mixed-effects spline regression models. Unlike \code{\link[MicrobTiSDA]{Reg.SPLR}}, this function
#'     captures the overall temporal dynamics of microbial features across different groups. Assuming the dataset contains
#'     \eqn{\textit{m}} groups, each with \eqn{\textit{n}} subjects, the model is formulated as:
#'     \deqn{x_{(mi)}(t) = \beta_{m0} + \sum_{k=1}^{K} \beta_{mk} \cdot N_{mk}(t) + b_{n(m)} + \epsilon}
#'     where \eqn{x_{(mi)}(t)} represents the abundance of microbial feature \eqn{\textit{i}} at time point \eqn{\textit{t}}
#'     in group \eqn{\textit{m}}. The random effect \eqn{b_{n(m)}} reflects the departure of individual \eqn{\textit{n}} in group
#'     \eqn{\textit{m}} from overall population average effects. The parameter \eqn{K} refers to the number of basis functions,
#'     which is equal to the number of knots plus one. (i.e., \eqn{K = number of knots + 1}).
#'
#' The \code{Reg.MESR} function first extracts independent variables from a design matrix (produced by the \code{\link[MicrobTiSDA]{Design}})
#'     by removing columns corresponding to the pre-processed OTU data. For each design dummy variable (excluding the first
#'     and last columns), the function subsets the data to include only the observations where the dummy variable equals 1.
#'     Then, for each OTU in the pre-processed data, it optionally filters out outliers based on a specified z-score threshold.
#'     If the number of unique transformed OTU values exceeds a given threshold (\code{unique_values}), a mixed-effects spline
#'     regression model is fitted using a natural spline on the time variable along with a random effect for the sample ID.
#'     When the \code{Knots} parameter is not provided, the function iterates over a range of knot numbers (from 1
#'     to \code{max_Knots}), selects the optimal model by minimizing the Generalized Cross-Validation (GCV) criterion,
#'     and extracts the corresponding knot locations. Alternatively, if \code{Knots} is provided, it is directly used in
#'     model fitting. The resulting fitted models and associated knots information are organized into nested lists and returned.
#'
#'
#' @param Data_for_Reg The data frame output from the \code{\link[MicrobTiSDA]{Design}}.
#' @param pre_processed_data The transformed data output from the \code{\link[MicrobTiSDA]{Data.trans}} function. A
#'     pre-processed OTU data frame with sample IDs as row names and OTU IDs as column names.
#' @param unique_values An integer specifying the minimum number of unique values required in an OTU for spline fitting (default: 5).
#' @param z_score A numeric value specifying the z-score threshold for outlier filtering; if \code{NA}, no
#'     outlier filtering is performed (default: \code{NA}).
#' @param Knots An optional numeric vector specifying the knots to use in the spline regression. If \code{NULL}, the optimal number of
#'     knots is determined by minimizing the GCV criterion (default: \code{NULL}).
#' @param max_Knots An integer indicating the maximum number of knots to consider when selecting the optimal spline model (default: 3).
#' @param seed Random seed.
#' @import mgcv splines
#' @return An object of class \code{RegMESR} with two elements: \code{fitted_model} is a nested list containing the fitted mixed-effects GAM models for each design
#'     dummy variable and OTU; \code{knots_info_each_model} is a corresponding nested list with the knots used for each model.
#' @export
#' @author Shijia Li
#' @examples
#' \donttest{
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' fit_result <- Reg.MESR(Data_for_Reg = design_data,
#'                        pre_processed_data = Pre_processed_Data,
#'                        unique_values = 5,
#'                        z_score = 2,
#'                        Knots = NULL,
#'                        max_Knots = 5)
#' }
Reg.MESR <- function(Data_for_Reg,
                     pre_processed_data,
                     unique_values = 5,
                     z_score = NA,
                     Knots = NULL,
                     max_Knots = 3,
                     seed = 123) {

  if (inherits(Data_for_Reg,"Design")) {
    Data_for_Reg <- Data_for_Reg$data
  }

  if (inherits(pre_processed_data,"TransformedData")) {
    pre_processed_data <- pre_processed_data$transformed_data
  }

  inde_var <- setdiff(colnames(Data_for_Reg),colnames(pre_processed_data))
  inde_var_data <- Data_for_Reg[inde_var]

  fit_data <- list()
  knots_info <- list()

  for (i in colnames(inde_var_data[,-c(1,ncol(inde_var_data))])) {

    if (!is.null(seed)) {
      set.seed(seed)
    }
    GCV <- rep(0,max_Knots)
    Sub_data <- cbind(pre_processed_data,inde_var_data[1],inde_var_data[i],inde_var_data[ncol(inde_var_data)])
    Sub_data <- Sub_data[Sub_data[,i] == 1,]

    for (j in colnames(pre_processed_data)) {

      Sub_data_for_fit <- cbind(Sub_data[j],Sub_data[ncol(Sub_data)-2],Sub_data[ncol(Sub_data)])

      if (!is.na(z_score)) {
        z_scores <- scale(Sub_data_for_fit[1])
        outliers_idx <- which(abs(z_scores) > z_score)
        Sub_data_for_fit <- Sub_data_for_fit[-outliers_idx,]
      } else {
        Sub_data_for_fit <- Sub_data_for_fit
      }

      Sub_data_for_fit$ID <- as.factor(Sub_data_for_fit$ID)

      if (length(unique(Sub_data_for_fit[,1])) > unique_values) {

        if (is.null(Knots)) {
          for (k in 1:max_Knots) {
            formula <- stats::as.formula(paste(j," ~ splines::ns(Time, df = ",1+k,")+s(ID,bs = 're')",
                                        sep = ""))
            sp_model <- mgcv::gam(formula = formula,
                                  data = Sub_data_for_fit,
                                  method = "GCV.Cp")
            GCV[k] <- sp_model$gcv.ubre
          }
          best_knots <- c(1:max_Knots)[which.min(GCV)]
          knots <- attr(splines::ns(Sub_data_for_fit$Time,df = 1+best_knots),"knots")
          knots_info[[i]][[j]] <- knots
          best_formula <- stats::as.formula(paste(j," ~ splines::ns(Time, df = ",1+best_knots,")+s(ID,bs = 're')",
                                           sep = ""))
          fit_data[[i]][[j]] <- mgcv::gam(formula = best_formula,
                                          data = Sub_data_for_fit,
                                          method = 'GCV.Cp')
        } else {
          Knots_str <- paste(Knots,collapse = ",")
          formula <- stats::as.formula(paste(j," ~ splines::ns(Time, knots = c(",Knots_str,"))+s(ID,bs='re')",
                                      sep = ''))
          fit_data[[i]][[j]] <- mgcv::gam(formula = formula,data = Sub_data_for_fit,
                                          method = 'GCV.Cp')
          knots_info[[i]][[j]] <- Knots
        }
      } else {
        fit_data[[i]][[j]] <- NULL
        knots_info[[i]][[j]] <- NULL
        message("The number of non-zero value for ",j," in group ",i," is ",
                length(unique(Sub_data_for_fit[,1]))," less than ",
                unique_values,", thus the model was excluded.\n")
      }
    }
  }
  fit_result <- list(
    fitted_model = fit_data,
    knots_info_each_model = knots_info,
    parameters = list(unique_values = unique_values,
                      z_score = z_score,
                      Knots = Knots,
                      max_Knots = max_Knots)
  )

  class(fit_result) <- "RegMESR"
  return(fit_result)
}




#' @title Predict Time-Series Data from Fitted Mixed-Effect Spline regression Models
#' @description
#' This function generates predicted OTU abundances at new time points using previously fitted spline regression models.
#'     For each group defined in the fitted models, it extracts the time range from the metadata, creates a new sequence of time points
#'     with a specified interval, and uses the corresponding fitted models to predict values at these new time points.
#'
#' @details
#' The function accepts a list of fitted models (typically the output from \code{\link[MicrobTiSDA]{Reg.MESR}}) and sample metadata.
#'     For each group (as defined by the \code{Group} parameter), it extracts the subset of metadata corresponding to that group. The minimum and maximum time
#'     values are determined, and a new sequence of time points is generated using the provided \code{time_step}. For each OTU model of the
#'     group, predictions are obtained via the \code{predict} function applied to the new time points, and the results are compiled into a data
#'     frame with an additional column \code{Predicted_Time} indicating the prediction time. The output is a list of data frames, each corresponding
#'     to a group, containing the predicted OTU abundances.
#'
#'
#' @param Fitted_models A list object generated by \code{\link[MicrobTiSDA]{Reg.MESR}}.
#' @param metadata A data frame. Containing information about all samples, including at least the grouping of all samples as well as
#'     individual information (\code{Group} and \code{ID}), the sampling \code{Time} point for each sample, and other relevant information.
#' @param Group A character string indicating the column name in \code{metadata} that defines group information for prediction.
#' @param time_step A numeric value specifying the interval between new time points in the prediction sequence.
#' @param Sample_Time A character string indicating the column name in \code{metadata} that contains sample time information.
#' @author Shijia Li
#' @return An object of \code{PredictedDataMESR}. Each element of the list corresponds to a group and contains the predicted OTU abundances at the
#'   new time points, with rows labeled by the group and time (formatted as \code{Group_T_Time}) and an additional column \code{Predicted_Time}.
#' @export
#'
#' @examples
#' \donttest{
#' metadata <- data.frame(
#'   TimePoint = c(1, 2, 3, 4),
#'   Sample = c('S1', 'S2', 'S3', 'S4'),
#'   GroupA = c('A', 'A', 'B', 'B'),
#'   GroupB = c('X', 'Y', 'X', 'Y')
#' )
#'
#' # Example pre-processed data (e.g., transformed abundance data)
#' Pre_processed_Data <- data.frame(
#'   Feature1 = rnorm(4),
#'   Feature2 = rnorm(4)
#' )
#'
#' # Create design matrix using grouping variables
#' design_data <- Design(metadata, Group_var = c('GroupA', 'GroupB'), Pre_processed_Data,
#'                       Sample_Time = 'TimePoint', Sample_ID = 'Sample')
#'
#' fit_result <- Reg.MESR(Data_for_Reg = design_data,
#'                        pre_processed_data = Pre_processed_Data,
#'                        unique_values = 5,
#'                        z_score = 2,
#'                        Knots = NULL,
#'                        max_Knots = 5)
#'
#' predictions <- Pred.data.MESR(fit_result,
#'                              metadata,
#'                              Group = "Group",
#'                              time_step = 1,
#'                              Sample_Time = "TimePoint")
#' }
Pred.data.MESR <- function(Fitted_models, metadata, Group, time_step, Sample_Time) {

  predicted_data <- list()
  if (inherits(Fitted_models,"RegMESR")) {
    Fitted_models <- Fitted_models[[1]]
  }

  for (i in names(Fitted_models)) {
    colnames(metadata)[colnames(metadata) == Sample_Time] <- 'Time'

    meta_group <- metadata[metadata[[Group]] == i,]

    Time <- meta_group$Time
    ID <- meta_group$ID
    new_data <- data.frame(Time, ID)

    pred_data <- as.data.frame(matrix(0,nrow(new_data),
                                      length(Fitted_models[[i]])))
    rownames(pred_data) <- paste(i,"_T_",Time,"_",ID,sep = "")
    colnames(pred_data) <- names(Fitted_models[[i]])

    for (k in names(Fitted_models[[i]])) {
      pred_data[,k] <- stats::predict(Fitted_models[[i]][[k]],
                               newdata = new_data,
                               type = 'response',
                               re.form = NA)
    }

    pred_data <- as.data.frame(pred_data) %>%
      mutate(.,Predicted_Time = Time)

    predicted_data[[i]] <- pred_data
  }

  structure(
    list(
      data = predicted_data,
      groups = names(predicted_data),
      params = list(
        Group = Group,
        Sample_Time = Sample_Time,
        time_step = time_step
      )
    ),
    class = "PredictedDataMESR"
  )
}



#' @title Visualize Group-Level OTU Temporal Profiles from Clustered Predicted Data
#' @description
#' This function visualizes the temporal patterns of microbial features at the group level, specifically tailored for data
#'     derived from mixed-effects spline regression (MESR) analyses. It leverages clustering results to segregate features into clusters
#'     based on their temporal trends, and then generates smoothed time-series plots for each cluster.
#' @details
#' The function begins by selecting branches from hierarchical clustering objects (provided in \code{cluster_results}) using either
#'     a specified cut-off height or a predefined number of clusters, as determined by the \code{cutree_by} parameter. For each group, it extracts
#'     the corresponding raw data from \code{Design_data} and determines the y-axis limits based on both the pre-processed data and the predicted data.
#'     Then, for each cluster within a group, the function subsets the predicted data to include only those features belonging to that cluster. If
#'     taxonomic annotation data (\code{Taxa}) is provided, feature names are augmented with species-level labels. The data is then reshaped into
#'     a long format and plotted using \code{ggplot2}, where smoothed curves (via \code{stat_smooth}) depict the predicted temporal profiles.
#'     Optionally, raw data points can be overlaid (if \code{plot_dots} is \code{TRUE}), and a linear model is fitted to each cluster’s data to test
#'     for significant trends. When the linear model meets criteria based on p-value (< 0.05), R² (greater than \code{lm_R2}), and a minimum absolute slope
#'     (greater than \code{lm_abs_slope}), a dashed regression line is added with an annotation indicating the trend direction (upward or downward) along with
#'     the R² and slope values. Various parameters allow customization of plot appearance including axis scales, font sizes, and legend properties.
#'
#' @param cluster_results A list object output from the \code{\link[MicrobTiSDA]{Data.cluster}}).
#' @param cutree_by A character string specifying the method to cut the dendrogram, either by \code{"height"} or by \code{"branches"}.
#' @param cluster_height A numeric vector specifying the cut-off height for each group when \code{cutree_by = "height"}.
#' @param cluster_branches A numeric vector specifying the number of clusters for each group when \code{cutree_by = "branches"}.
#' @param predicted_data The output data frame from the \code{\link[MicrobTiSDA]{Pred.data.MESR}}).
#' @param Design_data The output data from the \code{\link[MicrobTiSDA]{Design}}).
#' @param pre_processed_data The transformed data output from the \code{\link[MicrobTiSDA]{Data.trans}} function. A
#'     pre-processed OTU data frame with sample IDs as row names and OTU IDs as column names.
#' @param Taxa A data frame providing taxonomic annotations for microbial species.
#' @param plot_dots Logical; if \code{TRUE}, raw data points are overlaid on the temporal curves (default: \code{TRUE}).
#' @param figure_x_scale A numeric value specifying the interval for x-axis breaks in the figures (default: \code{5}).
#' @param plot_lm Logical; if \code{TRUE}, a linear model is fitted to the predicted data to detect trends, and the regression line
#'     is added (default: \code{FALSE}).
#' @param lm_R2 A numeric threshold for the minimum R-squared value required to annotate the linear model (default: \code{0.01}).
#' @param lm_abs_slope A numeric threshold for the minimum absolute slope required to annotate the linear model (default: \code{0.005}).
#' @param title_size A numeric value specifying the font size for the plot title (default: \code{10}).
#' @param axis_title_size A numeric value specifying the font size for the axis titles (default: \code{8}).
#' @param axis_y_size A numeric value specifying the font size for the y-axis text (default: \code{5}).
#' @param axis_x_size A numeric value specifying the font size for the x-axis text (default: \code{5}).
#' @param lm_sig_size A numeric value specifying the font size for linear model annotation text (default: \code{5}).
#' @param legend_title_size A numeric value specifying the font size for legend titles (default: \code{5}).
#' @param legend_text_size A numeric value specifying the font size for legend text (default: \code{5}).
#' @param dots_size A numeric value specifying the size of the overlaid raw data points (default: \code{0.7}).
#'
#' @return An object of class \code{MicrobTiSDA.MSERvisual} which contains lists of ggplot2 objects, where each top-level element corresponds to a
#'     group and each sub-element corresponds to a cluster within that group. Each plot visualizes the temporal profiles of microbial features in that cluster.
#' @importFrom ggplot2 ggplot
#' @importFrom utils head str tail
#' @export
#'
Data.visual.MESR <- function(cluster_results,
                             cutree_by = "height",
                             cluster_height = NA,
                             cluster_branches = NA,
                             predicted_data,
                             Design_data,
                             pre_processed_data,
                             Taxa = NULL,
                             plot_dots = TRUE,
                             figure_x_scale = 10,
                             plot_lm = TRUE,
                             lm_R2 = 0.01,
                             lm_abs_slope = 0.005,
                             title_size = 10,
                             axis_title_size = 8,
                             axis_y_size = 5,
                             axis_x_size = 5,
                             lm_sig_size = 5,
                             legend_title_size = 5,
                             legend_text_size = 5,
                             dots_size = 0.7) {

  curves_plot <- list()
  if (inherits(cluster_results,"MicrobTiSDA.clusterCut")) {
    cluster_results <- cluster_results[[1]]
  }

  if (inherits(Design_data,"Design")) {
    Design_data <- Design_data[[1]]
  }

  if (inherits(pre_processed_data,"TransformedData")) {
    pre_processed_data <- pre_processed_data[[1]]
  }

  if (inherits(predicted_data,"PredictedDataMESR")) {
    predicted_data <- predicted_data[[1]]
  }

  n <- 0
  for (i in names(cluster_results)) {
    n <- n + 1
    # 1. Select branches based on the dendrogram of clustering tree
    if (cutree_by == 'height') {
      cutrees <- stats::cutree(cluster_results[[i]], h = cluster_height[n])
    } else if (cutree_by == 'branches') {
      cutrees <- stats::cutree(cluster_results[[i]], k = cluster_branches[n])
    } else {
      stop("cutree_by must be either 'height' or 'branches'")
    }

    # 2. Extract the data from pre-processed data for plotting the transformed OTU count at each sampling time point
    rawdata <- Design_data[Design_data[,i] == 1, ]

    ylim_min <- min(min(pre_processed_data),
                    min(predicted_data[[i]][,-ncol(predicted_data[[i]])]))
    ylim_max <- max(max(pre_processed_data),
                    max(predicted_data[[i]][,-ncol(predicted_data[[i]])]))

    # 3. Temporal profile visualizations
    curves_plot[[i]] <- list()

    Pred_data_for_plot <- predicted_data[[i]]
    for (j in as.numeric(levels(as.factor(cutrees)))) {
      cols_to_remove <- names(which(cutrees != j))
      data_for_plot <- Pred_data_for_plot[,!colnames(Pred_data_for_plot) %in% cols_to_remove]

      if(!is.null(Taxa)) {
        Annotate <- Taxa[match(colnames(data_for_plot)[-ncol(data_for_plot)], rownames(Taxa)), ]
        # added mutate(across(where(is.factor), as.character)) %>%
        Annotate <- Annotate %>% mutate(across(where(is.factor), as.character)) %>% mutate(across(everything(), ~na_if(.,"")))
        Taxon <- apply(Annotate, 1, function(x) { tail(na.omit(x), 1) })
        Taxon <- data.frame(Species_annotation = Taxon)
        colnames(data_for_plot) <- c(paste(colnames(data_for_plot)[-ncol(data_for_plot)],
                                           " (", Taxon[, 1], ")", sep = ""),
                                     "Predicted_Time")
      } else {
        colnames(data_for_plot) <- colnames(data_for_plot)
      }

      data_for_plot_long <- data_for_plot %>%
        tidyr::pivot_longer(cols = -Predicted_Time,names_to = "Features",values_to = "Abundance")

      curves_plot[[i]][[j]] <- ggplot(data_for_plot_long, aes(x = Predicted_Time, y = Abundance, color = Features)) +
        stat_smooth(se = FALSE) +
        labs(title = paste(i,"_cluster_",j,sep = ""),
             x = "Time",y = "Predicted values")+
        theme_bw()+
        theme(title = element_text(size = title_size),
              axis.title = element_text(size = axis_title_size),
              axis.text.y = element_text(size = axis_y_size),
              axis.text.x = element_text(size = axis_x_size),
              legend.title = element_text(size = legend_title_size),
              legend.text = element_text(size = legend_text_size))+
        scale_x_continuous(breaks = seq(1,max(data_for_plot_long$Predicted_Time),figure_x_scale),
                           labels = seq(1,max(data_for_plot_long$Predicted_Time),figure_x_scale))+
        ylim(ylim_min-2,ylim_max+2)

      if (ncol(data_for_plot) > 9 & ncol(data_for_plot) < 19) {
        curves_plot[[i]][[j]] <- curves_plot[[i]][[j]]+
          guides(color = guide_legend(ncol = 2))
      }

      if (plot_dots == TRUE) {
        cols_to_reserve <- names(which(cutrees == j))
        point_data_for_plot <- rawdata[colnames(rawdata) %in% cols_to_reserve]
        point_data_for_plot$Time <- rawdata$Time
        point_data_for_plot$ID <- rawdata$ID

        if (!is.null(Taxa)) {
          Annotate <- Taxa[match(colnames(point_data_for_plot)[-c(ncol(point_data_for_plot)-1,
                                                                  ncol(point_data_for_plot))],
                                 rownames(Taxa)),]
          Annotate <- Annotate %>%
            mutate(across(where(is.factor), as.character)) %>%
            mutate(across(everything(), ~na_if(.,"")))
          Taxon <- apply(Annotate,1,function(x) {tail(na.omit(x),1)})
          Taxon <- data.frame(Species_annotation = Taxon)
          colnames(point_data_for_plot) <- c(paste(colnames(point_data_for_plot)[-c(ncol(point_data_for_plot)-1,
                                                                                    ncol(point_data_for_plot))],
                                                   " (",Taxon[,1],")",sep = ""),"Time","ID")
        } else {
          colnames(point_data_for_plot) <- colnames(point_data_for_plot)
        }

        point_data_for_plot_long <- point_data_for_plot %>%
          tidyr::pivot_longer(cols = -c(Time,ID),names_to = "Features",values_to = "Abundance")
        curves_plot[[i]][[j]] <- curves_plot[[i]][[j]]+
          geom_point(data = point_data_for_plot_long,
                     aes(x = Time,
                         y = Abundance,
                         color = Features,
                         shape = ID))+
          guides(color = guide_legend(order = 1),
                 shape = guide_legend(order = 2))

      } else {
        curves_plot[[i]][[j]] <- curves_plot[[i]][[j]]
      }

      if (plot_lm == TRUE) {
        lm_model <- stats:lm(Abundance ~ Predicted_Time,data = data_for_plot_long)
        lm_model_summary <- summary(lm_model)
        r_squared <- lm_model_summary$r.squared
        slope <- as.numeric(lm_model$coefficients[2])
        p_value <- lm_model_summary$coefficients[, "Pr(>|t|)"]
        slope_p <- as.numeric(p_value[2])
        if (slope_p < 0.05 && r_squared > lm_R2 && abs(slope) > lm_abs_slope) {
          if (slope > 0) {
            curves_plot[[i]][[j]] <- curves_plot[[i]][[j]]+
              stat_smooth(data = data_for_plot_long,aes(x = Predicted_Time,y = Abundance),
                          method = 'lm',linetype = 'dashed',se = TRUE,color = 'darkgrey')+
              geom_text(aes(x = -Inf,y = Inf),label = paste("R2=",round(r_squared,4),"\n",
                                                            "Slope=",round(slope,4),"\n",
                                                            "Upward trend",sep = ""),
                        hjust = -1.1, vjust = 1.5,size = 4,color = 'darkgray')
          } else {
            curves_plot[[i]][[j]] <- curves_plot[[i]][[j]]+
              stat_smooth(data = data_for_plot_long,aes(x = Predicted_Time,y = Abundance),
                          method = 'lm',linetype = 'dashed',se = TRUE,color = 'darkgrey')+
              geom_text(aes(x = -Inf,y = Inf),label = paste("R2=",round(r_squared,4),"\n",
                                                            "Slope=",round(slope,4),"\n",
                                                            "Downward trend",sep = ""),
                        hjust = -1.1, vjust = 1.5,size = 4,color = 'darkgray')
          }
        } else {
          curves_plot[[i]][[j]] <- NULL
        }
      }
    }
  }

  # 4. Return the visualized temporal profiles of OTUs for each group
  result <- list(
    plots = curves_plot,
    params = list(
      cutree_by = cutree_by,
      cluster_height = cluster_height,
      cluster_branches = cluster_branches,
      predicted_data = predicted_data,
      Design_data = Design_data,
      pre_processed_data = pre_processed_data,
      Taxa = Taxa,
      plot_dots = plot_dots,
      figure_x_scale = figure_x_scale,
      plot_lm = plot_lm,
      lm_R2 = lm_R2,
      lm_abs_slope = lm_abs_slope,
      title_size = title_size,
      axis_title_size = axis_title_size,
      axis_y_size = axis_y_size,
      axis_x_size = axis_x_size,
      lm_sig_size = lm_sig_size,
      legend_title_size = legend_title_size,
      legend_text_size = legend_text_size,
      dots_size = dots_size
    )
  )
  class(result) <- "MicrobTiSDA.MSERvisual"

  return(result)
}
