#' Internal function. Find the intensities corresponding to the closest irradiance match between intended and calibration.
#'
#' @param irradiance_matrix Matrix of intended irradiances. rows = leds and columns = events
#' @param calibration_df Dataframe of calibration data generated by internal.calibCombine
#' @param peaks Vector of length 8 or 9. Containing wavelengths at which each LED peaks.
#'
#' @return Matrix of closest intensities, in the same format as `irradiance_matrix`
#' @export
#'
#' @examples
#'
#' # Format calibration data correctly
#' calib <- LightFitR::calibration[, c(3, 5, 4, 6)]
#'
#' # Run function
#' internal.closestIntensities(LightFitR::target_irradiance, calib)
#'
internal.closestIntensities = function(irradiance_matrix, calibration_df, peaks=LightFitR::helio.dyna.leds$wavelength){

  # Setup
  irradMat = internal.rmWhite(irradiance_matrix)
  peakWavelengths = internal.closestWavelength(unique(calibration_df$wavelength), peaks)

  # Calculate closest
  closestIntensityMatrix = sapply(1:(nrow(LightFitR::helio.dyna.leds)-1), function(a){ # Go through each LED and calculate the closest we can get with calibration data
    intendedSubset = c(as.numeric(irradMat[a,]))

    criteria = (calibration_df$led == LightFitR::helio.dyna.leds[a, 'wavelength']) & (calibration_df$wavelength == peakWavelengths[a])
    calibSubset = calibration_df[criteria,] #subset calibration_df data based on the led in question, and teh wavelength at the peak

    ## Closest intensity for each timepoint
    closestIntensity = sapply(intendedSubset, function(i){

      if(i==0){
        closest=0
        } else{

        closestIrradiance = which.min(abs(calibSubset$irradiance - i))[1] #Which is the 1st closest to intended irradiance?
        closest = calibSubset[closestIrradiance, 'intensity'] #Get the corresponding intensity
      }

      closest
    })

    closestIntensity
  })


  # Formatting

  closestIntensityMatrix = t(closestIntensityMatrix)
  closestIntensityMatrix = internal.addWhiteZero(closestIntensityMatrix)

  # Checks
  if(sum(is.na(closestIntensityMatrix)) > 0){
    warning("An internal function (internal.closestIntensities) has produced NAs. Please double check your input data")}

  return(closestIntensityMatrix)
}
