\name{LSMsim}
\alias{LSMsim}

\title{
  Simulating Data according to the Latent Space Item Response Model}
\description{
  This function simulates data according to the Latent Space Item Response Model (LSIRM) with an \code{R} dimensional latent space and binary and/or ordinal item scores.}

\usage{
LSMsim(N, nit, ndim_z, nc=NULL, theta=NULL, b=NULL, z=NULL, w=NULL, gamma=NULL)

}
\arguments{

  \item{N}{Sample size}
  \item{nit}{ Number of items}
  \item{ndim_z}{ Number of dimensions of the latent space, \code{R}.}

  \item{nc}{ Vector of length \code{nit} containng the number of response categories for each item. If NULL (the default) all items are simulated to be binary}

  \item{theta}{ \code{N}-dimensional vector of true person intercepts \eqn{\theta_p}, if NULL these are drawn from a standard normal distribution}
  \item{b}{ \code{nit}-dimensional vector of true item intercepts \eqn{b_p}, if NULL these are drawn from a uniform distribution}
\item{z}{ \code{N} by \code{ndim_z} matrix of true latent space person coordinates \eqn{z_{pr}}, if NULL these are drawn from a standard normal distribution}
\item{w}{ \code{nit} by \code{ndim_z} matrix of true latent space item coordinates \eqn{w_{ir}}, if NULL these are drawn from a standard normal distribution}
  \item{gamma}{ a weight parameter for the Euclidean distances (see \bold{details}), if NULL gamma=1}

  }

\details{
  Data is simulated according to the original LSIRM by Jeon et al. (2021):
    \deqn{\text{logit}(E(X_{pi})) = \theta_p + b_i - \gamma(\Sigma_{r=1}^{R} (z_{pr}-w_{ir})^2)^{1/2}}
  In \link{LSMfit}, \eqn{\gamma} is fixed to one as it is not identified in a joint maximum likelihood framework (see Molenaar & Jeon, submitted). However, for data simulation, \eqn{\gamma} can be used to change the strength of the effect of \eqn{z_{pr}} and \eqn{w_{ir}}.

}
\value{
  An list containing:
   \item{ X}{ the simulated data}
   \item{ par}{ a list containing the true parameter values, including \code{wt} and \code{zt}, the rotated     matrices of \eqn{w_{ir}} and \eqn{z_{pr}} parameters.}
   }

\references{

Jeon, M., Jin, I. H., Schweinberger, M., & Baugh, S. (2021). Mapping unobserved item–respondent
interactions: A latent space item response model with interaction map. Psychometrika, 86(2), 378-403. doi:10.1007/s11336-021-09762-5

Molenaar, D., & Jeon, M.J. (in press). Joint maximum likelihood estimation of latent space item response models. \emph{Psychometrika}.



}
\author{ Dylan Molenaar \email{d.molenaar@uva.nl}}

\seealso{
\code{\link{LSMfit}} for fitting LSIRM models using joint maximum likelihood.
}
\examples{
 # data sim with 1000 subjects and 20 items according to 2 dimensional
 # latent space model (R=2) with both binary and ordinal items
 set.seed(1111)
 N=1000
 nit=20
 ndim_z=2
 nc=sample(c(2,3,5),nit,replace=TRUE)    # mix of 2, 3, and 5 point scales
 dat_obj=LSMsim(N,nit,ndim_z,nc=nc)
 X=dat_obj$X
 zt=dat_obj$par$zt   # rotated z
 wt=dat_obj$par$wt   # rotated w

 #fit model
 results=LSMfit(X,2)

 #plot the parameter recovery results
 oldpar=par(mfrow=c(2,2))

 s_p=sign(cor(results$z,zt))          # to correct for sign switches in the plots
 s_i=sign(cor(results$w,wt))

 plot(s_p[1,1]*zt[,1],results$z[,1]); abline(0,1)
 plot(s_p[2,2]*zt[,2],results$z[,2]); abline(0,1)
 plot(s_i[1,1]*wt[,1],results$w[,1]); abline(0,1)
 plot(s_i[2,2]*wt[,2],results$w[,2]); abline(0,1)

 par(oldpar)
}
\keyword{models}
\keyword{multivariate}
