% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fun_estimate_mean_treat.R
\name{fit.adj2.adj2c.Super}
\alias{fit.adj2.adj2c.Super}
\title{Covariate-Adjusted Treatment Effect Estimation under the Super-Population Framework}
\usage{
fit.adj2.adj2c.Super(
  Y,
  X,
  A,
  intercept = TRUE,
  pi1 = NULL,
  target = "ATE",
  lc = FALSE
)
}
\arguments{
\item{Y}{Numeric vector of length n containing observed responses.}

\item{X}{Numeric matrix (n x p) of covariates. Centering is required. May include intercept column.}

\item{A}{Binary vector of length n indicating treatment assignment (1 = treatment, 0 = control).}

\item{intercept}{Logical. If TRUE (default), X already contains intercept. Set FALSE if X does not contain intercept.}

\item{pi1}{Default is NULL. The assignment probability for the randomization assignment.}

\item{target}{A character string specifying the target estimand. Must be one of:
- `"ATE"` (default): Average Treatment Effect (difference between treatment and control arms).
- `"EY1"`: Expected outcome under treatment (estimates the effect for the treated group).
- `"EY0"`: Expected outcome under control (estimates the effect for the control group).}

\item{lc}{Default is FALSE. If TRUE, then performs linear calibration to achieve efficiency gain using \eqn{\hat{\mu}_0(X_i)} and \eqn{\hat{\mu}_1(X_i)}.}
}
\value{
A list containing three named vectors, including point estimates and variance estimates:
\describe{
  \item{tau_vec}{Point estimates:
    \itemize{
      \item{\code{adj2}:} Point estimation of the HOIF-inspired debiased estimator (Zhao et al., 2024).
      \item{\code{adj2c}:} Point estimation of the the HOIF-inspired debiased estimator (Zhao et al., 2024), which is also the debiased estimator given by Lu et al. (2023).
    }}
  \item{var_infl_vec}{Influence function-based variance estimates:
    \itemize{
      \item{\code{adj2}:} Variance for \code{adj2} via the sample variance of its influence function formula.
      \item{\code{adj2c}:} Variance for \code{adj2c} via the sample variance of its influence function formula.
    }}
  \item{var_rb_vec}{Variance estimates inspired by Bannick et al. (2025):
    \itemize{
      \item{\code{adj2}:} Variance  for \code{adj2} following the asymptotic variance given by Bannick et al. (2025).
      \item{\code{adj2c}:} Variance  for \code{adj2c} following the asymptotic variance given by Bannick et al. (2025).
    }}
}
}
\description{
Implements HOIF-inspired debiased estimators for average treatment effect (ATE)  or treatment effect on the treatment/control arm with variance estimation
using influence function-based and asymptotic-variance. Designed for randomized experiments with moderately high-dimensional covariates.
}
\examples{

set.seed(120)
alpha0 <- 0.1;
n <- 400;

p0 <- ceiling(n * alpha0)
beta0_full <- 1 / (1:p0) ^ (1 / 2) * (-1) ^ c(1:p0)
beta <- beta0_full / norm(beta0_full,type='2')

Sigma_true <- matrix(0, nrow = p0, ncol = p0)
for (i in 1:p0) {
  for (j in 1:p0) {
    Sigma_true[i, j] <- 0.1 ** (abs(i - j))
  }
}

X <- mvtnorm::rmvt(n, sigma = Sigma_true, df = 3)

lp0 <- X \%*\% beta
delta_X <- 1  -  1/4 * X[, 2] -  1/8 * X[, 3]
lp1 <- lp0 + delta_X

Y0 <- lp0 + rnorm(n)
Y1 <- lp1 + rnorm(n)


pi1 <- 1 / 2
A <- rbinom(n, size = 1, prob = pi1)
Y <- A * Y1 + (1 - A) * Y0

Xc <- cbind(1, scale(X, scale = FALSE))
result.adj2.adj2c.sp.ate.ls <- fit.adj2.adj2c.Super(Y, Xc, A, intercept = TRUE,
                                                    target = 'ATE', lc = TRUE)
result.adj2.adj2c.sp.ate.ls
result.adj2.adj2c.sp.treat.ls <- fit.adj2.adj2c.Super(Y, Xc, A, intercept = TRUE,
                                                      target = 'EY1', lc = TRUE)
result.adj2.adj2c.sp.treat.ls
result.adj2.adj2c.sp.control.ls <- fit.adj2.adj2c.Super(Y, Xc, A, intercept = TRUE,
                                                        target = 'EY0', lc = TRUE)
result.adj2.adj2c.sp.control.ls
}
\references{
Bannick, M. S., Shao, J., Liu, J., Du, Y., Yi, Y. and Ye, T. (2025) \emph{A General Form of Covariate Adjustment in Clinical Trials under Covariate-Adaptive Randomization. Biometrika, Vol. xx(x), 1-xx}, \doi{10.1093/biomet/asaf029}.\cr
Lu, X., Yang, F. and Wang, Y. (2023) \emph{Debiased regression adjustment in completely randomized experiments with moderately high-dimensional covariates. arXiv preprint, arXiv:2309.02073}, \doi{10.48550/arXiv.2309.02073}. \cr
Zhao, S., Wang, X., Liu, L. and Zhang, X. (2024) \emph{Covariate Adjustment in Randomized Experiments Motivated by Higher-Order Influence Functions. arXiv preprint, arXiv:2411.08491}, \doi{10.48550/arXiv.2411.08491}.
}
