findmleHSMMnostarting <- function(J, x, obsdist, dwelldist, M = NA, no.initials = 50, verbose = TRUE, seed = NULL, ...) {

  # Set seed if provided for reproducibility
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Input validation
  if (!is.numeric(J) || floor(J) != J || J < 2) {
    stop("J must be an integer strictly greater than 1")
  }

  if (!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("observation distribution is not supported")
  }

  if (!dwelldist %in% c("pois", "nbinom", "betabinom")) {
    stop("dwell distribution is not supported")
  }

  # Initialize tracking variables
  n <- length(x)
  iter <- 1
  min_loglik <- -1e300
  best_HSMM <- NULL
  best_estimate <- NULL
  best_iter <- NULL

  # Pre-compute parameter candidates from data subsets for GEV
  if (obsdist == "gev") {
    locls <- c()
    scalels <- c()
    shapels <- c()

    i <- 1
    attempts <- 0
    max_attempts <- 20000

    # Generate GEV parameter candidates from random data subsets
    while (length(locls) < 1000 && attempts < max_attempts) {
      attempts <- attempts + 1
      tryCatch({
        startpoint<- runif(1, 1, n)
        length <- runif(1, 10, 100)
        if(startpoint + length>n){
          data <- x[startpoint : length(n)]
        } else {
          data <- x[startpoint : (startpoint+length)]
        }
        estimate <- evd::fgev(data)$estimate
        if (!any(is.na(estimate)) && !any(is.infinite(estimate))) {
          locls[i] <- estimate[1]    # location parameter
          scalels[i] <- estimate[2]  # scale parameter
          shapels[i] <- estimate[3]  # shape parameter
          i <- i + 1
        }
      }, error = function(e) {}, warning = function(w) {})
    }

    if (length(locls) < 100) {
      warning(paste("Could only obtain", length(locls), "valid GEV estimates out of", attempts, "attempts"))
    }
  }

  # Helper function to generate new parameters for each attempt
  generate_new_parameters <- function() {
    # Initialize parameter structure
    start <- list(obspar = list(), dwellpar = list())

    # Use clustering-based initialization for non-GEV distributions
    if (obsdist != "gev") {
      room <- rpois(1, 3)  # Random minimum consecutive observations
      cluster_result <- tryCatch({
        clusterHSMM(J, x, obsdist, dwelldist, min_consecutive = room, seed = seed)
      }, error = function(e) {
        if (verbose) message(paste("clusterHSMM error:", e$message))
        NULL
      })

      # Return NULL if clustering failed - this will trigger a retry
      if (is.null(cluster_result) || any(sapply(cluster_result, function(param_list) any(is.na(param_list))))) {
        return(NULL)
      }

      start <- cluster_result
    }

    # Sample GEV parameters from pre-computed candidates
    if (obsdist == "gev") {
      start$obspar$loc <- numeric(J)
      start$obspar$scale <- numeric(J)
      start$obspar$shape <- numeric(J)

      for (j in 1:J) {
        start$obspar$loc[j] <- sample(locls, 1, replace = TRUE)
        start$obspar$scale[j] <- sample(scalels, 1, replace = TRUE)
        start$obspar$shape[j] <- sample(shapels, 1, replace = TRUE)
      }
    }

    # Generate random transition matrix (off-diagonal elements only)
    Pi <- matrix(runif(J * J, 0, 1), ncol = J)
    diag(Pi) <- 0  # No self-transitions in semi-Markov models
    Pi <- Pi / matrix(rep(apply(Pi, 1, sum), each = J), nrow = J)  # Row normalize

    # Generate random initial distribution
    delta <- runif(J, 0, 1)
    delta <- delta / sum(delta)

    # Add noise to observation parameters to avoid identical starts
    if (!is.null(start$obspar)) {
      for (param_name in names(start$obspar)) {
        param_vector <- start$obspar[[param_name]]
        for (j in 1:length(param_vector)) {
          if (abs(param_vector[j]) > 1e-10) {
            range <- abs(param_vector[j]) / 4  # 25% noise range
            param_vector[j] <- param_vector[j] + runif(1, -range, range)

            # Enforce parameter constraints
            if (obsdist != "gev") {
              if (param_name %in% c("lambda", "shape", "scale", "size", "mu", "rate", "sdlog", "sd") &&
                  param_vector[j] <= 0) {
                param_vector[j] <- abs(param_vector[j])  # Positive parameters
              }
            }
            if (obsdist == "gev") {
              if (param_name == "scale" && param_vector[j] <= 0) {
                param_vector[j] <- abs(param_vector[j])  # Scale must be positive
              }
            }
            if (param_name == "pi" && (param_vector[j] < 0 || param_vector[j] > 1)) {
              param_vector[j] <- max(0, min(1, param_vector[j]))  # Probabilities in [0,1]
            }
          }
        }
        start$obspar[[param_name]] <- param_vector
      }
    }

    # Initialize dwell-time parameters for GEV case
    if(obsdist=="gev"){
      if(dwelldist=="pois"){
        start$dwellpar$lambda <- rpois(J, 10)
        start$dwellpar$shift <- rep(1, J)
      }

      if(dwelldist=="nbinom"){
        K <- rpois(1, 20)
        start$dwellpar$mu <- runif(J, 3, K)
        start$dwellpar$size <- numeric(J)
        for(i in 1:J){
          start$dwellpar$size[i] <- runif(1, start$dwellpar$mu[i]/5, 3*start$dwellpar$mu[i]/5)
        }
        start$dwellpar$shift <- rep(1, J)
      }
    }

    # Add noise to dwell-time parameters
    if(!is.null(start$dwellpar)){
      for (param_name in names(start$dwellpar)) {
        param_vector <- start$dwellpar[[param_name]]
        for (j in 1:length(param_vector)) {
          if (abs(param_vector[j]) > 1e-10 && param_name!="shift") {
            range <- abs(param_vector[j]) / 4
            param_vector[j] <- param_vector[j] + runif(1, -range, range)

            # Enforce positive constraints
            if (param_name %in% c("lambda", "size", "mu", "alpha", "beta") && param_vector[j] <= 0) {
              param_vector[j] <- abs(param_vector[j])
            }
          }
        }
        start$dwellpar[[param_name]] <- param_vector
      }
    }

    return(list(obspar = start$obspar, dwellpar = start$dwellpar, Pi = Pi, delta = delta))
  }

  # Main initialization loop
  while (iter <= no.initials) {

    if (!is.null(seed)) set.seed(seed + iter)

    success <- FALSE
    attempt_count <- 0
    max_attempts_per_iter <- 100 # Prevent infinite loops

    # Keep trying until we get a successful fit for this iteration
    while (!success && attempt_count < max_attempts_per_iter) {
      attempt_count <- attempt_count + 1

      # Generate new parameters for each attempt
      new_params <- generate_new_parameters()

      # Skip this attempt if parameter generation failed (e.g., clustering failed)
      if (is.null(new_params)) {
        next
      }

      # Extract parameters
      obspar <- new_params$obspar
      dwellpar <- new_params$dwellpar
      Pi <- new_params$Pi
      delta <- new_params$delta

      # Fit HSMM with current initialization
      tryCatch({
        HSMM <- findmleHSMM(x = x, J = J, M = M, obsdist = obsdist, dwelldist = dwelldist,
                            obspar = obspar, dwellpar = dwellpar,
                            Pi = Pi, delta = delta, verbose = verbose,
                            seed = seed, ...)
        current_loglik <- HSMM$loglikelihoods[length(HSMM$loglikelihoods)]

        # Keep track of best fit
        if (min_loglik <= current_loglik) {
          best_HSMM <- HSMM
          best_estimate <- HSMM
          best_iter <- iter
          min_loglik <- current_loglik
        }

        if (verbose) message(paste("iteration=", iter, sep=""))
        success <- TRUE  # Mark as successful to exit the inner loop

      }, error = function(e) {
        if (verbose) {
          message(paste("Error in attempt", attempt_count, "for iteration", iter, ":", e$message))
          if (attempt_count < max_attempts_per_iter) {
            message("Generating new parameters and trying again...")
          }
        }
        # success remains FALSE, so we'll try again with NEW parameters
      })
    }

    # Check if we exceeded max attempts for this iteration
    if (!success) {
      warning(paste("Failed to find valid parameters after", max_attempts_per_iter,
                    "attempts for iteration", iter, ". Skipping this iteration."))
    }

    # Only increment iteration counter after we've either succeeded or given up
    iter <- iter + 1
  }

  return(HSMM=best_HSMM)
}
