dbetabinom <- function(x, size, alpha, beta, log = FALSE) {
  # Validate shape parameters
  if (alpha <= 0 | beta <= 0) {
    stop("alpha and beta must be positive")
  }

  # Initialize result vector
  d <- numeric(length(x))

  # Identify invalid values (outside support)
  invalid <- x < 0 | x > size
  d[invalid] <- 0
  valid <- !invalid

  # Calculate probabilities for valid values
  if (any(valid)) {
    x_valid <- x[valid]

    # Use log-scale computation for numerical stability:
    # log P(X=x) = log C(size,x) + log B(x+alpha, size-x+beta) - log B(alpha,beta)
    logd <- lchoose(size, x_valid) +
      lbeta(x_valid + alpha, size - x_valid + beta) -
      lbeta(alpha, beta)

    if (log) {
      d[valid] <- logd
    } else {
      d[valid] <- exp(logd)  # Transform back to probability scale
    }
  }

  return(d)
}



rbetabinom <- function(n, size, alpha, beta) {
  # Validate all parameters are positive
  if (alpha <= 0 || beta <= 0 || size <= 0) {
    stop("alpha, beta and size must all be positive")
  }

  # 1. Generate success probability from Beta(alpha, beta)
  p <- rbeta(n, alpha, beta)

  # 2. Generate count from Binomial(size, p) using random probability
  x <- rbinom(n, size, p)

  return(x)
}

