#ifndef ROOT_ILLINOIS_H
#define ROOT_ILLINOIS_H

#include "roots.h"

namespace root_finding
{

/**
 * @brief The "Illinois" method. Provide a function object and an initial
 * valid bracket. Bracket is valid if the sign of the function differs at
 * the end points.
 *
 * @details The "Illinois" method, so called because it was developed at
 * the University of Illinois in the 1950s, is effectively the same as
 * "regula falsi" except it catches the failure mode of the regula falsi.
 *
 * Regula falsi is slow if the same end point is retained twice in a row.
 * In the Illinois method, if the same end point is retained twice then
 * the value of the function is reduced by half for computing the next
 * iterate.
 *
 * Let \f$ [x_{n-1}, x_n]\f$ be a bracket, where the "right" side is most recent
 * endpoint. It is possible to have $x_n < x_{n-1}$ even though intervals are not
 * normally written that way. A new point is generated by the secant formula:
 *
 * \f[ x_{n+1} = \frac{f(x_n) x_{n-1} - f(x_{n-1}) x_{n}}{f(x_n) - f(x_{n-1})} \f]
 *
 * If the sign of \f$f(x_{n+1})\f$ is the same as \f$f(x_{n-1})\f$ then we form
 * a new bracket \f$ [x_n , x_{n+1} ] \f$ as normal in "regula falsi". Otherwise,
 * if the sign of \f$f(x_{n+1})\f$ is the same as \f$f(x_{n})\f$ then we build a
 * new bracket \f$[x_{n-1} , x_{n+1} ]\f$ but for the next iteration, we use the
 * value \f$f(x_{n-1})/2 \f$ instead of \f$f(x_{n-1})\f$. In practice, the saved
 * result `left.y = f(left.x)` is simply halved: ` left.y /= 2 `.
 *
 * The illinois method is the simplest of a family of methods, which
 * all rescale the value of the function `f` at the retained endpoint when
 * that endpoint is retained for a second iteration. `left.y *= gamma`.
 *
 * This method is basically always faster than "regula falsi" and has
 * robustness similar to the bisection method.
 *
 * References:
 * - Ford, J. A. (1995). "Improved Illinois-type methods for the solution
 *   of nonlinear equations." Technical Report, University of Essex Press.
 *
 * - Dowell, M.; Jarratt, P. (1971). "A modified regula falsi method for
 *   computing the root of an equation". BIT. 11 (2): 168–174.
 *   doi:10.1007/BF01934364
 */
struct illinois : public root_finding_method<illinois> {
    using root_finding_method::root_finding_method;
    // does not preserve left and right. Treats right as best guess.
    template <typename F>
    bool iterate(F&& fun)
    {
        proposal.x = get_secant_update(left, right);
        proposal.y = fun(proposal.x);
        update_bracket();
        return true;
    }

    void update_bracket()
    {
        if (opposite_signs(proposal.y, right.y)) {
            left = right;
        } else {
            left.y *= 0.5;
        }
        right = proposal;
    }

    // same as bisection

    graph_t left;
    graph_t right;
    graph_t proposal;

    template <typename F>
    bool initialize(F&& fun, double a, double b)
    {
        left.x = a;
        right.x = b;
        left.y = fun(a);
        right.y = fun(b);
        proposal = left;

        if (is_zero(left.y)) {
            flag = Flag::residual_zero;
            return false;
        }

        if (is_zero(right.y)) {
            flag = Flag::residual_zero;
            proposal = right;
            return false;
        }

        if (same_signs(left.y, right.y)) {
            flag = Flag::invalid_bracket;
            return false;
        }

        return true;
    }

    bool has_converged()
    {
        if (is_zero(proposal.y)) {
            flag = Flag::residual_zero;
            return true;
        }

        if (is_close(left.x, right.x)) {
            flag = Flag::bracket_width_zero;
            return true;
        }

        return false;
    }

    double root() const
    {
        return proposal.x;
    }

    double residual() const
    {
        return proposal.y;
    }
};
}  // namespace root_finding
#endif
