/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5Cprivate_H
#define H5Cprivate_H

#include "H5Cpublic.h" 

#include "H5private.h"  
#include "H5Fprivate.h" 

#define H5C__MAX_NUM_TYPE_IDS 30
#define H5C__PREFIX_LEN       32

#define H5C_MAX_ENTRY_SIZE ((size_t)(32 * 1024 * 1024))

#ifdef H5_HAVE_PARALLEL

#define H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS 1
#else 

#define H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS 0
#endif 

#define H5C__CLASS_NO_FLAGS_SET          ((unsigned)0x0)
#define H5C__CLASS_SPECULATIVE_LOAD_FLAG ((unsigned)0x1)

#define H5C__CLASS_SKIP_READS  ((unsigned)0x2)
#define H5C__CLASS_SKIP_WRITES ((unsigned)0x4)

#define H5C__SERIALIZE_NO_FLAGS_SET ((unsigned)0)
#define H5C__SERIALIZE_RESIZED_FLAG ((unsigned)0x1)
#define H5C__SERIALIZE_MOVED_FLAG   ((unsigned)0x2)

#define H5C__MAX_MAX_CACHE_SIZE ((size_t)(128 * 1024 * 1024))
#define H5C__MIN_MAX_CACHE_SIZE ((size_t)(1024))

#define H5C__DEFAULT_MAX_CACHE_SIZE ((size_t)(4 * 1024 * 1024))
#define H5C__DEFAULT_MIN_CLEAN_SIZE ((size_t)(2 * 1024 * 1024))

#define H5C_RESIZE_CFG__VALIDATE_GENERAL      0x1
#define H5C_RESIZE_CFG__VALIDATE_INCREMENT    0x2
#define H5C_RESIZE_CFG__VALIDATE_DECREMENT    0x4
#define H5C_RESIZE_CFG__VALIDATE_INTERACTIONS 0x8

#define H5C_RESIZE_CFG__VALIDATE_ALL      \
(                                         \
    H5C_RESIZE_CFG__VALIDATE_GENERAL |    \
    H5C_RESIZE_CFG__VALIDATE_INCREMENT |  \
    H5C_RESIZE_CFG__VALIDATE_DECREMENT |  \
    H5C_RESIZE_CFG__VALIDATE_INTERACTIONS \
)

#define H5C__CURR_AUTO_SIZE_CTL_VER       1
#define H5C__CURR_AUTO_RESIZE_RPT_FCN_VER 1
#define H5C__CURR_CACHE_IMAGE_CTL_VER     1

#define H5C__DEF_AR_UPPER_THRESHHOLD 0.9999
#define H5C__DEF_AR_LOWER_THRESHHOLD 0.9
#define H5C__DEF_AR_MAX_SIZE         ((size_t)(16 * 1024 * 1024))
#define H5C__DEF_AR_INIT_SIZE        ((size_t)(1 * 1024 * 1024))
#define H5C__DEF_AR_MIN_SIZE         ((size_t)(1 * 1024 * 1024))
#define H5C__DEF_AR_MIN_CLEAN_FRAC   0.5
#define H5C__DEF_AR_INCREMENT        2.0
#define H5C__DEF_AR_MAX_INCREMENT    ((size_t)(2 * 1024 * 1024))
#define H5C__DEF_AR_FLASH_MULTIPLE   1.0
#define H5C__DEV_AR_FLASH_THRESHOLD  0.25
#define H5C__DEF_AR_DECREMENT        0.9
#define H5C__DEF_AR_MAX_DECREMENT    ((size_t)(1 * 1024 * 1024))
#define H5C__DEF_AR_EPCHS_B4_EVICT   3
#define H5C__DEF_AR_EMPTY_RESERVE    0.05
#define H5C__MIN_AR_EPOCH_LENGTH     100
#define H5C__DEF_AR_EPOCH_LENGTH     50000
#define H5C__MAX_AR_EPOCH_LENGTH     1000000

#define H5C__NO_FLAGS_SET                   0x00000
#define H5C__DELETED_FLAG                   0x00001
#define H5C__DIRTIED_FLAG                   0x00002
#define H5C__PIN_ENTRY_FLAG                 0x00004
#define H5C__UNPIN_ENTRY_FLAG               0x00008
#define H5C__FLUSH_INVALIDATE_FLAG          0x00010
#define H5C__FLUSH_CLEAR_ONLY_FLAG          0x00020
#define H5C__FLUSH_IGNORE_PROTECTED_FLAG    0x00040
#define H5C__READ_ONLY_FLAG                 0x00080
#define H5C__FREE_FILE_SPACE_FLAG           0x00100
#define H5C__TAKE_OWNERSHIP_FLAG            0x00200
#define H5C__FLUSH_LAST_FLAG                0x00400
#define H5C__FLUSH_COLLECTIVELY_FLAG        0x00800
#define H5C__EVICT_ALLOW_LAST_PINS_FLAG     0x01000
#define H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG 0x02000
#define H5C__DURING_FLUSH_FLAG              0x04000 
#define H5C__GENERATE_IMAGE_FLAG            0x08000 
#define H5C__UPDATE_PAGE_BUFFER_FLAG        0x10000 

#ifndef NDEBUG
#ifndef H5C_DO_SANITY_CHECKS
#define H5C_DO_SANITY_CHECKS
#endif

#ifndef H5C_DO_TAGGING_SANITY_CHECKS
#define H5C_DO_TAGGING_SANITY_CHECKS
#endif
#endif

#define H5C__SET_CORK   0x1
#define H5C__UNCORK     0x2
#define H5C__GET_CORKED 0x4

#define H5C_DO_MEMORY_SANITY_CHECKS 0

#ifndef NDEBUG
#define H5C_COLLECT_CACHE_STATS 1
#else 
#define H5C_COLLECT_CACHE_STATS 0
#endif 

#if H5C_COLLECT_CACHE_STATS
#define H5C_COLLECT_CACHE_ENTRY_STATS 1
#else
#define H5C_COLLECT_CACHE_ENTRY_STATS 0
#endif 

typedef struct H5C_t H5C_t;

typedef enum H5C_notify_action_t {
    H5C_NOTIFY_ACTION_AFTER_INSERT,       
    H5C_NOTIFY_ACTION_AFTER_LOAD,         
    H5C_NOTIFY_ACTION_AFTER_FLUSH,        
    H5C_NOTIFY_ACTION_BEFORE_EVICT,       
    H5C_NOTIFY_ACTION_ENTRY_DIRTIED,      
    H5C_NOTIFY_ACTION_ENTRY_CLEANED,      
    H5C_NOTIFY_ACTION_CHILD_DIRTIED,      
    H5C_NOTIFY_ACTION_CHILD_CLEANED,      
    H5C_NOTIFY_ACTION_CHILD_UNSERIALIZED, 
    H5C_NOTIFY_ACTION_CHILD_SERIALIZED    
} H5C_notify_action_t;

typedef herr_t (*H5C_get_initial_load_size_func_t)(void *udata_ptr, size_t *image_len_ptr);
typedef herr_t (*H5C_get_final_load_size_func_t)(const void *image_ptr, size_t image_len, void *udata_ptr,
                                                 size_t *actual_len_ptr);
typedef htri_t (*H5C_verify_chksum_func_t)(const void *image_ptr, size_t len, void *udata_ptr);
typedef void *(*H5C_deserialize_func_t)(const void *image_ptr, size_t len, void *udata_ptr, bool *dirty_ptr);
typedef herr_t (*H5C_image_len_func_t)(const void *thing, size_t *image_len_ptr);
typedef herr_t (*H5C_pre_serialize_func_t)(H5F_t *f, void *thing, haddr_t addr, size_t len,
                                           haddr_t *new_addr_ptr, size_t *new_len_ptr, unsigned *flags_ptr);
typedef herr_t (*H5C_serialize_func_t)(const H5F_t *f, void *image_ptr, size_t len, void *thing);
typedef herr_t (*H5C_notify_func_t)(H5C_notify_action_t action, void *thing);
typedef herr_t (*H5C_free_icr_func_t)(void *thing);
typedef herr_t (*H5C_get_fsf_size_t)(const void *thing, hsize_t *fsf_size_ptr);

typedef struct H5C_class_t {
    int                              id;
    const char                      *name;
    H5FD_mem_t                       mem_type;
    unsigned                         flags;
    H5C_get_initial_load_size_func_t get_initial_load_size;
    H5C_get_final_load_size_func_t   get_final_load_size;
    H5C_verify_chksum_func_t         verify_chksum;
    H5C_deserialize_func_t           deserialize;
    H5C_image_len_func_t             image_len;
    H5C_pre_serialize_func_t         pre_serialize;
    H5C_serialize_func_t             serialize;
    H5C_notify_func_t                notify;
    H5C_free_icr_func_t              free_icr;
    H5C_get_fsf_size_t               fsf_size;
} H5C_class_t;

typedef herr_t (*H5C_write_permitted_func_t)(const H5F_t *f, bool *write_permitted_ptr);
typedef herr_t (*H5C_log_flush_func_t)(H5C_t *cache_ptr, haddr_t addr, bool was_dirty, unsigned flags);

#define H5C_RING_UNDEFINED 0 
#define H5C_RING_USER      1 
#define H5C_RING_RDFSM     2
#define H5C_RING_MDFSM     3
#define H5C_RING_SBE       4
#define H5C_RING_SB        5 
#define H5C_RING_NTYPES    6

typedef int H5C_ring_t;

typedef struct H5C_cache_entry_t {
    H5C_t             *cache_ptr;
    haddr_t            addr;
    size_t             size;
    void              *image_ptr;
    bool               image_up_to_date;
    const H5C_class_t *type;
    bool               is_dirty;
    bool               dirtied;
    bool               is_protected;
    bool               is_read_only;
    int                ro_ref_count;
    bool               is_pinned;
    bool               in_slist;
    bool               flush_me_last;
#ifdef H5_HAVE_PARALLEL
    bool clear_on_unprotect;
    bool flush_immediately;
    bool coll_access;
#endif 
    bool flush_in_progress;
    bool destroy_in_progress;

    
    H5C_ring_t ring;

    
    struct H5C_cache_entry_t **flush_dep_parent;
    unsigned                   flush_dep_nparents;
    unsigned                   flush_dep_parent_nalloc;
    unsigned                   flush_dep_nchildren;
    unsigned                   flush_dep_ndirty_children;
    unsigned                   flush_dep_nunser_children;
    bool                       pinned_from_client;
    bool                       pinned_from_cache;

    
    struct H5C_cache_entry_t *ht_next;
    struct H5C_cache_entry_t *ht_prev;
    struct H5C_cache_entry_t *il_next;
    struct H5C_cache_entry_t *il_prev;

    
    struct H5C_cache_entry_t *next;
    struct H5C_cache_entry_t *prev;
#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS
    struct H5C_cache_entry_t *aux_next;
    struct H5C_cache_entry_t *aux_prev;
#endif 
#ifdef H5_HAVE_PARALLEL
    struct H5C_cache_entry_t *coll_next;
    struct H5C_cache_entry_t *coll_prev;
#endif 

    
    bool     include_in_image;
    int32_t  lru_rank;
    bool     image_dirty;
    uint64_t fd_parent_count;
    haddr_t *fd_parent_addrs;
    uint64_t fd_child_count;
    uint64_t fd_dirty_child_count;
    uint32_t image_fd_height;
    bool     prefetched;
    int      prefetch_type_id;
    int32_t  age;
    bool     prefetched_dirty;

#ifndef NDEBUG 
    int serialization_count;
#endif 

    
    struct H5C_cache_entry_t *tl_next;
    struct H5C_cache_entry_t *tl_prev;
    struct H5C_tag_info_t    *tag_info;

#if H5C_COLLECT_CACHE_ENTRY_STATS
    
    int32_t accesses;
    int32_t clears;
    int32_t flushes;
    int32_t pins;
#endif 
} H5C_cache_entry_t;

typedef struct H5C_image_entry_t {
    haddr_t    addr;
    size_t     size;
    H5C_ring_t ring;
    int32_t    age;
    int32_t    type_id;
    int32_t    lru_rank;
    bool       is_dirty;
    unsigned   image_fd_height;
    uint64_t   fd_parent_count;
    haddr_t   *fd_parent_addrs;
    uint64_t   fd_child_count;
    uint64_t   fd_dirty_child_count;
    void      *image_ptr;
} H5C_image_entry_t;

enum H5C_resize_status {
    in_spec,
    increase,
    flash_increase,
    decrease,
    at_max_size,
    at_min_size,
    increase_disabled,
    decrease_disabled,
    not_full
}; 

typedef void (*H5C_auto_resize_rpt_fcn)(H5C_t *cache_ptr, int32_t version, double hit_rate,
                                        enum H5C_resize_status status, size_t old_max_cache_size,
                                        size_t new_max_cache_size, size_t old_min_clean_size,
                                        size_t new_min_clean_size);

typedef struct H5C_auto_size_ctl_t {
    
    int32_t                 version;
    H5C_auto_resize_rpt_fcn rpt_fcn;
    bool                    set_initial_size;
    size_t                  initial_size;
    double                  min_clean_fraction;
    size_t                  max_size;
    size_t                  min_size;
    int64_t                 epoch_length;

    
    enum H5C_cache_incr_mode       incr_mode;
    double                         lower_hr_threshold;
    double                         increment;
    bool                           apply_max_increment;
    size_t                         max_increment;
    enum H5C_cache_flash_incr_mode flash_incr_mode;
    double                         flash_multiple;
    double                         flash_threshold;

    
    enum H5C_cache_decr_mode decr_mode;
    double                   upper_hr_threshold;
    double                   decrement;
    bool                     apply_max_decrement;
    size_t                   max_decrement;
    int32_t                  epochs_before_eviction;
    bool                     apply_empty_reserve;
    double                   empty_reserve;
} H5C_auto_size_ctl_t;

#define H5C_CI__GEN_MDCI_SBE_MESG    ((unsigned)0x0001)
#define H5C_CI__GEN_MDC_IMAGE_BLK    ((unsigned)0x0002)
#define H5C_CI__SUPRESS_ENTRY_WRITES ((unsigned)0x0004)
#define H5C_CI__WRITE_CACHE_IMAGE    ((unsigned)0x0008)

#define H5C_CI__ALL_FLAGS ((unsigned)0x000F)

#define H5C__DEFAULT_CACHE_IMAGE_CTL                                                                         \
    {                                                                                                        \
        H5C__CURR_CACHE_IMAGE_CTL_VER,                                                        \
            false,                                                                     \
            false,                                                                 \
            H5AC__CACHE_IMAGE__ENTRY_AGEOUT__NONE,                                       \
            H5C_CI__ALL_FLAGS                                                                   \
    }

typedef struct H5C_cache_image_ctl_t {
    int32_t  version;
    bool     generate_image;
    bool     save_resize_status;
    int32_t  entry_ageout;
    unsigned flags;
} H5C_cache_image_ctl_t;

typedef enum H5C_log_style_t { H5C_LOG_STYLE_JSON, H5C_LOG_STYLE_TRACE } H5C_log_style_t;

H5_DLL H5C_t *H5C_create(size_t max_cache_size, size_t min_clean_size, int max_type_id,
                         const H5C_class_t *const  *class_table_ptr,
                         H5C_write_permitted_func_t check_write_permitted, bool write_permitted,
                         H5C_log_flush_func_t log_flush, void *aux_ptr);
H5_DLL void   H5C_def_auto_resize_rpt_fcn(H5C_t *cache_ptr, int32_t version, double hit_rate,
                                          enum H5C_resize_status status, size_t old_max_cache_size,
                                          size_t new_max_cache_size, size_t old_min_clean_size,
                                          size_t new_min_clean_size);
H5_DLL herr_t H5C_dest(H5F_t *f);
H5_DLL herr_t H5C_evict(H5F_t *f);
H5_DLL herr_t H5C_expunge_entry(H5F_t *f, const H5C_class_t *type, haddr_t addr, unsigned flags);
H5_DLL herr_t H5C_flush_cache(H5F_t *f, unsigned flags);
H5_DLL herr_t H5C_flush_tagged_entries(H5F_t *f, haddr_t tag);
H5_DLL herr_t H5C_evict_tagged_entries(H5F_t *f, haddr_t tag, bool match_global);
H5_DLL herr_t H5C_expunge_tag_type_metadata(H5F_t *f, haddr_t tag, int type_id, unsigned flags);
H5_DLL herr_t H5C_get_tag(const void *thing,  haddr_t *tag);
#ifdef H5C_DO_TAGGING_SANITY_CHECKS
herr_t H5C_verify_tag(int id, haddr_t tag);
#endif
H5_DLL herr_t H5C_flush_to_min_clean(H5F_t *f);
H5_DLL herr_t H5C_get_cache_auto_resize_config(const H5C_t *cache_ptr, H5C_auto_size_ctl_t *config_ptr);
H5_DLL herr_t H5C_get_cache_size(const H5C_t *cache_ptr, size_t *max_size_ptr, size_t *min_clean_size_ptr,
                                 size_t *cur_size_ptr, uint32_t *cur_num_entries_ptr);
H5_DLL herr_t H5C_get_cache_flush_in_progress(const H5C_t *cache_ptr, bool *flush_in_progress_ptr);
H5_DLL herr_t H5C_get_cache_hit_rate(const H5C_t *cache_ptr, double *hit_rate_ptr);
H5_DLL herr_t H5C_get_entry_status(const H5F_t *f, haddr_t addr, size_t *size_ptr, bool *in_cache_ptr,
                                   bool *is_dirty_ptr, bool *is_protected_ptr, bool *is_pinned_ptr,
                                   bool *is_corked_ptr, bool *is_flush_dep_parent_ptr,
                                   bool *is_flush_dep_child_ptr, bool *image_up_to_date_ptr);
H5_DLL herr_t H5C_get_evictions_enabled(const H5C_t *cache_ptr, bool *evictions_enabled_ptr);
H5_DLL void  *H5C_get_aux_ptr(const H5C_t *cache_ptr);
H5_DLL herr_t H5C_insert_entry(H5F_t *f, const H5C_class_t *type, haddr_t addr, void *thing,
                               unsigned int flags);
H5_DLL herr_t H5C_load_cache_image_on_next_protect(H5F_t *f, haddr_t addr, hsize_t len, bool rw);
H5_DLL herr_t H5C_mark_entry_dirty(void *thing);
H5_DLL herr_t H5C_mark_entry_clean(void *thing);
H5_DLL herr_t H5C_mark_entry_unserialized(void *thing);
H5_DLL herr_t H5C_mark_entry_serialized(void *thing);
H5_DLL herr_t H5C_move_entry(H5C_t *cache_ptr, const H5C_class_t *type, haddr_t old_addr, haddr_t new_addr);
H5_DLL herr_t H5C_pin_protected_entry(void *thing);
H5_DLL herr_t H5C_prep_for_file_close(H5F_t *f);
H5_DLL herr_t H5C_create_flush_dependency(void *parent_thing, void *child_thing);
H5_DLL void  *H5C_protect(H5F_t *f, const H5C_class_t *type, haddr_t addr, void *udata, unsigned flags);
H5_DLL herr_t H5C_reset_cache_hit_rate_stats(H5C_t *cache_ptr);
H5_DLL herr_t H5C_resize_entry(void *thing, size_t new_size);
H5_DLL herr_t H5C_set_cache_auto_resize_config(H5C_t *cache_ptr, H5C_auto_size_ctl_t *config_ptr);
H5_DLL herr_t H5C_set_cache_image_config(const H5F_t *f, H5C_t *cache_ptr, H5C_cache_image_ctl_t *config_ptr);
H5_DLL herr_t H5C_set_evictions_enabled(H5C_t *cache_ptr, bool evictions_enabled);
H5_DLL herr_t H5C_set_slist_enabled(H5C_t *cache_ptr, bool slist_enabled, bool populate_slist);
H5_DLL herr_t H5C_set_prefix(H5C_t *cache_ptr, char *prefix);
H5_DLL herr_t H5C_stats(H5C_t *cache_ptr, const char *cache_name, bool display_detailed_stats);
H5_DLL void   H5C_stats__reset(H5C_t *cache_ptr);
H5_DLL herr_t H5C_unpin_entry(void *thing);
H5_DLL herr_t H5C_destroy_flush_dependency(void *parent_thing, void *child_thing);
H5_DLL herr_t H5C_unprotect(H5F_t *f, haddr_t addr, void *thing, unsigned int flags);
H5_DLL herr_t H5C_validate_cache_image_config(H5C_cache_image_ctl_t *ctl_ptr);
H5_DLL herr_t H5C_validate_resize_config(H5C_auto_size_ctl_t *config_ptr, unsigned int tests);
H5_DLL herr_t H5C_ignore_tags(H5C_t *cache_ptr);
H5_DLL bool   H5C_get_ignore_tags(const H5C_t *cache_ptr);
H5_DLL uint32_t H5C_get_num_objs_corked(const H5C_t *cache_ptr);
H5_DLL herr_t   H5C_retag_entries(H5C_t *cache_ptr, haddr_t src_tag, haddr_t dest_tag);
H5_DLL herr_t   H5C_cork(H5C_t *cache_ptr, haddr_t obj_addr, unsigned action, bool *corked);
H5_DLL herr_t   H5C_get_entry_ring(const H5F_t *f, haddr_t addr, H5C_ring_t *ring);
H5_DLL herr_t   H5C_unsettle_entry_ring(void *thing);
H5_DLL herr_t   H5C_unsettle_ring(H5F_t *f, H5C_ring_t ring);
H5_DLL herr_t   H5C_remove_entry(void *thing);
H5_DLL herr_t   H5C_cache_image_status(H5F_t *f, bool *load_ci_ptr, bool *write_ci_ptr);
H5_DLL bool     H5C_cache_image_pending(const H5C_t *cache_ptr);
H5_DLL herr_t   H5C_get_mdc_image_info(const H5C_t *cache_ptr, haddr_t *image_addr, hsize_t *image_len);

H5_DLL herr_t H5C_start_logging(H5C_t *cache);
H5_DLL herr_t H5C_stop_logging(H5C_t *cache);
H5_DLL herr_t H5C_get_logging_status(const H5C_t *cache,  bool *is_enabled,
                                      bool *is_currently_logging);

#ifdef H5_HAVE_PARALLEL
H5_DLL herr_t H5C_apply_candidate_list(H5F_t *f, H5C_t *cache_ptr, unsigned num_candidates,
                                       haddr_t *candidates_list_ptr, int mpi_rank, int mpi_size);
H5_DLL herr_t H5C_construct_candidate_list__clean_cache(H5C_t *cache_ptr);
H5_DLL herr_t H5C_construct_candidate_list__min_clean(H5C_t *cache_ptr);
H5_DLL herr_t H5C_clear_coll_entries(H5C_t *cache_ptr, bool partial);
H5_DLL herr_t H5C_mark_entries_as_clean(H5F_t *f, unsigned ce_array_len, haddr_t *ce_array_ptr);
#endif 

#ifndef NDEBUG 
H5_DLL herr_t H5C_dump_cache(H5C_t *cache_ptr, const char *cache_name);
H5_DLL herr_t H5C_dump_cache_LRU(H5C_t *cache_ptr, const char *cache_name);
H5_DLL bool   H5C_get_serialization_in_progress(const H5C_t *cache_ptr);
H5_DLL bool   H5C_cache_is_clean(const H5C_t *cache_ptr, H5C_ring_t inner_ring);
H5_DLL herr_t H5C_dump_cache_skip_list(H5C_t *cache_ptr, char *calling_fcn);
H5_DLL herr_t H5C_get_entry_ptr_from_addr(H5C_t *cache_ptr, haddr_t addr, void **entry_ptr_ptr);
H5_DLL herr_t H5C_flush_dependency_exists(H5C_t *cache_ptr, haddr_t parent_addr, haddr_t child_addr,
                                          bool *fd_exists_ptr);
H5_DLL herr_t H5C_verify_entry_type(H5C_t *cache_ptr, haddr_t addr, const H5C_class_t *expected_type,
                                    bool *in_cache_ptr, bool *type_ok_ptr);
H5_DLL herr_t H5C_validate_index_list(H5C_t *cache_ptr);
#endif 

#endif 
