/****************************************************************************
 *      $Id: main.c,v 1.2 1998/01/22 05:50:47 kevine Exp $
 *      Copyright (C) 1997, 1998 Kevin Elphinstone, University of New South
 *      Wales
 *
 *      This file is part of the L4/MIPS micro-kernel distribution.
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version 2
 *      of the License, or (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *      
 ****************************************************************************/
#include <stdio.h>
#include <elf.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <assert.h>


main(int argc, char *argv[])
{
    int i, r, bp, bsize, loadcount;
    unsigned int fileend;
    int infile, outfile;
    Elf64_Ehdr *ehdr;

    Elf64_Phdr *phdr;
    char *pbuff;
    
    Elf64_Shdr *shdr;
    char *sbuff;

    Elf32_Ehdr e32;
    Elf32_Phdr p32[6]; /* only six program segments supported by PMON */
    Elf32_Shdr s32 ; /* put in one null section to avoid breaking PMON */
    
    if (argc != 3)
    {
	fprintf(stderr, "Usage: %s infile outfile\n", argv[0]);
	exit(1);
    }

    infile = open(argv[1],O_RDONLY);
    if (infile <0)
    {
	perror("while opening infile");
	exit(1);
    }

    ehdr = malloc(sizeof(Elf64_Ehdr));
    assert(ehdr != NULL);
    
    r = read(infile,ehdr,sizeof(Elf64_Ehdr));

    assert(r == sizeof(Elf64_Ehdr)); /* assume we get enough
					to read file hdr */
    
    /* now check if we have the right file type */
    if (ehdr->e_ident[EI_MAG0] != 0x7f ||
	ehdr->e_ident[EI_MAG1] != 'E' ||
	ehdr->e_ident[EI_MAG2] != 'L' ||
	ehdr->e_ident[EI_MAG3] != 'F')
    {
	fprintf(stderr,"invalid file type, expecting elf format\n");
	exit(1);
    }

    /* check if 64 bit elf*/
    if(ehdr->e_ident[EI_CLASS] != ELFCLASS64)
    {
	fprintf(stderr,"file not elf64 format\n");
	exit(1);
    }

    /* check big endian */
    if(ehdr->e_ident[EI_DATA]!= ELFDATA2MSB)
    {
	fprintf(stderr,"expecting ELFDATA2MSB, got something else?\n");
	exit(1);
    }

    /* check for a mips */
    if (ehdr->e_machine != EM_MIPS)
    {
	fprintf(stderr,"not a mips binary???\n");
	exit(1);
    }

    /* check file is an executable */
    if (ehdr->e_type != ET_EXEC)
    {
	fprintf(stderr,"file not executable\n");
	exit(1);
    }

    if (ehdr->e_flags != EF_MIPS_ARCH_3)
    {
	fprintf(stderr,"expecting mips3 e_flag, got %x instead\n",
		ehdr->e_flags);
	exit(1);
    }

    /* okay lets read in program headers */

    /* move to start */
    assert(ehdr->e_phoff.hi32 == 0);
    r = lseek(infile, ehdr->e_phoff.lo32, SEEK_SET);
    assert(r == ehdr->e_phoff.lo32);

    /* alloc space */
    pbuff = calloc(ehdr->e_phnum,ehdr->e_phentsize);
    assert(pbuff != NULL);

    /* read them in */

    bp = 0;
    bsize = ehdr->e_phnum * ehdr->e_phentsize;
    do {
	r = read(infile, &pbuff[bp], bsize - bp);
	assert(r >= 0);
	bp += r;
    } while (bp != bsize);
	
     /* okay lets read in section headers */

    /* move to start */
    assert(ehdr->e_shoff.hi32 == 0);
    r = lseek(infile, ehdr->e_shoff.lo32, SEEK_SET);
    assert(r == ehdr->e_shoff.lo32);

    /* alloc space */
    sbuff = calloc(ehdr->e_shnum,ehdr->e_shentsize);
    assert(pbuff != NULL);

    /* read them in */

    bp = 0;
    bsize = ehdr->e_shnum * ehdr->e_shentsize;
    do {
	r = read(infile, &sbuff[bp], bsize - bp);
	assert(r >= 0);
	bp += r;
    } while (bp != bsize);

    /* okay we have all the info, lets build a elf32 executable from the
       elf64 one */

    /* fill in what we can now in elf32 header*/

    e32.e_ident[EI_MAG0] = 0x7f;
    e32.e_ident[EI_MAG1] = 'E';
    e32.e_ident[EI_MAG2] = 'L';
    e32.e_ident[EI_MAG3] = 'F';
    e32.e_ident[EI_CLASS] = ELFCLASS32;
    e32.e_ident[EI_DATA] = ELFDATA2MSB;
    e32.e_ident[EI_VERSION] = EV_CURRENT;

    e32.e_type = ET_EXEC;
    e32.e_machine = EM_MIPS;
    e32.e_version = EV_CURRENT;
    e32.e_entry = ehdr->e_entry.lo32; /* assume linked somewhere that will
					 work */
    e32.e_flags = EF_MIPS_ARCH_3;
    e32.e_ehsize = sizeof(Elf32_Ehdr);
    e32.e_phentsize = sizeof(Elf32_Phdr);
    e32.e_shentsize = sizeof(Elf32_Shdr);
    e32.e_shstrndx = SHN_UNDEF;

    /* fill out section 0 */
    s32.sh_name = 0;
    s32.sh_type = SHT_NULL;
    s32.sh_flags = 0;
    s32.sh_addr = 0;
    s32.sh_offset = 0;
    s32.sh_size = 0;
    s32.sh_link = SHN_UNDEF;
    s32.sh_info = 0;
    s32.sh_addralign = 0;
    s32.sh_entsize = 0;


    /* open the outfile now */
    outfile = open(argv[2],O_TRUNC | O_WRONLY | O_CREAT, 0644);
    if (outfile < 0)
    {
	perror("while opening outfile");
	exit(1);
    }

    /* Hmmm, now go through each program header in turn */
    loadcount = 0;
    fileend = 0;
    for (i = 0; i < ehdr->e_phnum; i++)
    {
	phdr = (Elf64_Phdr *) &pbuff[i * ehdr->e_phentsize];
	if (phdr->p_type == PT_LOAD)
	{
	    int pi, po;
	    char buff[1024];
	    
	    /* we have a segment to load so build a elf32 phdr
	     and copy segment into new file */
	    p32[loadcount].p_type = PT_LOAD;
	    p32[loadcount].p_flags = phdr->p_flags;
	    
	    assert(phdr->p_offset.hi32 == 0);
	    p32[loadcount].p_offset = phdr->p_offset.lo32;
	    
	    assert(phdr->p_vaddr.hi32 == 0xffffffff);
	    p32[loadcount].p_vaddr = phdr->p_vaddr.lo32;

	    assert(phdr->p_paddr.hi32 == 0xffffffff);
	    p32[loadcount].p_paddr = phdr->p_paddr.lo32;

	    assert(phdr->p_filesz.hi32 == 0);
	    p32[loadcount].p_filesz = phdr->p_filesz.lo32;

	    assert(phdr->p_memsz.hi32 == 0);
	    p32[loadcount].p_memsz = phdr->p_memsz.lo32;

	    assert(phdr->p_align.hi32 == 0);
	    p32[loadcount].p_align = phdr->p_align.lo32;
	    
	    r = lseek(outfile, p32[loadcount].p_offset, SEEK_SET);
	    assert(r == p32[loadcount].p_offset);
	    
	    r = lseek(infile, p32[loadcount].p_offset, SEEK_SET);
	    assert(r == p32[loadcount].p_offset);

	    if (p32[loadcount].p_offset + p32[loadcount].p_filesz > fileend)
	    {
		fileend = p32[loadcount].p_offset + p32[loadcount].p_filesz;
	    }
	    
	    pi = po = 0;
	    bsize = p32[loadcount].p_filesz;
	    do {
		r = read(infile, buff,
			 (p32[loadcount].p_filesz - po) < 1024 ?
			 (p32[loadcount].p_filesz - po) : 1024);
		assert(r >= 0);
		
		pi = r;
		bp = 0;
		
		while (bp != pi)
		{
		    r = write(outfile, &buff[bp], pi - bp);
		    assert(r >= 0);
		    bp += r;
		    po += r;
		}
	    } while (po != bsize);
	    
	    loadcount++;
	}
	
    }
    assert(loadcount > 0);
    /* now lets fix up the rest of the headers and write them out */

    e32.e_phoff = e32.e_ehsize;
    e32.e_phnum = loadcount;
    e32.e_shoff = fileend;
    e32.e_shnum =  1;
    
    r = lseek(outfile, 0, SEEK_SET);
    assert(r == 0);
    
    bp = 0;
    bsize = e32.e_ehsize;
    do {
	r = write(outfile, &((char *)(&e32))[bp], bsize - bp);
	assert(r >= 0);
	bp += r;
    } while (bp != bsize);
    
    r = lseek(outfile, fileend, SEEK_SET);
    assert(r == fileend);

    bp = 0;
    bsize = e32.e_shentsize;
    do {
	r = write(outfile, &((char *)(&s32))[bp], bsize - bp);
	assert(r >= 0);
	bp += r;
    } while (bp != bsize);
    
    r = lseek(outfile, e32.e_ehsize, SEEK_SET);
    assert(r == e32.e_ehsize);

    for (i = 0; i < loadcount; i++)
    {
	bp = 0;
	bsize = e32.e_phentsize;
	do {
	    r = write(outfile, &((char *)(&p32[i]))[bp], bsize - bp);
	    assert(r >= 0);
	    bp += r;
	} while (bp != bsize);
    }
    
    close(outfile);
    close(infile);
    exit(0);
}




