/****************************************************************************
 *      $Id: append_elf32.c,v 1.3 1998/01/27 01:10:04 kevine Exp $
 *      Copyright (C) 1997, 1998 Kevin Elphinstone, University of New South
 *      Wales
 *
 *      This file is part of the L4/MIPS micro-kernel distribution.
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version 2
 *      of the License, or (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *      
 ****************************************************************************/
#include "elf.h"
#include "protos.h"
#include "../../include/kernel/dit.h"

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/stat.h>

void check_elf32(Elf32_Ehdr *ehdr)
{


  /* now check if we have the right file type */
  if (ehdr->e_ident[EI_MAG0] != 0x7f ||
      ehdr->e_ident[EI_MAG1] != 'E' ||
      ehdr->e_ident[EI_MAG2] != 'L' ||
      ehdr->e_ident[EI_MAG3] != 'F')
  {
    fprintf(stderr,"invalid file type, expecting elf format\n");
    exit(1);
  }
  
  /* check if 32 bit elf*/
  if(ehdr->e_ident[EI_CLASS] != ELFCLASS32)
  {
	fprintf(stderr,"file not correct class (elf32)\n");
	exit(1);
  }
  
  /* check big endian */
  if(ehdr->e_ident[EI_DATA]!= ELFDATA2MSB)
  {
    fprintf(stderr,"expecting ELFDATA2MSB, got something else?\n");
    exit(1);
  }

  /* check for a mips */
  if (swap16(ehdr->e_machine) != EM_MIPS)
  {
    fprintf(stderr,"not a mips binary???\n");
    exit(1);
  }

  /* check file is an executable */
  if (swap16(ehdr->e_type) != ET_EXEC)
  {
    fprintf(stderr,"file not executable\n");
    exit(1);
  }
  
  if ((swap32(ehdr->e_flags) != EF_MIPS_ARCH_3) &&
      (swap32(ehdr->e_flags) != (EF_MIPS_ARCH_3 | EF_MIPS_ABI2) ))
  {
    fprintf(stderr,"expecting mips3 e_flag, got %x instead\n",
	    ehdr->e_flags);
    exit(1);
  }
}
void append_32_file(int c, char *v[])
{
  char *appendname, *imagename;
  int r, imagefile, appendfile, bp, bsize, i, padding;
  unsigned int vaddr_base, file_offset_base, cur_file_offset;
  Elf32_Ehdr *ehdr;
  Elf32_Phdr *phdr;
  Elf32_Ehdr *e32;
  Elf32_Phdr *p32;
  char *p64buff;
  
  
  Dit_Dhdr *dhdr;
  Dit_Phdr *dphdr;
  char *dbuff;
  Elf32_Shdr s32;
  /* open file */

  if (c != 2)
  {
    usage();
  }
  appendname = v[0];
  imagename = v[1];
  imagefile = open(imagename,O_RDWR);
  if (imagefile <0)
  {
    perror("while opening imagefile");
    exit(1);
  }

  
  /* check if elf 32 */
  
  ehdr = malloc(sizeof(Elf32_Ehdr));
  assert(ehdr != NULL);
   
  r = read(imagefile,ehdr,sizeof(Elf32_Ehdr));
    
  assert(r == sizeof(Elf32_Ehdr)); /* assume we get enough
				      to read file hdr */
  check_elf32(ehdr);


  /* move to the last program header table and read in */

  assert(ehdr->e_phnum > 1);
  r = lseek(imagefile, ehdr->e_phoff + (ehdr->e_phnum -1) * ehdr->e_phentsize
	    , SEEK_SET);
  assert(r == ehdr->e_phoff + (ehdr->e_phnum -1) * ehdr->e_phentsize);
  
  phdr = malloc(sizeof(Elf32_Phdr));
  assert(phdr != NULL);

  r = read(imagefile,phdr,sizeof(Elf32_Phdr));
    
  assert(r == sizeof(Elf32_Phdr)); /* assume we get enough
				      to read file hdr */

  /* now move to DIT header and read in */
  
  r = lseek(imagefile, phdr->p_offset, SEEK_SET);
  assert(r == phdr->p_offset);

  
  dhdr = (Dit_Dhdr *) malloc(DHDR_SEG_SIZE);
  dbuff = (char *)dhdr;
  assert(dhdr != NULL);

  r = read(imagefile,dhdr,DHDR_SEG_SIZE);
    
  assert(r == DHDR_SEG_SIZE); /* assume we get enough
				      to read file hdr */

  /* now check if correct format */

  if (dhdr->d_ident[0] != 'd' ||
      dhdr->d_ident[1] != 'h' ||
      dhdr->d_ident[2] != 'd' ||
      dhdr->d_ident[3] != 'r')
  {
    fprintf(stderr,"file doesn't contain dhdr\n");
    exit(1);
  }

  /* okay we can do our business */
  /* open the file to append */
  
  appendfile = open(appendname,O_RDONLY);
  if (appendfile <0)
  {
    perror("while opening appendfile");
    exit(1);
  }

  
  /* check if elf 32 */
  
  e32 = malloc(sizeof(Elf32_Ehdr));
  assert(ehdr != NULL);
   
  r = read(appendfile,e32,sizeof(Elf32_Ehdr));
    
  assert(r == sizeof(Elf32_Ehdr)); /* assume we get enough
				      to read file hdr */
  check_elf32(e32);

  
  /* read in the program headers */
  /* move to start */

  r = lseek(appendfile, e32->e_phoff, SEEK_SET);
  assert(r == e32->e_phoff);
  
    /* alloc space */
  p64buff = calloc(e32->e_phnum,e32->e_phentsize);
  assert(p64buff != NULL);

  /* read them in */
  
  bp = 0;
  bsize = e32->e_phnum * e32->e_phentsize;
  do {
    r = read(appendfile, &p64buff[bp], bsize - bp);
    assert(r >= 0);
    bp += r;
  } while (bp != bsize);
  
  
  dphdr = (Dit_Phdr *) &dbuff[dhdr->d_phnum * dhdr->d_phsize + dhdr->d_phoff];
  dphdr->p_base = dphdr->p_size = 0;

  vaddr_base = dhdr->d_vaddrend;
  file_offset_base = dhdr->d_fileend;
  cur_file_offset = 0;
  
  for (i = 0; i < e32->e_phnum; i++)
  {
    p32 = (Elf32_Phdr *) &p64buff[i * e32->e_phentsize];
    if (p32->p_type == PT_LOAD)
    {
      int pi, po;
      char buff[BUFF_SIZE];
      
      /* we have a segment to load so copy segment into new file */
      if (dphdr->p_base == 0)
      {
	dphdr->p_base = p32->p_vaddr;
	padding = 0;
	if (dhdr->d_vaddrend != p32->p_vaddr)
	{
	  if (dhdr->d_vaddrend < p32->p_vaddr)
	  {
	    fprintf(stderr,
		    "Warning: available space at 0x%lx, binary linked at 0x%lx\n"
		    ,dhdr->d_vaddrend, p32->p_vaddr);

	    /* zero fill the space in between */
	    r = lseek(imagefile, file_offset_base, SEEK_SET);
	    assert(r == file_offset_base );

	    memset(buff,0, BUFF_SIZE);
	    padding = bsize = p32->p_vaddr - dhdr->d_vaddrend;
	    bp = 0;
	    
	    do {
	      r = write(imagefile, buff, (bsize - bp) < BUFF_SIZE ?
			(bsize - bp) : BUFF_SIZE);
	      assert(r >= 0);
	      bp += r;
	    } while (bp != bsize);
	    
	    vaddr_base =  p32->p_vaddr;
	    file_offset_base += p32->p_vaddr - dhdr->d_vaddrend;
	  }
	  else
	  {
	    fprintf(stderr,"Binary to append is not linked at correct address\n");
	    exit(1);
	  }
	}
      }
      else
      {
	if ((p32->p_vaddr - vaddr_base) > cur_file_offset)
	{
	  cur_file_offset = p32->p_vaddr - vaddr_base;
	}
      }
      assert(p32->p_paddr == p32->p_vaddr);
      assert(p32->p_align == DHDR_ALIGN);

      /* move to right offset in image file */
      r = lseek(imagefile, file_offset_base + cur_file_offset, SEEK_SET);
      assert(r == file_offset_base + cur_file_offset);
      
      r = lseek(appendfile, p32->p_offset, SEEK_SET);
      assert(r == p32->p_offset);
      
      /* copy all the loadable segments into downloadable image file */

      pi = po = 0;
      bsize = p32->p_filesz;
      do {
	r = read(appendfile, buff,
		 (p32->p_filesz - po) < BUFF_SIZE ?
		 (p32->p_filesz - po) : BUFF_SIZE);
	assert(r >= 0);
	
	pi = r;
	bp = 0;
	
	while (bp != pi)
	{
	  r = write(imagefile, &buff[bp], pi - bp);
	  assert(r >= 0);
	  bp += r;
	  po += r;
	}
      } while (po != bsize);
      
      /* now zero fill between end of filesz and memsz */
      for (pi = 0; pi < BUFF_SIZE; pi++)
      {
	buff[pi] = 0;
      }
      
      bsize = ((p32->p_memsz | (4096 - 1)) + 1)- p32->p_filesz;
      po = 0;
      
      r = lseek(imagefile,
		file_offset_base + cur_file_offset + p32->p_filesz,
		SEEK_SET);
      assert(r == file_offset_base + cur_file_offset + p32->p_filesz);

      while (po != bsize)
      {
	r = write(imagefile, buff,
		  (bsize - po) < BUFF_SIZE ?
		  (bsize - po) : BUFF_SIZE);
	assert(r >= 0);
	po += r;
      }
      
      cur_file_offset +=  ((p32->p_memsz | (4096 - 1)) + 1);
    }
    
  }

  dphdr->p_size = cur_file_offset;
  dphdr->p_entry = e32->e_entry;
  strncpy(dphdr->p_name, appendname, DIT_NPNAME);
  dphdr->p_name[DIT_NPNAME - 1] = 0;
  dphdr->p_flags = flags;
  dhdr->d_phnum++;
  dhdr->d_fileend = file_offset_base + cur_file_offset;
  dhdr->d_vaddrend = vaddr_base + cur_file_offset;
  

  /* now write DIT header back */
  r = lseek(imagefile, phdr->p_offset, SEEK_SET);
  assert(r == phdr->p_offset);

  bp = 0;
  bsize =  DHDR_SEG_SIZE;
  do {
    r = write(imagefile, &dbuff[bp], bsize - bp);
    assert(r >= 0);
    bp += r;
  } while (bp != bsize);
  /* now patch the original image program header */
  r = lseek(imagefile, ehdr->e_phoff + (ehdr->e_phnum -1) * ehdr->e_phentsize
	    , SEEK_SET);
  assert(r == ehdr->e_phoff + (ehdr->e_phnum -1) * ehdr->e_phentsize);

  phdr->p_filesz += padding + cur_file_offset;
  phdr->p_memsz += padding + cur_file_offset;

  r = write(imagefile,phdr,sizeof(Elf32_Phdr));


  /* take a section header on the end */
    
  assert(r == sizeof(Elf32_Phdr)); /* assume we get enough
				      to read file hdr */
  s32.sh_name = 0;
  s32.sh_type = SHT_NULL;
  s32.sh_flags = 0;
  s32.sh_addr = 0;
  s32.sh_offset = 0;
  s32.sh_size = 0;
  s32.sh_link = SHN_UNDEF;
  s32.sh_info = 0;
  s32.sh_addralign = 0;
  s32.sh_entsize = 0;
  
  r = lseek(imagefile, dhdr->d_fileend, SEEK_SET);
  assert(r == dhdr->d_fileend);
  
  bp = 0;
  bsize = ehdr->e_shentsize;
  do {
    r = write(imagefile, &((char *)(&s32))[bp], bsize - bp);
    assert(r >= 0);
    bp += r;
  } while (bp != bsize);
  
  /* fix initial header */

  ehdr->e_shoff = dhdr->d_fileend;
  
  r = lseek(imagefile, 0, SEEK_SET);
  assert(r == 0);
  
  bp = 0;
  bsize = ehdr->e_ehsize;
  do {
    r = write(imagefile, &((char *)(ehdr))[bp], bsize - bp);
    assert(r >= 0);
    bp += r;
  } while (bp != bsize);

  
  close(imagefile);
  close(appendfile);
  
}
  
