/* fft.c
 * uses: fft.h for included constants (currently 16384 points max, 2^14)
 * uses: bitsw.c to reverse the order of the bits in an index
 *
 * routines:
 *
 * int for_fft(float *yre,float *yim,float *zre,float *zim,int npts)
 * 
 * computes a forward fft from the source data in (yre,yim). If yim is
 * 0, then it is assumed that the complex part of the input data is zero.
 *
 *
 * int inv_fft(float *yre,float *yim,float *zre,float *zim,int npts)
 * 
 * computes an inverse fft from the source data in (yre,yim). If yim is
 * 0, then it is assumed that the complex part of the input data is zero.
 *
 * 
 */
#include <math.h>
extern "C" {
#include "fft.h"

/* These are the static arrays used for in-place work */
/* so we don't waste time doing mallocs.	      */
/* Note that the real and imaginary parts of a number */
/* are adjacent.				      */
static float x_co[Max_fft][2];		/* transform data */
static float w_co[Max_fft>>1][2];	/* sin/cos data   */

/* Call this routine to do an inverse fft */
int 
inv_fft(yre,yim,zre,zim,npts)
float *yre,*yim,*zre,*zim;
int npts;
{
int gamma,fft_size;
/***************************/
/* get size of fft needed: */
gamma = 1 + (int)(log((float) npts-1.0)/log(2.0));
if (gamma > Max_gamma) {
	printf("Too many points to fft.\n");
	return(0);
	}
/* Init bitswitch */
init_bitsw();
/***************************************/
/* Calculate mixing factors, if needed */
	make_w(gamma);
/************************/
/* copy data into array */
fftcopy(yre,yim,x_co,npts,0);
fft_size = 1<<gamma;
fftfill(x_co,npts,fft_size);
/**************************/
/*    Do in-place ifft    */
dofft(gamma);

/*****************************/
/* copy into destination     */
/* and correct for inversion */
ifftcopy(x_co,zre,zim,gamma, 1.0,-1.0);
return(fft_size);
}

/* Call this routine to do a forward fft */
int 
for_fft(yre,yim,zre,zim,npts)
float *yre,*yim,*zre,*zim;
int npts;
{
int gamma,fft_size;
/***************************/
/* get size of fft needed: */
gamma = 1 + (int)(log(npts-1.0)/log(2.0));
if (gamma > Max_gamma) {
	printf("Too many points to fft.\n");
	return(0);
	}
/* Init bitsw */
init_bitsw();
/***************************************/
/* Calculate mixing factors, if needed */
	make_w(gamma);
/************************/
/* copy data into array */
fftcopy(yre,yim,x_co,npts,1);
fft_size = 1<<gamma;
fftfill(x_co,npts,fft_size);
/**************************/
/*    Do in-place fft     */
dofft(gamma);

/*************************/
/* copy into destination */
/* and re-order numbers  */
ifftcopy(x_co,zre,zim,gamma,
	1.0/(float) fft_size,1.0/(float) fft_size);
return(fft_size);
}


dofft(gamma)
register int gamma;
{
register int k;

/**********************/
/* do fft butterflies */
for (k = 1; k <= gamma; k++) {
	butterfly(k,gamma);
	}
/********************************/

return;
}

fftcopy(pr_from,pi_from,p_to,npts,is_plus)
register float *pr_from,*pi_from,*p_to;
register int npts;
register int is_plus;
{
register float zero=0.0;
register float *p_end;

if (pi_from != NULL) {
	if (is_plus) {
		p_end = p_to + 16*(int)(npts/8);
		while (p_to < p_end) {
			p_to[0] = pr_from[0];
			p_to[2] = pr_from[1];
			p_to[4] = pr_from[2];
			p_to[6] = pr_from[3];
			p_to[8] = pr_from[4];
			p_to[10] = pr_from[5];
			p_to[12] = pr_from[6];
			p_to[14] = pr_from[7];
			pr_from += 8;

			p_to[1] = pi_from[0];
			p_to[3] = pi_from[1];
			p_to[5] = pi_from[2];
			p_to[7] = pi_from[3];
			p_to[9] = pi_from[4];
			p_to[11] = pi_from[5];
			p_to[13] = pi_from[6];
			p_to[15] = pi_from[7];
			pi_from += 8;
			p_to += 16;
			}
		p_end += 2*(npts % 8);
		while (p_to < p_end) {
			p_to[0] = pr_from[0];
			p_to[1] = pi_from[0];
			p_to += 2;
			pr_from++;
			pi_from++;
			}
		}
	else {
		p_end = p_to + 16*(npts / 8);
		while (p_to < p_end) {
			p_to[0] = pr_from[0];
			p_to[2] = pr_from[1];
			p_to[4] = pr_from[2];
			p_to[6] = pr_from[3];
			p_to[8] = pr_from[4];
			p_to[10] = pr_from[5];
			p_to[12] = pr_from[6];
			p_to[14] = pr_from[7];
			pr_from += 8;

			p_to[1] = -pi_from[0];
			p_to[3] = -pi_from[1];
			p_to[5] = -pi_from[2];
			p_to[7] = -pi_from[3];
			p_to[9] = -pi_from[4];
			p_to[11] = -pi_from[5];
			p_to[13] = -pi_from[6];
			p_to[15] = -pi_from[7];
			pi_from += 8;
			p_to += 16;
			}
		p_end += 2*(npts % 8);
		while (p_to < p_end) {
			p_to[0] = pr_from[0];
			p_to[1] = -pi_from[0];
			p_to += 2;
			pr_from++;
			pi_from++;
			}
		}
	}
else {
		p_end = p_to + 16*(npts / 8);
		while (p_to < p_end) {
			p_to[0] = pr_from[0];
			p_to[2] = pr_from[1];
			p_to[4] = pr_from[2];
			p_to[6] = pr_from[3];
			p_to[8] = pr_from[4];
			p_to[10] = pr_from[5];
			p_to[12] = pr_from[6];
			p_to[14] = pr_from[7];
			pr_from += 8;

			p_to[1] = zero;
			p_to[3] = zero;
			p_to[5] = zero;
			p_to[7] = zero;
			p_to[9] = zero;
			p_to[11] = zero;
			p_to[13] = zero;
			p_to[15] = zero;
			pi_from += 8;
			p_to += 16;
			}
		p_end += 2*(npts % 8);
		while (p_to < p_end) {
			p_to[0] = pr_from[0];
			p_to[1] = zero;
			p_to += 2;
			pr_from++;
			pi_from++;
			}
	}
return;
}

/* Copy fft or ifft to destination (and scale) while also putting numbers
 * back in correct order using bitswitch()
 */
ifftcopy(p_from,pr_to,pi_to,gamma,factr,facti)
register float *p_from;
register float *pr_to,*pi_to;
register float factr,facti;
register int gamma;
{
register int k,k1;
register int m;
k1 = 1<<gamma;
for (k=0 ; k < k1; k += 2) {
	m = bitswitch(k,gamma);
	pr_to[m] = factr * *(p_from++);
	pi_to[m] = facti * *(p_from++);
	m = bitswitch(k+1,gamma);
	pr_to[m] = factr * *(p_from++);
	pi_to[m] = facti * *(p_from++);
	}
return;
}


fftfill(p_co,part,all)
register float *p_co;
register int all;
register int part;
{
register int k;
register float r_sum=0.,i_sum=0.;
if (all == part) return;
for (k=part;k>0;k--) {
	r_sum += *(p_co++);
	i_sum += *(p_co++);
	}
r_sum /= part;
i_sum /= part;
for (k=(all-part); k>0; k--) {
	*(p_co++) = r_sum;
	*(p_co++) = i_sum;
	}
return;
}


make_w(new)
register int new;
{
register int npts;
register int k;
register float temp;
register int offs;
static int old = 0;
if (new == old) return;
/* Only need (real_npts/2) sin & cos values: */
npts = 1<<(new-1);
if (old == 0 || new != old) {
	temp = PI/npts;
	for (k = 0; k < npts; k++) {
		offs = bitswitch(k,new) >> 1;
		w_co[k][0] = fcos(temp*offs);
		w_co[k][1] = fsin(temp*offs);
		}
	}
old = new;
return;
}


butterfly(fly,gamma)
int fly;
register int gamma;
{
register int i,j,step;
register int out_size;
register float *smix_co,*sk_co;
register float *wco;
register float x1,x2;
register float w0,w1;

out_size = 1<<(fly-1);
step = 1<<(gamma-fly);
wco = &w_co[0][0];
sk_co = &x_co[0][0];
smix_co = sk_co + (step<<1);

for (j=0; j<out_size; j++) {
	/* Find location of (sin,cos) mixing values */
	w0 = *wco++;
	w1 = *wco++;

	for (i=step; i > 0; i--) {
		/* Calculate real (x1) and imaginary (x2) parts of */
		/* mix with smix_co				   */
		x1 =  *(smix_co) * w0 - *(smix_co+1) * w1;
		x2 =  *(smix_co+1) * w0 + *(smix_co) * w1;

		/* Subtract product from *smix_co */
		/* and Add product into *sk_co */

		/* Real parts */
		*smix_co = *sk_co - x1;
		*sk_co = *sk_co + x1;

		/* Imaginary parts */
		*(smix_co+1) = *(sk_co+1) - x2;
		*(sk_co+1) = *(sk_co+1) + x2;
		
		/* Increment to next complex number */
		smix_co += 2;
		sk_co += 2;
		}
	sk_co += step<<1;
	smix_co += step<<1;
	}
return;
}

}
